import os

from si_tests import logger
from si_tests import settings
from si_tests.deployments.utils import file_utils
from si_tests.managers import helm3_manager
from si_tests.utils import waiters

LOG = logger.logger


def test_k8s_fio(kaas_manager):
    """This test creates Helm Chart for running storage
    performance test (fio). Storage options and fio workloads
    can be configured to your needs using variables. It is possible
    to take the test multiple times to get more data to analyze.
    The result of each test will be saved in a separate json
    for future analysis.

    Scenario:
        1. Setting variables
        2. Acquiring child kubeconfig
        3. Overrides for fio chart
        4. Creating namespace
        5. Runing test
        6. Deleting namespace
        7. Checking pod
    """
    # Setting variables
    child_name = settings.TARGET_CLUSTER
    child_namespace = settings.TARGET_NAMESPACE
    fio_namespace = settings.FIO_NAMESPACE
    artifacts_dir = settings.ARTIFACTS_DIR

    fio_node_selector = settings.FIO_NODE_LABEL if settings.FIO_NODE_LABEL else {}
    fio_replicas = settings.FIO_SERVER_COUNT
    fio_storage_class = settings.FIO_STORAGE_CLASS_NAME
    fio_pvc_size = settings.FIO_STORAGE_SIZE
    fio_with_warmup = settings.FIO_WITH_WARMUP
    fio_chart = settings.FIO_CHART_URL
    fio_jobs_path = settings.FIO_JOBS_CONFIG_PATH
    fio_test_timeout = settings.FIO_TEST_TIMEOUT
    loops = settings.FIO_LOOPS + 1

    # Acquiring child kubeconfig
    tg_ns = kaas_manager.get_namespace(child_namespace)
    tg_cluster = tg_ns.get_cluster(child_name)
    tg_client = tg_cluster.k8sclient
    kubeconfig_name, child_kubeconfig = tg_cluster.get_kubeconfig()
    kubeconfig_path = os.path.join(artifacts_dir, kubeconfig_name)
    LOG.info(f"Saving child cluster kubeconfig to {kubeconfig_path}")
    with open(kubeconfig_path, "w") as f:
        f.write(child_kubeconfig)
    os.chmod(path=kubeconfig_path, mode=400)

    # Overrides for fio chart
    values_fio = {
        "server": {
            "nodeSelector": fio_node_selector,
            "replicaCount": fio_replicas,
            "storageClassName": fio_storage_class,
            "storageSize": fio_pvc_size,
            "warmup": {
                "enabled": fio_with_warmup,
            }
        }
    }
    fio_values_path = os.path.join(artifacts_dir, "fio_overrides.yaml")
    file_utils.save_to_yaml(values_fio, fio_values_path)

    # Creating namespace
    LOG.info("Creating fio namespace")
    if tg_client.namespaces.present(fio_namespace):
        LOG.info(f"Namespace '{fio_namespace}' already exists")
        fio_ns = tg_client.namespaces.get(fio_namespace)
    else:
        fio_ns = tg_client.namespaces.create(
            name=fio_namespace,
            body={'metadata': {'name': fio_namespace}})

    # Run test
    helm = helm3_manager.Helm3Manager(namespace=fio_namespace)
    chart_name = "fio"
    statefulset_name = "fio-server"
    deployment_name = "fio-client"
    container_name = "fio"

    for loop in range(1, loops):
        LOG.info(f'{"-" * 50} Starting round {loop} {"-" * 50}')

        # Installing fio chart
        LOG.info("Installing fio Helm chart")
        LOG.debug("Using fio chart {}".format(fio_chart))
        install_result = helm.install_chart(kubeconfig_path, fio_namespace, chart_name, fio_chart,
                                            values_path=[fio_jobs_path, fio_values_path])
        assert install_result, "fio Helm Chart has not been installed"

        # Waiting for Helm Chart to be ready
        deployment = tg_client.deployments.get(name=deployment_name, namespace=fio_namespace)
        deployment.wait_ready()
        statefulset = tg_client.statefulsets.get(name=statefulset_name, namespace=fio_namespace)
        statefulset.wait_readiness()

        # Waiting for the end of fio test
        client_pod_list = tg_client.pods.list(namespace=fio_namespace,
                                              label_selector="app.kubernetes.io/name=fio,type=client")
        assert len(client_pod_list) == 1, "FIO client is not working properly"
        client_pod = client_pod_list[0]
        client_pod.wait_test(filepath="./test_passed",
                             timeout=fio_test_timeout,
                             container=container_name)

        # Changing file name before copy
        path_to_results = "/results"
        base_results_name = "fio-results.json"
        new_results_name = f"fio-results-{loop}.json"
        cmd = ["/bin/sh", "-c",
               f"mv {path_to_results}/{base_results_name} {path_to_results}/{new_results_name}"]
        client_pod.exec(cmd, container=container_name)
        LOG.debug(f"Remote file {base_results_name} has been renamed to {new_results_name}")

        # Copy fio result
        LOG.info(f'{"-" * 50} Save results {"-" * 50}')
        client_pod.cp_from_pod(source_dir="/results", container=container_name)

        # Deleting fio chart
        LOG.info("Deleting fio Helm chart")
        delete_result = helm.delete_chart(kubeconfig_path, fio_namespace, chart_name)
        assert delete_result, "fio Helm Chart has not been deleted"

        # Waiting for Helm Chart to be deleted
        waiters.wait(lambda:
                     not tg_client.deployments.present(name=deployment_name,
                                                       namespace=fio_namespace) and
                     not tg_client.statefulsets.present(name=statefulset_name,
                                                        namespace=fio_namespace),
                     interval=5, timeout=60)

    # Deleting namespace for fio test
    LOG.info("Deleting fio namespace")
    fio_ns.delete(fio_namespace)

    # Checking all pods are in consistent state after test run
    tg_cluster.check.check_k8s_pods()
