import base64
from datetime import datetime

import jwt
import pytest
from kubernetes.client.rest import ApiException

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters

LOG = logger.logger


def decode_license(license_string):
    """
    Decode license string to data dict
    Args:
        license_string: encoded license string

    Returns: dict with license data

    """
    decode_string = jwt.decode(license_string.strip(), key=None, options={"verify_signature": False})
    return decode_string


def verify_license(license_content, cluster_license):
    """
    Compare cluster license data and data from license file
    Args:
        license_content: data from license file
        cluster_license: data from cluster license

    Returns: None

    """
    msg = ""
    license_exp = datetime.utcfromtimestamp(license_content['exp'])
    license_is_dev = license_content['license'].get('dev', False)
    license_limit_clusters = (license_content['license'].get('limits') or {}).get('clusters', 0)
    license_limit_workers = (license_content['license'].get('limits') or {}).get('workers_per_cluster', 0)
    license_os_clusters = (license_content['license'].get('openstack') or {}).get('clusters', 0)
    license_os_workers = (license_content['license'].get('openstack') or {}).get('workers_per_cluster', 0)
    cluster_exp = datetime.fromisoformat(cluster_license['status']['expirationTime'][:-1])
    cluster_is_dev = cluster_license['status'].get('dev', False)
    cluster_limit_clusters = (cluster_license['status'].get('limits') or {}).get('clusters', 0)
    cluster_limit_workers = (cluster_license['status'].get('limits') or {}).get('workersPerCluster', 0)
    cluster_os_clusters = (cluster_license['status'].get('openstack') or {}).get('clusters', 0)
    cluster_os_workers = (cluster_license['status'].get('openstack') or {}).get('workersPerCluster', 0)
    if license_exp != cluster_exp:
        msg += f"\n Expiration time is different. License contains {license_exp}, but actually {cluster_exp}"
    if license_is_dev != cluster_is_dev:
        msg += f"\n License type is different. License contains {license_is_dev}, but actually {cluster_is_dev}"
    if license_limit_clusters != cluster_limit_clusters:
        msg += f"\n Cluster limits is different. License contains {license_limit_clusters}, " \
               f"but actually {cluster_limit_clusters}"
    if license_limit_workers != cluster_limit_workers:
        msg += f"\n Worker limits is different. License contains {license_limit_workers}, " \
               f"but actually {cluster_limit_workers}"
    if license_os_clusters != cluster_os_clusters:
        msg += f"\n OS clusters limits is different. License contains {license_os_clusters}, " \
               f"but actually {cluster_os_clusters}"
    if license_os_workers != cluster_os_workers:
        msg += f"\n OS worker limits is different. License contains {license_os_workers}, " \
               f"but actually {cluster_os_workers}"
    if msg == "":
        LOG.info("The license data from the file and applied to the cluster are identical")
    else:
        raise Exception(msg)


def verify_mke_license(license_data, mke_license):
    """
    Compare MKE license data and data from license file
    Args:
        license_data: data from license file
        mke_license: data from MKE cluster

    Returns:

    """
    license_content = decode_license(license_data)
    assert license_data.rstrip() == mke_license['license_config']['jwt'].rstrip(), \
        "Decoded license data from original file and MKE is different"

    license_exp = datetime.utcfromtimestamp(license_content['exp'])
    cluster_exp = datetime.fromisoformat(mke_license['license_details']['expiration'][:-1])
    assert license_content['sub'] == mke_license['license_config']['key_id'], \
        "License subject from original file and MKE is different"
    assert (license_content['license'].get('limits') or {}).get('workers_per_cluster', 0) == \
           mke_license['license_details']['maxEngines'], "License limits from original file and MKE is different"
    assert license_content['sub'] == mke_license['license_details']['mirantis_license_subject'], \
        "License subject from original file and MKE is different"
    assert license_exp == cluster_exp, "License expiration from original file and MKE is different"


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")  # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('log_method_time')
def test_update_license(kaas_manager, show_step):
    """Update license test.

    Scenario:
        1. Get current license data
        2. Update management license
        3. Check new license in a management cluster
        4. Check new license in a regional/child clusters
        5. Check cluster(s) readiness

    """
    new_secret_name = None
    new_license = None
    mke_license = None

    def _check_license():
        nonlocal new_secret_name
        nonlocal new_license
        new_license = kaas_manager.get_license()
        new_secret_name = new_license.data['spec']['license']['secret']['name']
        LOG.info(f"Wait new secret name, old name = {old_secret_name}, new name = {new_secret_name}")
        return new_secret_name != old_secret_name

    def _check_license_secret(secret_name, target_cluster, original_license, namespace=None):
        try:
            secret = target_cluster.k8sclient.secrets.get(name=secret_name, namespace=namespace).read()
        except ApiException as e:
            LOG.info(f"Failed to read '{secret_name}' secret. Wait...\n{e.body}")
            return False
        else:
            decoded_data = base64.b64decode(secret.data['value']).decode('utf-8')
            if decoded_data.rstrip() == original_license.rstrip():
                LOG.info(f"Secret '{secret_name}' contain new data.")
                return True
            else:
                LOG.info(f"Secret '{secret_name}' does not contain new data yet. Waiting...")

    def _check_mke_license_token(target_cluster, original_license):
        nonlocal mke_license
        mke_license = target_cluster.mkedashboardclient.get_license()
        mke_license_token = mke_license.get('license_config', {}).get('jwt', '')
        if mke_license_token.rstrip() == original_license.rstrip():
            LOG.info("The license data from the file and applied to the MKE cluster are identical")
            return True
        else:
            LOG.info("The license data from the file and applied to the MKE cluster are not identical. Waiting...")
            return False

    managed_ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = managed_ns.get_cluster(settings.TARGET_CLUSTER)

    show_step(1)
    with open(settings.KAAS_BOOTSTRAP_LICENSE_FILE, 'r') as license_file:
        license_data = license_file.read()

    new_license_content = decode_license(license_data)
    is_dev_new_license = new_license_content['license']['dev']
    new_license_sub = new_license_content['sub']

    old_license = kaas_manager.get_license()
    old_secret_name = old_license.data['spec']['license']['secret']['name']
    customer_id = old_license.data['status']['customerID']
    is_dev_old_license = old_license.data.get('status', {}).get('dev', False)
    LOG.info("For cluster '{}' found already installed '{}' license: {}".format(
        cluster.name,
        "DEV" if is_dev_old_license else "PROD",
        customer_id
    ))
    LOG.info("Found '{}' license for update: {}".format(
        "DEV" if is_dev_new_license else "PROD",
        new_license_sub
    ))

    if is_dev_new_license:
        msg = "License configuration error! You are trying to upgrade to DEV licenses that are not supported by MKE."
        LOG.error(msg)
        raise Exception(msg)

    regional_clusters = kaas_manager.get_regional_clusters()
    child_clusters = kaas_manager.get_child_clusters()

    show_step(2)
    if settings.UPDATE_LICENSE:
        kaas_manager.update_license(license_data)
        waiters.wait(lambda: _check_license(), timeout=360, timeout_msg="Timeout waiting for updated secret name")
    else:
        LOG.info("Test without update license. Only check")
        new_license = kaas_manager.get_license()
        new_secret_name = new_license.data['spec']['license']['secret']['name']

    show_step(3)
    waiters.wait(lambda: _check_license_secret(new_secret_name, cluster, license_data),
                 timeout=180,
                 timeout_msg="Timeout waiting for mgmt secret data.")

    customer_id = new_license.data['status']['customerID']
    is_dev_new_license = new_license.data.get('status', {}).get('dev', False)
    LOG.info("For cluster {} found installed '{}' license: {}".format(
        cluster.name,
        "DEV" if is_dev_new_license else "PROD",
        customer_id
    ))
    verify_license(new_license_content, new_license.data)

    if settings.UPDATE_LICENSE:
        LOG.info("Check MKE license for MGMT cluster")
        waiters.wait(lambda: _check_mke_license_token(cluster, license_data),
                     timeout=600,
                     interval=30,
                     timeout_msg="Timeout waiting for updated MKE license token")
        verify_mke_license(license_data, mke_license)

    show_step(4)
    for regional_cluster in regional_clusters:
        LOG.info(f"Check license secret in {regional_cluster.name}")
        waiters.wait(lambda: _check_license_secret("mcc-license", regional_cluster, license_data),
                     timeout=180,
                     timeout_msg="Timeout waiting for mgmt secret data.")

        LOG.info(f"Check MKE license for regional cluster '{regional_cluster.name}'")
        mke_license = None  # cleanup existing data
        waiters.wait(lambda: _check_mke_license_token(regional_cluster, license_data),
                     timeout=180,
                     timeout_msg="Timeout waiting for updated MKE license token")
        verify_mke_license(license_data, mke_license)
    for child_cluster in child_clusters:
        LOG.info(f"Check MKE license for child cluster '{child_cluster.name}'")
        mke_license = None  # cleanup existing data
        waiters.wait(lambda: _check_mke_license_token(child_cluster, license_data),
                     timeout=180,
                     timeout_msg="Timeout waiting for updated MKE license token")
        verify_mke_license(license_data, mke_license)

    show_step(5)
    LOG.info("Check management cluster readiness")
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_pods()
    for regional_cluster in regional_clusters:
        LOG.info(f"Check regional cluster '{regional_cluster.name}' readiness")
        regional_cluster.check.check_cluster_readiness()
        regional_cluster.check.check_k8s_pods()
    for child_cluster in child_clusters:
        LOG.info(f"Check child cluster '{child_cluster.name}' readiness")
        child_cluster.check.check_cluster_readiness()
        child_cluster.check.check_k8s_pods()
