import random
import re

import pytest
from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters
from si_tests.utils import packaging_version as version

LOG = logger.logger


def check_success_result_on_machine(machine, cmd):
    """ For exit code == 0
        0: command was executed successfully
    """
    LOG.info(f"Check command {cmd} on machine: {machine.name}")
    res = machine.exec_pod_cmd(cmd)
    if res.get('exit_code') != 0:
        raise Exception(f"Command {cmd} finished on Machine '{machine.name}' "
                        f"with exit.code='{res.get('exit_code')}'\n{res.get('events')}, "
                        f"but expected value was SUCCESS with code 0")
    return True


def check_missing_package_result_on_machine(machine, cmd):
    """ For exit code = 127:
        127: return code which means: 'command not found', we expect this return code
        when we try to execute not installed package
    """
    LOG.info(f"Check that command {cmd} failed on machine: {machine.name} with exit code 127")
    res = machine.exec_pod_cmd(cmd)
    if res.get('exit_code') != 127:
        raise Exception(f"Command {cmd} finished on Machine '{machine.name}' "
                        f"with exit.code='{res.get('exit_code')}'\n{res.get('events')}, "
                        f"but expected value was FAIL with code 127, because here we "
                        f"expect that package is not installed on machine")
    return True


def check_failure_result_on_machine(machine, cmd):
    """ For exit code = 1:
        1: command return general error, for example 'ls /var | grep fail'
        returns 1 if there are no file with name 'fail' in dir /var
    """
    LOG.info(f"Check that command {cmd} failed on machine: {machine.name} with exit code 1")
    res = machine.exec_pod_cmd(cmd)
    if res.get('exit_code') != 1:
        raise Exception(f"Command {cmd} finished on Machine '{machine.name}' "
                        f"with exit.code='{res.get('exit_code')}'\n{res.get('events')}, "
                        f"but expected value was FAIL with code 1, because here we "
                        f"unspecified ERROR return from command")
    return True


def get_pkg_available_versions(machine, pkg_name):
    """ Extract package versions available to install using "apt list -a" command

    :return list of package versions

    """
    cmd = f"apt list -a {pkg_name}"
    res = machine.exec_pod_cmd(cmd)
    if res.get('exit_code') != 0:
        raise Exception(f"'apt -a {pkg_name}' call on {machine.name} failed. Logs:\n{res['logs']}")

    pattern = fr"{pkg_name}/\S*\s(\S*)\s"
    ret_val = []
    for line in res['logs'].split("\n"):
        re_match = re.search(pattern, line)
        if re_match:
            ret_val.append(re_match.group(1))
    return ret_val


def get_pkg_installed_version(machine, pkg_name):
    """ Extract installed version of given package.

    :return string with package version

    """
    cmd = f"dpkg -s {pkg_name}"
    res = machine.exec_pod_cmd(cmd)
    if res.get('exit_code') != 0:
        raise Exception(f"'dpkg -s {pkg_name}' call on {machine.name} failed. Logs:\n{res['logs']}")
    pattern = r"Version:\s(.*)\n"
    re_match = re.search(pattern, res['logs'])
    if re_match.groups():
        return re_match.group(1)
    raise Exception(f"Version not found in output: {res['logs']}")


def get_pinned_packages(machine):
    """ Extract all pinned packages with corresponding pinned versions from apt-cache policy output

    :return dict of package:version pairs

    """
    cmd = "apt-cache policy"
    res = machine.exec_pod_cmd(cmd)
    ret_val = {}
    pattern = r"\s*(.*)\s->\s(.*)\swith\spriority\s(\d*)"
    for log_line in res['logs'].split("\n"):
        if "->" in log_line:
            re_match = re.search(pattern, log_line)
            if re_match.groups():
                ret_val[re_match.group(1)] = re_match.group(2)
    return ret_val


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_hoc_package_add_remove_repository(kaas_manager, _, show_step):
    """ Test package module for adding/removing repository and install/remove package from added repository
    Scenario:
        1. Add “wrong” repo(with address from which we can’t download key) and check that we have error in hoc object
        2. Add correct repo and gpg key URL and check that repo will be added to selected node
        3. Install package from repo and check result on machine
        4. Set “absent” to package and repo check result on machines
        5. TODO: Set diff priority to the added repos (add bionic repo with less priority then mcc ones < 500)
        6. Delete HOC object and clean machine labels
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    hoc_name = "si-team-hoc-package-broken-repo" + str(random.randint(10, 99))
    machines = cluster.get_machines()
    assert machines, f"No machines in cluster: {cluster_name}"
    first_machine = machines[0]
    first_lcmmachine = cluster.get_cluster_lcmmachine(first_machine.name, cluster.namespace)
    repo_file_name = "si-custom-repo"
    cmd = f"cat /etc/apt/sources.list.d/{repo_file_name}.list"

    day2_label = {"day2-custom-si-label" + str(random.randint(10, 99)): "enabled"}

    # wrong repo package sample
    hostoscfg_config_traceroute = [
        {
            "module": "package",
            "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
            "values": {"repositories": [{"filename": repo_file_name,
                                         "key": "https://example.org/packages/key.gpg",
                                         "repo": "deb https://example.org/packages/ apt/stable/",
                                         "state": "present"}]},
        }
    ]

    hostoscfg_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "HostOSConfiguration",
        "metadata": {
            "name": hoc_name,
            "namespace": namespace_name,
        },
        "spec": {
            "configs": hostoscfg_config_traceroute,
            "machineSelector": {
                "matchLabels": day2_label,
            }
        }
    }

    show_step(1)
    LOG.info(f"Add label day-2-custom-si to first machine: {first_machine.name} in child cluster: {cluster_name}")
    first_machine.add_machine_labels(day2_label)
    LOG.info(f"Check that repo is not added to machine: {first_machine.name}")
    check_failure_result_on_machine(machine=first_machine, cmd=cmd)

    first_lcmmachine_timestamp_before = cluster.get_lcmmachine_timestamps(first_lcmmachine)
    LOG.info(f"Create HOC object with module package version: {settings.HOC_PACKAGE_MODULE_VERSION}")
    hostoscfg = ns.create_hostosconfiguration_raw(hostoscfg_data)
    cluster.check.wait_hoc_state_item_statuses_are_changed_in_lcmmachine(first_lcmmachine,
                                                                         first_lcmmachine_timestamp_before,
                                                                         timeout=1500)
    ns.wait_hostosconfiguration_config_state_items_status(hoc_name=hoc_name, expected_status="Failed")
    LOG.info("Check that we have only one machine")
    assert len(cluster.check.get_hostosconfig_machines_status(hostoscfg)) == 1, \
        "HOC object has more than 1 machine"
    LOG.info(f"Check that repo still is not added to machine: {first_machine.name} because of failed parameter")
    check_failure_result_on_machine(machine=first_machine, cmd=cmd)
    # We MUST delete this HOC with intentional Ansible fail, as it prevents other HOCs from execution
    # Patching the failed HOC also does not have a sense
    existing_config = ns.get_hostosconfiguration(name=hoc_name)
    LOG.info(f"Deleting HostOSConfiguration '{existing_config.name}'")
    existing_config.delete(async_req=True)
    timeout_msg = f"HostOSConfiguration {hoc_name} was not deleted"
    waiters.wait(lambda: not bool(ns.hostosconfiguration_is_present(name=hoc_name)),
                 timeout=1500,
                 interval=10,
                 timeout_msg=timeout_msg)

    show_step(2)
    hoc_name = "si-team-hoc-package-valid-repo" + str(random.randint(10, 99))
    # valid repo package sample
    hostoscfg_config_sublime = [
        {
            "module": "package",
            "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
            "values": {"repositories": [{"filename": repo_file_name,
                                         "key": "https://download.sublimetext.com/sublimehq-pub.gpg",
                                         "repo": "deb https://download.sublimetext.com/ apt/stable/",
                                         "state": "present"}]},
        }
    ]

    hostoscfg_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "HostOSConfiguration",
        "metadata": {
            "name": hoc_name,
            "namespace": namespace_name,
        },
        "spec": {
            "configs": hostoscfg_config_sublime,
            "machineSelector": {
                "matchLabels": day2_label,
            }
        }
    }
    LOG.info(f"Create HOC object with module package version: {settings.HOC_PACKAGE_MODULE_VERSION}")
    hostoscfg = ns.create_hostosconfiguration_raw(hostoscfg_data)
    first_lcmmachine_timestamp_before = cluster.get_lcmmachine_timestamps(first_lcmmachine)

    cluster.check.wait_hoc_state_item_statuses_are_changed_in_lcmmachine(first_lcmmachine,
                                                                         first_lcmmachine_timestamp_before,
                                                                         timeout=1500)
    ns.wait_hostosconfiguration_config_state_items_status(hoc_name=hoc_name, expected_status="Success")
    LOG.info(f"Check that repo is added to machine: {first_machine.name}")
    check_success_result_on_machine(machine=first_machine, cmd=cmd)

    show_step(3)
    sublime_cmd = "ls /usr/bin/ | grep subl"
    LOG.info(f"Check that package sublime-text is not installed on machine: {first_machine.name}")
    check_failure_result_on_machine(machine=first_machine, cmd=sublime_cmd)
    first_lcmmachine_timestamp_before = cluster.get_lcmmachine_timestamps(first_lcmmachine)
    hostoscfg.patch({"spec": {"configs": [
        {"module": "package",
         "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
         "phase": "reconfigure",
         "values": {"packages": [
                                {"name": "sublime-text",
                                 "state": "present"}],
                    "repositories": [
                                    {"filename": repo_file_name,
                                     "key": "https://download.sublimetext.com/sublimehq-pub.gpg",
                                     "repo": "deb https://download.sublimetext.com/ apt/stable/",
                                     "state": "present"}]
                    },
         }]}})
    cluster.check.wait_hoc_state_item_statuses_are_changed_in_lcmmachine(first_lcmmachine,
                                                                         first_lcmmachine_timestamp_before,
                                                                         timeout=1500)
    ns.wait_hostosconfiguration_config_state_items_status(hoc_name=hoc_name, expected_status="Success")
    LOG.info(f"Check that package sublime is installed on machine: {first_machine.name}")
    check_success_result_on_machine(machine=first_machine, cmd=sublime_cmd)

    show_step(4)
    first_lcmmachine_timestamp_before = cluster.get_lcmmachine_timestamps(first_lcmmachine)
    hostoscfg.patch({"spec": {"configs": [
        {"module": "package",
         "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
         "phase": "reconfigure",
         "values": {"packages": [
                                {"name": "sublime-text",
                                 "state": "absent"}],
                    "repositories": [
                                    {"filename": repo_file_name,
                                     "key": "https://download.sublimetext.com/sublimehq-pub.gpg",
                                     "repo": "deb https://download.sublimetext.com/ apt/stable/",
                                     "state": "absent"}]
                    },
         }]}})
    cluster.check.wait_hoc_state_item_statuses_are_changed_in_lcmmachine(first_lcmmachine,
                                                                         first_lcmmachine_timestamp_before,
                                                                         timeout=1500)
    ns.wait_hostosconfiguration_config_state_items_status(hoc_name=hoc_name, expected_status="Success")
    LOG.info(f"Check that repo and package sublime were removed from machine: {first_machine.name}")
    check_failure_result_on_machine(machine=first_machine, cmd=sublime_cmd)
    check_failure_result_on_machine(machine=first_machine, cmd=cmd)
    show_step(6)
    existing_config = ns.get_hostosconfiguration(name=hoc_name)
    LOG.info(f"Deleting HostOSConfiguration '{existing_config.name}'")
    existing_config.delete(async_req=True)
    timeout_msg = f"HostOSConfiguration {hoc_name} was not deleted"
    waiters.wait(lambda: not bool(ns.hostosconfiguration_is_present(name=hoc_name)),
                 timeout=1500,
                 interval=10,
                 timeout_msg=timeout_msg)
    first_machine.remove_machine_labels(list(day2_label.keys()))


def apply_hoc_wait(hoc_data, lcmmachine, cluster, ns, patch=False):
    """ Wrapper for creating or patching HOC object """
    # both 2 vars below are required for different checks
    # wait_lcmmachine_day2_stateitems() WILL accept single timestamp from get_lcmmachine_timestamps(lcmmachine)
    # and silently skip dict parsing
    # see clustercheck_manager:L5858: if lcm_obj.name in lcmmachines_timestamps_before:
    lcmmachines_timestamps_before = cluster.get_cluster_lcmmachines_timestamps()
    lcmmachine_timestamp_before = cluster.get_lcmmachine_timestamps(lcmmachine)
    hoc_name = hoc_data['metadata']['name']
    if not patch:
        LOG.info(f"Create HOC object with data '{hoc_data}'")
        ns.create_hostosconfiguration_raw(hoc_data)
    else:
        LOG.info(f"Patching HOC object with spec '{hoc_data['spec']}'")
        hostcfg = ns.get_hostosconfiguration(name=hoc_name)
        hostcfg.patch({"spec": hoc_data['spec']})

    # refresh hostcfg after patch or get it if hoc is new
    hostcfg = ns.get_hostosconfiguration(name=hoc_name)
    LOG.info("Check that machines from hostosconfiguration status field have labels used for machineSelector")
    cluster.check.check_hostosconfig_machine_selector(hostcfg)
    LOG.info("Check that new items added into machineTypes in LCMCluster")
    cluster.check.wait_lcmcluster_day2_machinetypes(hostcfg)
    LOG.info("Check that new items added into stateItems in LCMMachine")
    cluster.check.wait_lcmmachine_day2_stateitems(hostcfg, lcmmachines_timestamps_before)

    cluster.check.wait_hoc_state_item_statuses_are_changed_in_lcmmachine(lcmmachine,
                                                                         lcmmachine_timestamp_before,
                                                                         timeout=1500)
    ns.wait_hostosconfiguration_config_state_items_status(hoc_name=hoc_name, expected_status="Success")


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_hoc_package_install_pin_pkg(kaas_manager, _, show_step):
    """Test package module for pinning package version using apt_preferences.

    Scenario:
        1. Check that the package is not installed.
        2. Sequentially install all available versions of the package and check pinning are applied.
        3. Unpin version without package removing and check that pinning was removed.
        4. Restore pinning for the next step
        5. Delete package and check that pinning was removed.
        6. Cleanup HOC.

    """
    if version.parse(settings.HOC_PACKAGE_MODULE_VERSION) < version.parse("1.3.0"):
        pytest.skip("Package module < 1.3.0 is not supported by this test")
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)

    machines = cluster.get_machines()
    assert machines, f"No machines in cluster: {cluster_name}"
    first_machine = machines[0]
    first_lcmmachine = cluster.get_cluster_lcmmachine(first_machine.name, cluster.namespace)

    target_pkg = settings.HOC_PACKAGE_MODULE_SI_TEST_PKG_NAME
    target_pkg_available_versions = get_pkg_available_versions(first_machine, target_pkg)
    assert len(target_pkg_available_versions) > 1, f"Package {target_pkg} has only one version to be installed," \
                                                   f"consider switching to another package;\n" \
                                                   f"Available versions: {target_pkg_available_versions}"

    day2_label = {"day2-custom-si-label" + str(random.randint(10, 99)): "enabled"}
    hoc_name = "si-team-hoc-package-pin" + str(random.randint(10, 99))

    hoc_pkg_desc = {"name": target_pkg,
                    "version": "",  # stub to redefine in tests below
                    "state": "present",
                    "allow_downgrade": "yes"}

    hoc_configs_desc = [{"module": "package",
                         "moduleVersion": settings.HOC_PACKAGE_MODULE_VERSION,
                         "values": {"packages": [hoc_pkg_desc]}}]

    hoc_data = {"apiVersion": "kaas.mirantis.com/v1alpha1", "kind": "HostOSConfiguration",
                "metadata": {"name": hoc_name, "namespace": namespace_name},
                "spec": {"configs": hoc_configs_desc, "machineSelector": {"matchLabels": day2_label}}}

    pkg_installed_cmd = f"dpkg -l | grep {target_pkg}"

    show_step(1)
    LOG.info(f"Check that package {target_pkg} is not installed on machine: {first_machine.name}")
    check_failure_result_on_machine(machine=first_machine, cmd=pkg_installed_cmd)

    LOG.info(f"Add label day-2-custom-si to first machine: '{first_machine.name}' in child cluster: '{cluster_name}'")
    first_machine.add_machine_labels(day2_label)

    show_step(2)
    hoc_applied = False
    for pkg_target_version in target_pkg_available_versions:
        LOG.info(f"Trying HOC for package '{target_pkg}' with pinned version '{pkg_target_version}'")

        hoc_pkg_desc["version"] = pkg_target_version

        apply_hoc_wait(hoc_data, first_lcmmachine, cluster, ns, hoc_applied)
        hoc_applied = True

        LOG.info(f"Check that package '{target_pkg}' with version '{pkg_target_version}' was installed "
                 f"on machine: {first_machine.name}")
        check_success_result_on_machine(machine=first_machine, cmd=pkg_installed_cmd)
        pkg_installed_version = get_pkg_installed_version(first_machine, target_pkg)
        assert pkg_installed_version == pkg_target_version, \
            f"Installed version mismatch for package {target_pkg}:\n" \
            f"Target version: '{pkg_target_version}';\n" \
            f"Installed version: '{pkg_installed_version}'"
        LOG.info(f"Check that package '{target_pkg}' with version '{pkg_target_version}' was pinned "
                 f"on machine: {first_machine.name}")
        pinned_pkgs = get_pinned_packages(first_machine)
        assert target_pkg in pinned_pkgs, f"Package '{target_pkg}' was not found in pinned packages!"
        assert pinned_pkgs[target_pkg] == pkg_target_version, \
            f"Pinned version for package '{target_pkg}' is not matching expected;\n" \
            f"Pinned version from the machine: '{pinned_pkgs[target_pkg]}';\n" \
            f"Expected version '{pkg_target_version}'"

    show_step(3)
    hoc_pkg_desc.pop('version', None)
    apply_hoc_wait(hoc_data, first_lcmmachine, cluster, ns, hoc_applied)
    pinned_pkgs = get_pinned_packages(first_machine)
    assert target_pkg not in pinned_pkgs, f"Package '{target_pkg}': " \
                                          f"expected not to be pinned but found in pinned packages!"

    show_step(4)
    # Restore pinning to check "absent" state
    pkg_target_version = target_pkg_available_versions[0]
    hoc_pkg_desc['version'] = pkg_target_version
    apply_hoc_wait(hoc_data, first_lcmmachine, cluster, ns, hoc_applied)
    LOG.info(f"Check that package '{target_pkg}' with version '{pkg_target_version}' was pinned "
             f"on machine: {first_machine.name}")
    pinned_pkgs = get_pinned_packages(first_machine)
    assert target_pkg in pinned_pkgs, f"Package '{target_pkg}' was not found in pinned packages!"
    assert pinned_pkgs[target_pkg] == pkg_target_version, \
        f"Pinned version for package '{target_pkg}' is not matching expected;\n" \
        f"Pinned version from the machine: '{pinned_pkgs[target_pkg]}';\n" \
        f"Expected version '{pkg_target_version}'"

    show_step(5)
    hoc_pkg_desc.pop("version", None)
    hoc_pkg_desc["state"] = "absent"
    apply_hoc_wait(hoc_data, first_lcmmachine, cluster, ns, hoc_applied)

    check_failure_result_on_machine(machine=first_machine, cmd=pkg_installed_cmd)
    pinned_pkgs = get_pinned_packages(first_machine)
    assert target_pkg not in pinned_pkgs, f"Package '{target_pkg}': " \
                                          f"expected not to be pinned but found in pinned packages!"

    show_step(6)
    existing_config = ns.get_hostosconfiguration(name=hoc_name)
    LOG.info(f"Deleting HostOSConfiguration '{existing_config.name}'")
    existing_config.delete(async_req=True)
    timeout_msg = f"HostOSConfiguration {hoc_name} was not deleted"
    waiters.wait(lambda: not bool(ns.hostosconfiguration_is_present(name=hoc_name)),
                 timeout=1200,
                 interval=10,
                 timeout_msg=timeout_msg)
    first_machine.remove_machine_labels(list(day2_label.keys()))
