import pytest

from datetime import datetime
from functools import cached_property

from si_tests import logger, settings

from si_tests.deployments.utils.namespace import NAMESPACE
from si_tests.managers.openstack_client_manager import OpenStackClientManager
from si_tests.utils import waiters, utils

LOG = logger.logger


class AutoCleanupBase(object):
    os_namespace = NAMESPACE.openstack
    ocm = OpenStackClientManager()
    osm = ocm.os_manager
    service_name = ''
    databases = []
    deleted_criterion = 'WHERE deleted like "1"'
    osdpl_name = settings.OSH_DEPLOYMENT_NAME
    osdpl = osm.get_openstackdeployment(osdpl_name)

    def run_autocleanup_job(self):

        # Creating a unique postfix for a new DB cleanup job
        name_postfix = utils.gen_random_string(3)
        name_prefix = f'{self.service_name}-db-purge'
        try:
            purge_cron_job = self.osm.api.cronjobs.list(namespace=self.os_namespace, name_prefix=name_prefix)[0]
        except IndexError:
            LOG.error(f'Please check that cronjob {name_prefix} exists')
            raise AssertionError(f'{self.service_name} purge job not found')
        job = purge_cron_job.read()
        job_template = job.spec.job_template
        self.job_name = f'{self.service_name}-db-purge-manual-{name_postfix}'
        job_template.metadata.name = self.job_name

        LOG.info(f'Creating a new {self.service_name} cronjob for DB cleanup')

        self.ocm.os_manager.api.jobs.create(body=job_template, namespace=self.os_namespace, log_body=False)

        waiters.wait(lambda: "'succeeded': 1" in str(self.osm.api.jobs.list(namespace=self.os_namespace,
                                                                            name_prefix=job_template.metadata.name)
                                                     [0].read()), timeout=300, interval=30)
        LOG.info(f'Job {self.job_name} finished succesfully')

    def get_deleted_entries(self, timestamp):

        LOG.info(f'Counting the numbers of records with positive field "deleted" in all {self.service_name} tables')

        tables_with_deleted_entries = {}
        for db in self.database_tables.keys():
            for table in self.database_tables[db]:
                if table in self.skip_tables.keys():
                    continue
                cmd = f'''mariadb --user=root -p{self.osm.db_password} --database={db} --skip-column-names ''' \
                      f'''--execute='SELECT COUNT(deleted_at) FROM {table} {self.deleted_criterion} ''' \
                      f'''and deleted_at <= "{timestamp}"';'''
                response = self.osm.run_mysql_command(cmd, stderr=False).strip()
                result = int(response) if response else 0
                if result != 0:
                    if db not in tables_with_deleted_entries.keys():
                        tables_with_deleted_entries[db] = {}
                    tables_with_deleted_entries[db][table] = result

        LOG.info(f'The numbers of records with positive field "deleted" in all {self.service_name} tables were taken')

        return tables_with_deleted_entries

    @cached_property
    def database_tables(self):
        result = {}
        for db in self.databases:
            cmd = f"mariadb --user=root -p{self.osm.db_password} --database={db} --skip-column-names" \
                   " --execute='show tables;'"
            response = self.osm.run_mysql_command(cmd, stderr=False).strip()
            tables_list = [t for t in response.split('\n')]
            assert tables_list, f'No tables in DB for {self.service_name} were found'
            result[db] = tables_list

        return result

    def _test_autocleanup(self):
        # Getting the list of the tables in DB

        # Getting the current timestamp to check that before the beginning of the DB cleanup job there were any
        # records with a positive value in "deleted_at" field
        timestamp = datetime.now()

        result_before_cleanup = self.get_deleted_entries(timestamp)

        # Checking that there are deleted entries in DB
        if not result_before_cleanup:
            pytest.skip('There are no deleted entries in DB before cleanup job. Nothing to test. Skip test.')

        # Getting the timestamp of the end of the DB cleanup job to compare with values in "deleted_at"
        # tables' fields
        timestamp = datetime.now()

        self.run_autocleanup_job()

        result_after_cleanup = self.get_deleted_entries(timestamp)

        try:
            # Checking that all DB tables have no entries with a positive value in "deleted" fields
            assert not result_after_cleanup

        except AssertionError:
            for db in result_after_cleanup.keys():
                for table in result_after_cleanup[db].keys():
                    LOG.info(f'"{db}" data base has {result_after_cleanup[db][table]}'
                             f' record(s) with a positive value(s) in "deleted" field in "{table}" table')

            LOG.info(f'Creating {self.service_name} DB dump in /tmp/{self.service_name}-backup-file.sql file'
                     ' tables after previous created DB cleanup job')

            cmd = f'mariadb-dump {self.service_name} --user=root -p{self.osm.db_password}' \
                  f' > /tmp/{self.service_name}-backup-file.sql;'
            self.osm.run_mysql_command(cmd)

            raise AssertionError('Some record(s) in DB have positive value(s) in "deleted" field(s),'
                                 ' but they are not deleted from DB')

        LOG.info(f'There are no records with a positive value(s) in field(s) "deleted" in all {self.service_name}'
                 ' tables after previous created DB cleanup job')


class TestAutoCleanupCinder(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Cinder entries from DB
        Parameters required for the test execution:
            - KUBECONFIG
        Scenario:
            1. Count the number of deleted entries in all Cinder DB tables before cleanup
            2. Create a new Cinder cronjob for DB cleanup
            3. Count the number of deleted entries in all Cinder DB tables after cleanup
            4. Compare the results before and after the cleanup job
    '''
    service_name = 'cinder'
    databases = ['cinder']

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    def test_autocleanup_cinder(self):
        self._test_autocleanup()


class TestAutoCleanupBarbican(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Barbican entries from DB
            Parameters required for the test execution:
                - KUBECONFIG
            Scenario:
                1. Count the number of deleted entries in all Barbican DB tables before cleanup
                2. Create a new Barbican cronjob for DB cleanup
                3. Count the number of deleted entries in all Barbican DB tables after cleanup
                4. Compare the results before and after the cleanup job
    '''
    service_name = 'barbican'
    databases = ['barbican']

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    def test_autocleanup_barbican(self):
        self._test_autocleanup()


class TestAutoCleanupNova(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Nova entries from DB
            Parameters required for the test execution:
                - KUBECONFIG
            Scenario:
                1. Count the number of deleted entries in all Nova DB tables before cleanup
                2. Create a new Nova cronjob for DB cleanup
                3. Count the number of deleted entries in all Nova DB tables after cleanup
                4. Compare the results before and after the cleanup job
    '''
    service_name = 'nova'
    databases = ['nova', 'nova_api', 'nova_cell0']
    deleted_criterion = 'WHERE deleted not like "0"'

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    def test_autocleanup_nova(self):
        self._test_autocleanup()


class TestAutoCleanupHeat(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Heat entries from DB
            Parameters required for the test execution:
                - KUBECONFIG
            Scenario:
                1. Count the number of deleted entries in all Heat DB tables before cleanup
                2. Create a new Heat cronjob for DB cleanup
                3. Count the number of deleted entries in all Heat DB tables after cleanup
                4. Compare the results before and after the cleanup job
    '''
    service_name = 'heat'
    databases = ['heat']
    deleted_criterion = 'WHERE deleted_at is not NULL'

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    def test_autocleanup_heat(self):
        self._test_autocleanup()


class TestAutoCleanupGlance(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Glance entries from DB
            Parameters required for the test execution:
                - KUBECONFIG
            Scenario:
                1. Count the number of deleted entries in all Glance DB tables before cleanup
                2. Create a new Glance cronjob for DB cleanup
                3. Count the number of deleted entries in all Glance DB tables after cleanup
                4. Compare the results before and after the cleanup job
    '''
    service_name = 'glance'
    databases = ['glance']
    deleted_criterion = 'WHERE deleted_at is not NULL'

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    def test_autocleanup_glance(self):
        self._test_autocleanup()


class TestAutoCleanupMasakari(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Masakari entries from DB
            Parameters required for the test execution:
                - KUBECONFIG
            Scenario:
                1. Count the number of deleted entries in all Masakari DB tables before cleanup
                2. Create a new Masakari cronjob for DB cleanup
                3. Count the number of deleted entries in all Masakari DB tables after cleanup
                4. Compare the results before and after the cleanup job
    '''
    service_name = 'masakari'
    databases = ['masakari']
    deleted_criterion = 'WHERE deleted_at is not NULL'

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    @pytest.mark.skipif("instance-ha" not in OpenStackClientManager().os_manager.get_osdpl_deployment().data['spec'][
        'features']['services'],
        reason="Masakari is disabled on this deployment")
    def test_autocleanup_masakari(self):
        self._test_autocleanup()


class TestAutoCleanupManila(AutoCleanupBase):
    '''Verifies that autocleanup job purges deleted Manila entries from DB
            Parameters required for the test execution:
                - KUBECONFIG
            Scenario:
                1. Count the number of deleted entries in all Manila DB tables before cleanup
                2. Create a new Manila cronjob for DB cleanup
                3. Count the number of deleted entries in all Manila DB tables after cleanup
                4. Compare the results before and after the cleanup job
    '''
    service_name = 'manila'
    databases = ['manila']
    deleted_criterion = 'WHERE deleted_at is not NULL'

    # Dictionary of non-tested tables
    # Keys: names of skipped tables
    # Values: bug numbers
    skip_tables = {}

    @pytest.mark.skipif(
        "shared-file-system" not in OpenStackClientManager().os_manager.get_osdpl_deployment().data['spec']['features'][
            'services'], reason="Manila is disabled on this deployment")
    def test_autocleanup_manila(self):
        self._test_autocleanup()
