import pytest

from si_tests import logger
from si_tests import settings

LOG = logger.logger


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                         .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
def test_ha_kill_system_proc_in_cont(target_cluster, cluster_condition_check, _):
    """Kill main process in mke system containers
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node

    Scenario:
        1. SSH to each cluster node 1 by 1
        2. Collect list of container id's
        3. Kill containers 1 by 1
        4. Wait till all containers are running again
        5. Check that all pods are in a correct state

    Expected result - pods and their containers are Running and Ready.
    """
    target_cluster.check.check_ha_kill_system_proc_in_cont()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                         .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_ha_kill_ucp_cluster_agent(target_cluster, cluster_condition_check, _):
    """Kill ucp-cluster-agent container
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node

    Scenario:
        1. SSH to each cluster node 1 by 1
        2. Search for ucp-cluster-agent container
        3. Kill ucp-cluster-agent container
        4. Wait till container IS running
        5. Check that all pods are in a correct state
    Expected result - pods and container are Running and Ready.
    """
    target_cluster.check.check_ha_kill_ucp_cluster_agent()


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_ha_delete_helm_controller_leader_pod(target_cluster, cluster_condition_check, helmbundles_check, _):
    """Detect and delete leader pod for helm-controller app. Check that leader changed
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for selected application

    Scenario:
        1. Detect leader
        2. Delete leader pod
        3. Wait till leader was changed
        4. Check pods statuses
        5. Check helmbundles statuses

    Expected result - pods are recreated, number of replicas is restored, leader changed
    """
    target_cluster.check.check_ha_delete_leader_pod(app_group='helm-controller')
    target_cluster.check.check_ha_delete_leader_pod(app_group='stacklight-helm-controller')


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                         .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_ha_restart_docker_service(target_cluster, cluster_condition_check, _):
    """Restart docker service on every cluster node (one at a time)
    Precondition - all expected pods and their replicas must be presented
    The following scenario is executed for every node

    Scenario:
        1. SSH to node
        2. Restart docker service
        3. Check docker services are OK
        4. Check that all pods are Running and Ready

    Expected result - all docker services are OK after
    docker was restarted.
    """
    target_cluster.check.check_ha_restart_docker_service()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}"
                         .format(settings.TARGET_CLUSTER)])
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_ha_haproxy(target_cluster, cluster_condition_check, _):
    """Check that HAProxy balance load between api

    Scenario:
        1. Left keepalived on one node by one
        2. Stop api service on two of tree nodes for each node
        3. Check health of api
        4. Restore api service
        5. Resore keepalived

    Expected result - HAProxy will catch service degradation and api should
    be operational.
    """
    target_cluster.check.check_ha_haproxy_lb()


@pytest.mark.parametrize("_", ["CLUSTER_NAME={0}".format(settings.TARGET_CLUSTER)])
@pytest.mark.parametrize("svc_name,svc_ns,expected_downtime", settings.HA_SL_SVC_TIMEOUT)
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
def test_ha_sl_svc(target_cluster, svc_name, svc_ns, expected_downtime, _):
    """Delete svc pod or leader pod if STACKLIGHT_ENABLE_HA and check SVC downtime

     Scenario:
         1. Compare actual number of replicas for pod with expected
         2. Iterate by each replica
         3. Delete pod(s)
         4. Wait till number of replicas will be restored
         5. Check downtime
         6. Check pods statuses in this group (Running and Ready)

     Expected result - pods are recreated, number of replicas is restored.
     """

    if svc_name == "iam-proxy-kibana":
        if not target_cluster.logging_enabled():
            msg = "SL logging disabled. Kibana service is missing"
            LOG.warning(msg)
            pytest.skip(msg)
    target_cluster.check.check_svc_downtime(service_name=svc_name, namespace=svc_ns,
                                            expected_downtime=expected_downtime)
