import json

import pytest
import requests
import yaml

import si_tests.utils.waiters as helpers
from si_tests import logger
from si_tests import settings
from si_tests.clients.dashboard.mcc_dashboard_client import MCCDashboardClient
from si_tests.clients.dashboard.mke_dashboard_client import (
    MKEDashboardClientOpenid)
from si_tests.clients.iam import keycloak_client
from si_tests.managers.kaas_manager import Cluster
from si_tests.utils import utils, templates

LOG = logger.logger

FIRST_UUID_PART_LENGTH = 8


@pytest.fixture
def keycloak_client_fixture(kaas_manager):
    keycloak_ip = kaas_manager.get_keycloak_ip()
    pwd = kaas_manager.get_secret_data(
        "iam-api-secrets", "kaas", "keycloak_password")
    kclient = keycloak_client.KeycloakAdminClient(
        ip=keycloak_ip, user=settings.KEYCLOAK_USER, password=pwd)
    return kclient


@pytest.fixture
def keycloak_user_fixture(kaas_manager, keycloak_client_fixture):
    username = utils.gen_random_string(6)
    password = utils.gen_random_string(10)
    uid = keycloak_client_fixture.user_create(username=username,
                                              password=password)
    iamusername = f"{username}-{uid[:FIRST_UUID_PART_LENGTH]}"

    def user_created():
        user = None
        try:
            user = kaas_manager.get_iamuser(iamusername).read().to_dict()
        except Exception:
            pass

        return user is not None

    helpers.wait(
        user_created, timeout=150, interval=15,
        timeout_msg="Timeout waiting for iamuser.")

    LOG.info(f"Create a new user {username}/{password} with uid {uid}")

    yield username, password, uid, iamusername

    LOG.info(f"Delete user {username}/{password} with uid {uid}")

    keycloak_client_fixture.user_delete(username)

    def user_deleted():
        user_found = True
        try:
            kaas_manager.get_iamuser(iamusername).read()
        except Exception as e:
            user_found = e.status != requests.codes.not_found
        return not user_found

    helpers.wait(
        user_deleted, timeout=150, interval=15,
        timeout_msg="Timeout cleaning up iamuser.")


def check_resource(d_client, check_resource_path):
    def inner():
        try:
            res = d_client.get_resource(check_resource_path)
        except requests.HTTPError as e:
            if e.response.status_code not in [
                    requests.codes.unauthorized, requests.codes.forbidden]:
                raise
            LOG.debug("HTTP error %s when trying to GET endpoint %s: %s",
                      e.response.status_code, check_resource_path, e)
            return None
        return res
    return inner


def find_user_realm_roles(kclient, userid, target_roles):
    roles = kclient.get_user_roles(userid)
    LOG.info("User %s roles are: %s; target roles are %s",
             userid, roles, target_roles)

    roles_copy = target_roles[:]
    for role in roles:
        if role["name"] in roles_copy:
            roles_copy.remove(role["name"])
    return len(roles_copy) == 0


def mke_pod_crud(
        client: MKEDashboardClientOpenid, cluster: Cluster,
        expected_status=None):
    """
    Simple CRUD test to check access to namespace or pod resources via MKE

    Args:
        client: MKEDashboardClientOpenid
        cluster: Cluster
        expected_status: 403 if IAM role does not grant access to resources

    Returns: None

    """
    rnd = utils.gen_random_string(10)
    ns_name = f"ns-mke-{rnd}"
    pod_name = f"pod-mke-{rnd}"
    base_image_repo = cluster.determine_mcp_docker_registry()
    node_name = cluster.get_k8s_node_names()[0]
    template = templates.render_template(
        settings.DUMMY_TEST_POD_YAML,
        {'POD_NAME': pod_name, 'NODE_NAME': node_name,
         'IMAGE_BASE_REPO': base_image_repo})
    json_body = json.dumps(yaml.load(template, Loader=yaml.SafeLoader))

    try:
        LOG.info(f"Create test ns {ns_name}")
        client.create_k8s_namespace(name=ns_name)
        LOG.info(f"Create test pod {pod_name}")
        client.create_k8s_pod(namespace=ns_name, body=json_body)
        LOG.info(f"Get test pod {pod_name}")
        client.get_k8s_pod(name=pod_name, namespace=ns_name)
        LOG.info(f"Delete test pod {pod_name}")
        client.delete_k8s_pod(name=pod_name, namespace=ns_name)
        LOG.info(f"Delete test ns {ns_name}")
        client.delete_k8s_namespace(name=ns_name)
        return True
    except requests.HTTPError as e:
        if e.response.status_code == expected_status:
            LOG.info("Expected request failure.\nCode: %s.\nMessage: %s",
                     e.response.status_code, e.response.content)
            return True
        else:
            return False
    finally:
        if expected_status is None and ns_name in client.get_k8s_namespace_names(raise_on_error=False):
            LOG.info("Cleanup ns and pod after MKE crud attempt")
            if pod_name in client.get_k8s_pods(namespace=ns_name):
                client.delete_k8s_pod(name=pod_name, namespace=ns_name)
            client.delete_k8s_namespace(name=ns_name)


def mcc_pod_crud(
        client: MCCDashboardClient, cluster: Cluster, expected_status=None):
    """
    Simple CRUD test to check access to namespace or pod resources via MCC

    Args:
        client: MCCDashboardClient
        cluster: Cluster
        expected_status: 403 if IAM role does not grant access to resources

    Returns: None

    """
    rnd = utils.gen_random_string(10)
    ns_name = f"ns-mcc-{rnd}"
    pod_name = f"pod-mcc-{rnd}"
    base_image_repo = cluster.determine_mcp_docker_registry()
    node_name = cluster.get_k8s_node_names()[0]
    template = templates.render_template(
        settings.DUMMY_TEST_POD_YAML,
        {'POD_NAME': pod_name, 'NODE_NAME': node_name,
         'IMAGE_BASE_REPO': base_image_repo})
    json_body = json.dumps(yaml.load(template, Loader=yaml.SafeLoader))

    try:
        LOG.info(f"Create test ns {ns_name}")
        client.create_k8s_namespace(name=ns_name)
        LOG.info(f"Create test pod {pod_name}")
        client.create_k8s_pod(namespace=ns_name, body=json_body)
        LOG.info(f"Get test pod {pod_name}")
        client.get_k8s_pod(name=pod_name, namespace=ns_name)
        LOG.info(f"Delete test pod {pod_name}")
        client.delete_k8s_pod(name=pod_name, namespace=ns_name)
        LOG.info(f"Delete test ns {ns_name}")
        client.delete_k8s_namespace(name=ns_name)
        return True
    except requests.HTTPError as e:
        if e.response.status_code == expected_status:
            LOG.info("Expected request failure.\nCode: %s.\nMessage: %s",
                     e.response.status_code, e.response.content)
            return True
        else:
            return False
    finally:
        if expected_status is None and ns_name in client.get_k8s_namespace_names(raise_on_error=False):
            LOG.info("Cleanup ns and pod after MCC crud attempt")
            if pod_name in client.get_k8s_pods(namespace=ns_name):
                client.delete_k8s_pod(name=pod_name, namespace=ns_name)
            client.delete_k8s_namespace(name=ns_name)


def test_add_iamrolebinding_user_failed(kaas_manager):
    """
    1. Apply IAMRoleBinding with incorrect username.
    """
    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)

    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    user_name = "{rnd_string}-user".format(rnd_string=rnd_string)
    LOG.info("IAMRoleBinding name - {iamrolebinding_name}, "
             "username - {username}".format(
                 iamrolebinding_name=iamrolebinding_name,
                 username=user_name)
             )
    try:
        ns.create_iamrolebinding(iamrolebinding_name, "operator", user_name)
    except Exception as e:
        assert e.status == requests.codes.bad_request, (
            'Wrong response code. {}'.format(e))
    else:
        pytest.fail("It is possible to create iamrolebinding"
                    "with wrong username")


def test_add_iamrolebinding_role_failed(kaas_manager, keycloak_user_fixture):
    """
    2. Create a new user.
    3. Apply IAMRoleBinding with incorrect rolename.
    4. Clean up user.
    """
    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    _, __, ___, iamusername = keycloak_user_fixture

    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    role_name = "{rnd_string}-role".format(rnd_string=rnd_string)
    LOG.info("IAMRoleBinding name - {iamrolebinding_name}, "
             "rolename - {rolename}".format(
                 iamrolebinding_name=iamrolebinding_name,
                 rolename=role_name)
             )
    try:
        ns.create_iamrolebinding(iamrolebinding_name, role_name,
                                 iamusername)
    except Exception as e:
        assert e.status == requests.codes.bad_request, (
            'Wrong response code. {}'.format(e))
    else:
        pytest.fail("It is possible to create iamrolebinding"
                    "with wrong rolename")


def test_add_iamclusterrolebinding_cluster_failed(
        kaas_manager, keycloak_user_fixture):
    """
    5. Create a new user.
    6. Apply IAMClusterRoleBinding with incorrect clustername.
    7. Clean up user.
    """
    rnd_string = utils.gen_random_string(6)
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    _, __, ___, iamusername = keycloak_user_fixture

    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    cluster_name = "{rnd_string}-cluster".format(rnd_string=rnd_string)
    LOG.info("IAMClusterRoleBinding name - {iamrolebinding_name}, "
             "clustername - {clustername}".format(
                 iamrolebinding_name=iamrolebinding_name,
                 clustername=cluster_name)
             )
    try:
        ns.create_iamclusterrolebinding(
            iamrolebinding_name, "operator", iamusername, cluster_name)
    except Exception as e:
        assert e.status == requests.codes.bad_request, (
            'Wrong response code. {}'.format(e))
    else:
        pytest.fail("It is possible to create iamrolebinding"
                    "with wrong cluster name")


def test_add_iamrolebinding_success(
        kaas_manager, keycloak_client_fixture, keycloak_user_fixture):
    """
    8. Create a new user.
    9. Check that user with no roles can not access iamrolebindings.
    10. Create IAMRoleBinding with correct parameters.
    11. Request roles list from KeyCloak to check role presence.
    12. Check that user with operator role can access iamrolebindings.
    13. Remove IAMRoleBinding from k8s API.
    14. Request roles list from KeyCloak to check role absence.
    15. Create another IAMRoleBinding.
    16. Clean up user. Check that IAMUser object is cleaned up even if
        IAMRoleBinding is present before deletion.
    """
    username, password, uid, iamusername = keycloak_user_fixture

    check_resource_path = (
        f"apis/iam.mirantis.com/v1alpha1/namespaces/"
        f"{settings.TARGET_NAMESPACE}/iamrolebindings")
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    # Negative check
    LOG.info(f"Login to UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    d_resources = d_client.get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{d_resources}")
    assert (
        'code' in d_resources and
        d_resources['code'] == requests.codes.forbidden
    ), (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    rnd_string = utils.gen_random_string(6)
    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    ns.create_iamrolebinding(iamrolebinding_name, "operator", iamusername)
    target_roles = ["m:kaas:%s@operator" % settings.TARGET_NAMESPACE]
    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=15, interval=5,
        timeout_msg="Timeout waiting for keycloak role assignment.")

    # Re-login with new grants
    LOG.info(f"Re-login to UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed now")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)

    # Positive check
    helpers.wait(
        check_resource(d_client, check_resource_path), timeout=60, interval=5,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )

    iamrb = ns.get_iamrolebinding(iamrolebinding_name)
    iamrb.delete()
    helpers.wait(
        lambda: not find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=15, interval=5,
        timeout_msg="Timeout waiting for keycloak role removal.")

    rnd_string = utils.gen_random_string(6)
    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    ns.create_iamrolebinding(iamrolebinding_name, "user", iamusername)
    target_roles = ["m:kaas:%s@user" % settings.TARGET_NAMESPACE]
    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=15, interval=5,
        timeout_msg="Timeout waiting for keycloak role assignment.")


def test_add_iamuser_keycloak_external_legacy_reader(
        kaas_manager, keycloak_client_fixture, keycloak_user_fixture):
    """
    17. Create a user.
    18. Check that user with no roles can not list namespaces.
    19. Assign legacy m:kaas@reader role in keycloak.
    20. Check if there is a proper IAMGlobalRoleBinding object.
    21. Check that user with can list namespaces now.
    22. Set IAMGlobalRoleBinding legacy flag to false.
    23. Check if legacy roles' style changed to new in keycloak
    24. Create new namespace.
    25. Check that role corresponding to new namespace is assigned to user.
    26. Delete namespace.
    27. Check that corresponding role is deleted.
    28. Check that user still can list namespaces.
    29. Cleanup user.
    """
    username, password, uid, iamusername = keycloak_user_fixture

    check_resource_path = "api/v1/namespaces"
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    # Negative check
    LOG.info(f"Login to UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    d_resources = d_client.get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{d_resources}")
    assert (
        'code' in d_resources and
        d_resources['code'] == requests.codes.forbidden
    ), (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    scope = "m:kaas"
    role = "reader"
    roles = ["{scope}@{role}".format(scope=scope, role=role)]
    keycloak_client_fixture.assign_roles(username, roles=roles)

    expected_rb_name = "{username}-{role}".format(
        username=iamusername, role="user")

    def globalrolebinding_created():
        globalrb = None
        try:
            globalrb = kaas_manager.get_iamglobalrolebinding(
                expected_rb_name).read().to_dict()
        except Exception:
            pass

        return globalrb is not None

    helpers.wait(
        globalrolebinding_created, timeout=150, interval=15,
        timeout_msg="Timeout waiting for iamglobalrolebinding.")

    # Re-login with new grants
    LOG.info(f"Re-login to UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed now")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    # Positive check
    helpers.wait(
        check_resource(d_client, check_resource_path), timeout=60, interval=5,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )

    globalrb = kaas_manager.get_iamglobalrolebinding(expected_rb_name)
    globalrb.patch({"legacy": False})

    target_namespaces = [
        ns.name for ns in kaas_manager.get_namespaces()] + ["default"]
    target_roles = ["m:kaas:%s@user" % ns for ns in target_namespaces]

    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=15, interval=5,
        timeout_msg="Timeout waiting for keycloak role change.")

    ns = kaas_manager.create_namespace(username)
    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, ["m:kaas:%s@user" % username]),
        timeout=60, interval=5,
        timeout_msg="Timeout waiting for keycloak role addition.")
    ns.delete()
    helpers.wait(
        lambda: not find_user_realm_roles(
            keycloak_client_fixture, uid, ["m:kaas:%s@user" % username]),
        timeout=60, interval=5,
        timeout_msg="Timeout waiting for keycloak role removal.")

    # New style roles should work the same way
    LOG.info(f"Re-login to UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    # Positive check
    helpers.wait(
        check_resource(d_client, check_resource_path), timeout=60, interval=5,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )


def test_add_iamuser_keycloak_external_legacy_writer(
        kaas_manager, keycloak_client_fixture, keycloak_user_fixture):
    """
    30. Create a user.
    31. Check that user with no roles can not list iamglobalrolebindings.
    32. Assign legacy m:kaas@writer role in keycloak.
    33. Check if there are corresponding IAMGlobalRoleBinding objects.
    34. Check that user with can list iamglobalrolebindings now.
    35. Set IAMGlobalRoleBindings' legacy flag to false.
    36. Check if legacy roles' style changed to new in keycloak
    37. Check that user still can list iamglobalrolebindings.
    38. Cleanup user.
    """
    username, password, uid, iamusername = keycloak_user_fixture

    check_resource_path = (
        "apis/iam.mirantis.com/v1alpha1/iamglobalrolebindings")
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    # Negative check
    LOG.info(f"Login to UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    d_resources = d_client.get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{d_resources}")
    assert (
        'code' in d_resources and
        d_resources['code'] == requests.codes.forbidden
    ), (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    scope = "m:kaas"
    role = "writer"
    roles = ["{scope}@{role}".format(scope=scope, role=role)]
    keycloak_client_fixture.assign_roles(username, roles=roles)

    expected_rb_names = [
        "{username}-{role}".format(username=iamusername, role=r)
        for r in ("operator", "global-admin")]

    def globalrolebindings_created():
        globalrbs = []
        for expected_rb_name in expected_rb_names:
            try:
                globalrbs.append(
                    kaas_manager.get_iamglobalrolebinding(
                        expected_rb_name).read().to_dict())
            except Exception:
                pass

        return len(globalrbs) == 2

    helpers.wait(
        globalrolebindings_created, timeout=150, interval=15,
        timeout_msg="Timeout waiting for iamglobalrolebindings.")

    # Re-login with new grants
    LOG.info(f"Re-login to UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed now")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    # Positive check
    helpers.wait(
        check_resource(d_client, check_resource_path), timeout=60, interval=5,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )

    for expected_rb_name in expected_rb_names:
        globalrb = kaas_manager.get_iamglobalrolebinding(expected_rb_name)
        globalrb.patch({"legacy": False})

    target_namespaces = [
        ns.name for ns in kaas_manager.get_namespaces()] + ["default"]
    target_roles = ["m:kaas:%s@operator" % ns for ns in target_namespaces]
    target_roles.append("m:kaas@global-admin")

    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=15, interval=5,
        timeout_msg="Timeout waiting for keycloak role change.")
    helpers.wait(
        lambda: not find_user_realm_roles(
            keycloak_client_fixture, uid, ["m:kaas@writer"]),
        timeout=60, interval=5,
        timeout_msg="Timeout waiting for keycloak role removal.")

    # New style roles should work the same way
    LOG.info(f"Re-login to UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    # Positive check
    helpers.wait(
        check_resource(d_client, check_resource_path), timeout=60, interval=5,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )


def test_management_admin_with_relogin(
        kaas_manager, keycloak_client_fixture, keycloak_user_fixture):
    """
    39. Create a new user.
    40. Check that user with no roles cannot access pods.
    41. Check that user with no roles cannot manage pods.
    42. Create IAMGlobalRoleBinding with correct parameters.
    43. Wait for new role to appear in keycloak.
    44. Check that user with management-admin role can access pods with re-login.
    45. Check that user with management-admin role can manage pods with re-login.
    46. Delete IAMGlobalRoleBinding.
    47. Wait for new role to disappear in keycloak.
    48. Check that user without management-admin role cannot access pods with re-login.
    49. Check that user without management-admin role cannot manage pods with re-login.
    50. Clean up user.
    """
    username, password, uid, iamusername = keycloak_user_fixture
    namespace = "kaas"

    check_resource_path = f"api/v1/namespaces/{namespace}/pods"
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    client_id = 'k8s' if cluster.is_child else 'kaas'

    LOG.info(f"Login to MCC UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    d_resources = d_client.get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{d_resources}")
    assert (
        'code' in d_resources and
        d_resources['code'] == requests.codes.forbidden
    ), (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    LOG.info("Check MCC pod restrictions")
    mcc_pod_crud(d_client, cluster, expected_status=requests.codes.forbidden)

    LOG.info(f"Login to MKE UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    dashboard_url = cluster.data.get('status', {}).get(
        'providerStatus', {}).get('ucpDashboard')
    keycloak_ip = kaas_manager.get_keycloak_ip()
    mke_client = MKEDashboardClientOpenid(
        dashboard_url, keycloak_ip, username, password, client_id=client_id)

    mke_resources = mke_client._get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{mke_resources}")
    assert (
        'code' in mke_resources and
        mke_resources['code'] == requests.codes.forbidden
    ), (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    LOG.info("Check MKE pod restrictions")
    mke_pod_crud(mke_client, cluster, expected_status=requests.codes.forbidden)

    LOG.info("Check MKE isAdmin flag")
    helpers.wait(
        lambda: not mke_client._get_resource("id", raise_on_error=False).get('isAdmin'),
        timeout=900, interval=30,
        timeout_msg="MKE user isAdmin expected 'False' but actually 'True'")

    LOG.info("Apply new role")
    rnd_string = utils.gen_random_string(6)
    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    iamrolebinding = kaas_manager.create_iamglobalrolebinding(
        iamrolebinding_name, "management-admin", iamusername)
    target_roles = ["m:kaas@management-admin"]
    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=360, interval=15,
        timeout_msg="Timeout waiting for keycloak role assignment.")

    # Re-login with new grants
    LOG.info(f"Re-login to MCC UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed now")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    # MCC positive check. It takes up to 10 minutes for MKE to pick up admin permissions from
    # the token for a user
    helpers.wait(
        check_resource(d_client, check_resource_path),
        timeout=900, interval=30,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )
    LOG.info("Check MCC pod restrictions")
    helpers.wait(
        lambda: mcc_pod_crud(d_client, cluster),
        timeout=900, interval=60,
        timeout_msg="Timeout waiting MCC ns/pod crud with new role")

    LOG.info(f"Re-login to MKE UI using {username}, and check "
             f"that access to the {check_resource_path} is allowed now")
    mke_client = MKEDashboardClientOpenid(
        dashboard_url, keycloak_ip, username, password, client_id=client_id)

    if not cluster.workaround.field_6283():
        LOG.info("Check MKE isAdmin flag")
        helpers.wait(
            lambda: mke_client._get_resource("id", raise_on_error=False).get('isAdmin'),
            timeout=900, interval=30,
            timeout_msg="MKE user isAdmin expected 'True' but actually 'False'")

    # MKE positive check. It takes up to 10 minutes for MKE to pick up admin permissions from
    # the token for a user
    helpers.wait(
        lambda: mke_client._get_resource(check_resource_path, raise_on_error=False).get('items'),
        timeout=900, interval=30,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )

    LOG.info("Check MKE pod restrictions")
    helpers.wait(
        lambda: mke_pod_crud(mke_client, cluster),
        timeout=900, interval=60,
        timeout_msg="Timeout waiting MKE ns/pod crud with new role")

    LOG.info("Delete IAMGlobalRoleBinding")
    iamrolebinding.delete()

    LOG.info("Wait for new role to disappear in keycloak")
    helpers.wait(
        lambda: not find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=360, interval=15,
        timeout_msg="Timeout waiting for keycloak role assignment.")

    # Re-login with new grants
    LOG.info(f"Re-login to MCC UI using {username}, and check "
             f"that access to the {check_resource_path} is not allowed now")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)

    helpers.wait(
        lambda: d_client.get_resource(check_resource_path, raise_on_error=False).get('code') == 403,
        timeout=900, interval=30,
        timeout_msg=f"After timeout user still have access to {check_resource_path}")

    LOG.info("Check MCC pod restrictions")
    mcc_pod_crud(d_client, cluster, expected_status=requests.codes.forbidden)

    LOG.info(f"Re-login to MKE UI using {username}, and check "
             f"that access to the {check_resource_path} is not allowed now")
    mke_client = MKEDashboardClientOpenid(dashboard_url, keycloak_ip, username, password, client_id=client_id)

    LOG.info("Check MKE isAdmin flag")
    helpers.wait(
        lambda: not mke_client._get_resource("id", raise_on_error=False).get('isAdmin'),
        timeout=900, interval=30,
        timeout_msg="MKE user isAdmin expected 'False' but actually 'True'")

    helpers.wait(
        lambda: mke_client._get_resource(check_resource_path, raise_on_error=False).get('code') == 403,
        timeout=900, interval=30,
        timeout_msg=f"After timeout user still have access to {check_resource_path}")

    LOG.info("Check MKE pod restrictions")
    mke_pod_crud(mke_client, cluster, expected_status=requests.codes.forbidden)


def test_management_admin_without_relogin(
        kaas_manager, keycloak_client_fixture, keycloak_user_fixture):
    """
    51. Create a new user.
    52. Check that user with no roles cannot access pods.
    52. Check that user with no roles cannot manage pods.
    54. Create IAMGlobalRoleBinding with correct parameters.
    55. Wait for new role to appear in keycloak.
    56. Check that user with management-admin role can access pods without re-login.
    57. Check that user with management-admin role can manage pods without re-login.
    58. Delete IAMGlobalRoleBinding.
    59. Wait for new role to disappear in keycloak.
    50. Check that user without management-admin role cannot access pods without re-login.
    61. Check that user without management-admin role cannot manage pods without re-login.
    62. Clean up user.
    """
    username, password, uid, iamusername = keycloak_user_fixture
    namespace = "kaas"

    check_resource_path = f"api/v1/namespaces/{namespace}/pods"
    ns = kaas_manager.get_namespace(settings.TARGET_NAMESPACE)
    cluster = ns.get_cluster(settings.TARGET_CLUSTER)
    client_id = 'k8s' if cluster.is_child else 'kaas'

    if cluster.workaround.field_6283():
        pytest.skip("Test skip due to https://mirantis.jira.com/browse/FIELD-6283")

    LOG.info(f"Login to MCC UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    d_client = cluster.get_mcc_dashboardclient(
        user=username, password=password)
    d_resources = d_client.get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{d_resources}")
    assert 'code' in d_resources and d_resources['code'] == 403, (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    LOG.info("Check MCC pod restrictions")
    mcc_pod_crud(d_client, cluster, expected_status=requests.codes.forbidden)

    LOG.info(f"Login to MKE UI using {username}, and check that "
             f"access to the {check_resource_path} is restricted")
    dashboard_url = cluster.data.get('status', {}).get('providerStatus', {}).get('ucpDashboard')
    keycloak_ip = kaas_manager.get_keycloak_ip()
    mke_client = MKEDashboardClientOpenid(dashboard_url, keycloak_ip, username, password, client_id=client_id)

    mke_resources = mke_client._get_resource(
        check_resource_path, raise_on_error=False)
    LOG.debug(f"Response from {check_resource_path} using a new user "
              f"(must be restricted for the user with no roles):\n"
              f"{mke_resources}")
    assert 'code' in mke_resources and mke_resources['code'] == 403, (
        f"Access to {check_resource_path} is allowed "
        f"for the user with no roles, while it should be restricted")

    LOG.info("Check MKE pod restrictions")
    mke_pod_crud(mke_client, cluster, expected_status=requests.codes.forbidden)

    LOG.info("Check MKE isAdmin flag")
    helpers.wait(
        lambda: not mke_client._get_resource("id", raise_on_error=False).get('isAdmin'),
        timeout=900, interval=30,
        timeout_msg="MKE user isAdmin expected 'False' but actually 'True'")

    LOG.info("Apply new role")
    rnd_string = utils.gen_random_string(6)
    iamrolebinding_name = "{rnd_string}-iamrb".format(rnd_string=rnd_string)
    iamrolebinding = kaas_manager.create_iamglobalrolebinding(
        iamrolebinding_name, "management-admin", iamusername)
    target_roles = ["m:kaas@management-admin"]
    helpers.wait(
        lambda: find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=360, interval=15,
        timeout_msg="Timeout waiting for keycloak role assignment.")

    LOG.info("Check MKE isAdmin flag")
    helpers.wait(
        lambda: mke_client._get_resource("id", raise_on_error=False).get('isAdmin'),
        timeout=900, interval=30,
        timeout_msg="MKE user isAdmin expected 'True' but actually 'False'")

    LOG.info(f"Check MCC access to the {check_resource_path} is allowed now")
    helpers.wait(
        lambda: d_client.get_resource(check_resource_path, raise_on_error=False).get('items'),
        timeout=900, interval=30,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )

    LOG.info("Check MCC pod restrictions")
    helpers.wait(
        lambda: mcc_pod_crud(d_client, cluster),
        timeout=900, interval=60,
        timeout_msg="Timeout waiting MCC ns/pod crud with new role")

    LOG.info(f"Check MKE access to the {check_resource_path} is allowed now")
    helpers.wait(
        lambda: mke_client._get_resource(check_resource_path, raise_on_error=False).get('items'),
        timeout=900, interval=30,
        timeout_msg=f"Timeout waiting for {check_resource_path} to be allowed."
    )

    LOG.info("Check MKE pod restrictions")
    helpers.wait(
        lambda: mke_pod_crud(mke_client, cluster),
        timeout=900, interval=60,
        timeout_msg="Timeout waiting MKE ns/pod crud with new role")

    LOG.info("Delete IAMGlobalRoleBinding")
    iamrolebinding.delete()

    LOG.info("Wait for new role to disappear in keycloak")
    helpers.wait(
        lambda: not find_user_realm_roles(
            keycloak_client_fixture, uid, target_roles),
        timeout=360, interval=15,
        timeout_msg="Timeout waiting for keycloak role assignment.")

    LOG.info("Check MKE isAdmin flag")
    helpers.wait(
        lambda: not mke_client._get_resource("id", raise_on_error=False).get('isAdmin'),
        timeout=900, interval=30,
        timeout_msg="MKE user isAdmin expected 'False' but actually 'True'")

    LOG.info(f"Check MCC access to the {check_resource_path} is not allowed now")
    helpers.wait(
        lambda: d_client.get_resource(check_resource_path, raise_on_error=False).get('code') == 403,
        timeout=900, interval=30,
        timeout_msg=f"After timeout user still have access to {check_resource_path}")

    LOG.info("Check MCC pod restrictions")
    mcc_pod_crud(d_client, cluster, expected_status=requests.codes.forbidden)

    LOG.info(f"Check MKE access to the {check_resource_path} is not allowed now")
    helpers.wait(
        lambda: mke_client._get_resource(check_resource_path, raise_on_error=False).get('code') == 403,
        timeout=900, interval=30,
        timeout_msg=f"After timeout user still have access to {check_resource_path}")

    LOG.info("Check MKE pod restrictions")
    mke_pod_crud(mke_client, cluster, expected_status=requests.codes.forbidden)
