import yaml

from si_tests import logger
from si_tests import settings
from si_tests.utils import waiters, utils

LOG = logger.logger


def get_expected_checks(diagnostic_release, cluster):
    with open(f'si_tests/templates/expected-diagnostic-checks/{diagnostic_release}.yaml', 'r') as checks:
        available_checks = yaml.safe_load(checks).get('checks')
    if cluster.provider is not utils.Provider.baremetal:
        raise NotImplementedError("Checks list for non BM providers should be described in templates")
    base_checks = available_checks.get('core/baremetal')
    if cluster.is_management:
        return base_checks
    expected_checks = base_checks + available_checks.get('openstack')
    if cluster.is_sl_enabled():
        expected_checks += available_checks.get('stacklight')
    if cluster.tf_enabled():
        expected_checks += available_checks.get('tf')
    return expected_checks


def is_finished(diagnostic_object):
    status = diagnostic_object.data.get('status', {}) or {}
    f_at = status.get('finishedAt', 0)
    return True if f_at else False


def test_expected_diagnostic_checks_existance(kaas_manager, show_step):
    """Check that actual checks list as expected

     Scenario:
         1. Check init state cluster
         2. Create Diagnostic object for the cluster
         3. Wait for diagnostic checks to be finished
         4. Verify all expected checks are existed
     """

    cluster_name = settings.TARGET_CLUSTER
    ns_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(ns_name)
    cluster = ns.get_cluster(cluster_name)

    show_step(1)
    LOG.info(f"Check init state on the {cluster._cluster_type} cluster {cluster.namespace}/{cluster.name}")
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()

    show_step(2)
    active_diagnostic_release = kaas_manager.get_active_diagnosticrelease()
    active_diagnostic_release_name = active_diagnostic_release.name
    LOG.info(f"Active diagnostic release name is: {active_diagnostic_release_name}")

    expected_checks = get_expected_checks(active_diagnostic_release_name, cluster)
    LOG.info(f"Expected checks for {active_diagnostic_release_name}:\n{yaml.dump(expected_checks)}")

    test_obj_name = cluster_name + "-" + utils.gen_random_string(10)
    diagnostic_obj = cluster.create_diagnostic_object(object_name=test_obj_name)
    waiters.wait(lambda: is_finished(diagnostic_obj))
    existed_checks = diagnostic_obj.data.get('status', {}).get('result', {})
    existed_checks_names = list(existed_checks.keys())
    diff = set(existed_checks_names) ^ set(expected_checks)
    # Need to check are there extra checks in object, or some checks are missed
    if diff:
        extra_checks = set(existed_checks_names) - set(expected_checks)
        missing_checks = set(expected_checks) - set(existed_checks_names)
        if extra_checks:
            LOG.error(f"Found extra checks in diagnostic object:\n{extra_checks}")
        if missing_checks:
            LOG.error(f"Next checks are missed in diagnostic object:\n{missing_checks}")
        raise AssertionError(f"Actual checks list is not as expected.\n"
                             f"Expected:\n{yaml.dump(expected_checks)}\n"
                             f"Actual:\n{yaml.dump(existed_checks_names)}\n")

    LOG.info("All checks list as expected")
    # Check cluster is OK after diagnostic
    LOG.info("Check cluster readiness after diagnostic")
    cluster.check.check_machines_status()
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()
