import os

import yaml
from cassandra.cluster import Cluster
from packaging import version

import si_tests.utils.templates as template_utils
from si_tests import logger, settings
from si_tests.clients.k8s import K8sCluster
from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.daemonsets import K8sDaemonSet
from si_tests.clients.k8s.pods import K8sPod
from si_tests.clients.k8s.statefulsets import K8sStatefulSet
from si_tests.utils import utils, waiters

LOG = logger.logger


class TFManager(object):
    kubeconfig = None
    _tfoperator_version = None

    _tf_dbrestore = "tf-dbrestore"
    _tf_dbbackup_cronjob = "tf-dbbackup-job"
    _tf_dbrepair_cronjob = "tf-dbrepair-job"
    _cassandra_cluster_tf_config = "tf-cassandra-config"
    _backup_nfs_type = "pv_nfs"

    def __init__(self, kubeconfig=settings.KUBECONFIG_PATH):
        self._api = None
        self.kubeconfig = kubeconfig
        self.tf_namespace = "tf"
        self.tfoperator_name = settings.TF_OPERATOR_CR_NAME
        self.detect_api_version()
        self._pod_tf_cli = None

        # Remote name for rclone. Value defined in TFOperator
        self.rclone_remote = "tf_backup"

    def detect_api_version(self):
        self.apiv2 = False
        cr = self.get_k8s_resource(
            "tfoperator", self.tfoperator_name
        )
        cr_v2 = self.get_k8s_resource(
            "tfoperator_v2", self.tfoperator_name
        )
        if utils.is_k8s_res_exist(cr_v2):
            self.apiv2 = True
            LOG.info("TFOperator apiv2 detected")
        elif utils.is_k8s_res_exist(cr):
            self.apiv2 = False
            LOG.info("TFOperator apiv1 detected")
        else:
            LOG.info("TFOperator CR not found.")

    @property
    def api(self):
        """
        :rtype: cluster.K8sCluster
        """
        if self._api is None:
            if not os.path.isfile(self.kubeconfig):
                raise FileNotFoundError(
                    "kubeconfig file={} not found!".format(self.kubeconfig)
                )
            self._api = K8sCluster(kubeconfig=self.kubeconfig)
        return self._api

    def cql_session(self, ips):
        if not isinstance(ips, list):
            ips = [ips]
        return Cluster(ips).connect()

    def get_k8s_resource(self, resources_type, resource_name, read=False):
        resources_api = getattr(self.api, resources_type)
        resource = resources_api.get(name=resource_name, namespace=self.tf_namespace)
        if read:
            return resource.read()
        return resource

    def get_tf_helmbundle(self, read=False):
        return self.get_k8s_resource(
            "kaas_helmbundles", "tungstenfabric-operator", read
        )

    def get_tfoperator_deployment(self, read=False):
        return self.get_k8s_resource(
            "deployments", "tungstenfabric-operator", read
        )

    def tfoperator_version(self, cache=False):
        if self._tfoperator_version is None or cache is False:
            tfoperator_deployment = self.get_tfoperator_deployment()
            tf_deployment = tfoperator_deployment.read()
            tf_containers = tf_deployment.spec.template.spec.containers
            tf_container = [c for c in tf_containers if c.name == "manager" or c.name == "controller-v2"][0]
            img_ver = tf_container.image.split(':')[-1]
            # Cut stamp/hash for dev versions if present
            self._tfoperator_version = img_ver.split('-')[0]

        return self._tfoperator_version

    def get_tf_version(self):
        # Get TungstenFabric version from status
        return self.tfoperator(read=True).status.get("tfVersion")

    def tfoperator(self, read=False, detect_api=False):
        if detect_api:
            self.detect_api_version()
        resource_type = "tfoperator_v2" if self.apiv2 else "tfoperator"
        return self.get_k8s_resource(
            resource_type, self.tfoperator_name, read
        )

    def tfanalytics(self, read=False):
        resource_type = "tfanalytics_v2" if self.apiv2 else "tfanalytics"
        return self.get_k8s_resource(
            resource_type, "tf-analytics", read
        )

    def tfconfig(self, read=False):
        resource_type = "tfconfig_v2" if self.apiv2 else "tfconfig"
        return self.get_k8s_resource(
            resource_type, "tf-config", read
        )

    def tfcontrol(self, read=False):
        resource_type = "tfcontrol_v2" if self.apiv2 else "tfcontrol"
        return self.get_k8s_resource(
            resource_type, "tf-control", read
        )

    def tfvrouter(self, read=False):
        resource_type = "tfvrouter_v2" if self.apiv2 else "tfvrouter"
        return self.get_k8s_resource(
            resource_type, "tf-vrouter", read
        )

    def tftest(self, read=False):
        resource_type = "tftest_v2" if self.apiv2 else "tftest"
        return self.get_k8s_resource(
            resource_type, "tf-test", read
        )

    def tftool(self, read=False):
        resource_type = "tftool_v2" if self.apiv2 else "tftool"
        return self.get_k8s_resource(
            resource_type, "tf-tool", read
        )

    def tfwebui(self, read=False):
        resource_type = "tfwebui_v2" if self.apiv2 else "tfwebui"
        return self.get_k8s_resource(
            resource_type, "tf-webui", read
        )

    def tfimageprecaching(self, read=False):
        resource_type = "tfimageprecaching_v2" if self.apiv2 else "tfimageprecaching"
        return self.get_k8s_resource(
            resource_type, "tf-image-pre-caching", read
        )

    def tfdbbackup(self, read=False):
        resource_type = "tfdbbackup_v2" if self.apiv2 else "tfdbbackup"
        return self.get_k8s_resource(
            resource_type, "tf-dbbackup", read
        )

    def tfdbrestore(self, read=False):
        resource_type = "tfdbrestores_v2" if self.apiv2 else "tfdbrestores"
        return self.get_k8s_resource(
            resource_type, self._tf_dbrestore, read
        )

    def tfdbrepair(self, read=False):
        return self.get_k8s_resource(
            "tfdbrepairs", "tf-dbrepair", read
        )

    # 3rd party components
    def cassandracluster(self, read=False):
        return self.get_k8s_resource(
            "cassandracluster", self._cassandra_cluster_tf_config, read
        )

    def is_tf_tfdbbackup_remote_enabled(self):
        # Only apiv2 supports remoteSync
        path = 'spec:features:dbBackup:remoteSync:enabled'
        return self._get(self.tfoperator().data, path, default=False)

    def is_tfdbrestore_present(self):
        return self.api.tfdbrestores_v2.present(self._tf_dbrestore, self.tf_namespace)

    def _trigger_cronjob(self, cronjob):
        job_name = f"{cronjob.metadata.name}-{utils.gen_random_string(3)}"
        job_template = cronjob.spec.job_template
        job_template.metadata.name = job_name
        self.api.jobs.create(body=job_template, namespace=self.tf_namespace,
                             log_body=False)
        job = self.api.jobs.get(name=job_name, namespace=self.tf_namespace)
        return job

    def tfdbbackup_cronjob(self, read=False):
        return self.get_k8s_resource(
            "cronjobs", self._tf_dbbackup_cronjob, read
        )

    def is_tfdbbackup_cronjob_present(self):
        return self.api.cronjobs.present(self._tf_dbbackup_cronjob, self.tf_namespace)

    def trigger_tfdbbackup_cronjob(self):
        cronjob = self.tfdbbackup_cronjob(read=True)
        job = self._trigger_cronjob(cronjob)
        job.wait_finished()
        assert job.succeeded, f"Job {job.name} isn't succeeded"
        return job

    def get_db_pvc_name(self):
        path = 'spec:features:dbBackup:backupType'
        backup_type = self._get(self.tfoperator().data, path, default=None)
        pvc_name = "tf-dbbackup-nfs-pvc" if backup_type == self._backup_nfs_type else "tf-dbbackup-pvc"
        return pvc_name

    def tfdbrepair_cronjob(self, read=False):
        return self.get_k8s_resource(
            "cronjobs", self._tf_dbrepair_cronjob, read
        )

    def is_tfdbrepair_cronjob_present(self):
        return self.api.cronjobs.present(self._tf_dbrepair_cronjob, self.tf_namespace)

    def trigger_tfdbrepair_cronjob(self):
        cronjob = self.tfdbrepair_cronjob(read=True)
        job = self._trigger_cronjob(cronjob)
        job.wait_finished()
        assert job.succeeded, f"Job {job.name} isn't succeeded"
        return job

    def wait_casasndracluster_status(self, status="Running", timeout=600):
        cassandracluster = self.cassandracluster()
        waiters.wait(lambda: cassandracluster.data['status']['phase'] == status,
                     timeout=timeout, interval=5,
                     timeout_msg=f"Timeout wait for getting status {status}")

    def get_tf_control_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector='app=tf-control')

    def wait_tfcontrol_status(self, status="Ready", timeout=300):
        LOG.info(f"Wait until TF Control status becomes {status}")
        waiters.wait(lambda: self.tfcontrol().data['status']['health'] == status,
                     timeout=timeout, interval=3,
                     timeout_msg=f"Timeout wait for getting status {status}")

    def get_config_daemonset(self, read=False):
        return self.get_k8s_resource(
            "daemonsets", "tf-config", read
        )

    def get_tf_config_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector='tungstenfabric=config')

    def wait_tfconfig_status(self, status="Ready", timeout=300):
        LOG.info(f"Wait until TF Config status becomes {status}")
        waiters.wait(lambda: self.tfconfig().data['status']['health'] == status,
                     timeout=timeout, interval=3,
                     timeout_msg=f"Timeout wait for getting status {status}")

    def get_precaching_daemonsets(self):
        return self.api.daemonsets.list(namespace=self.tf_namespace,
                                        label_selector="app=tf-image-pre-caching")

    def get_vrouter_daemonsets(self):
        daemonsets = []
        for kind in ['agent', 'agent-dpdk']:
            label = ('app=tf-vrouter-{}'.format(kind))
            ds_filtered = self.api.daemonsets.list(namespace=self.tf_namespace,
                                                   label_selector=label)
            daemonsets.extend(ds_filtered)
        return daemonsets

    def get_vrouter_pods(self):
        pods = []
        for kind in ['agent', 'agent-dpdk']:
            label = ('app=tf-vrouter-{}'.format(kind))
            pods_filtered = self.api.pods.list(namespace=self.tf_namespace,
                                               label_selector=label)
            pods.extend(pods_filtered)
        return pods

    def get_vrouter_nodes(self):
        nodes = []
        selectors = ['tfvrouter=enabled', 'tfvrouter-dpdk=enabled']
        for selector in selectors:
            nodes.extend(self.api.nodes.list(
                namespace=self.tf_namespace,
                label_selector=selector
            ))
        return nodes

    def update_tfvrouter_pods(self):
        path = 'spec:containers'
        if not self.is_vrouter_component_updated():
            LOG.info("TFVrouter components weren't updated. Restart vRouter pods")
            for pod in self.get_vrouter_pods():
                containers = self._get(pod.data, path, default=[])
                container = self._get_item_by_attribute(containers, 'name', 'agent')
                LOG.info(f"Delete {pod.name} pod (image: {container['image']})")
                pod.delete()
            LOG.info("Wait until TFVrouter is updated.")
            waiters.wait(
                self.is_vrouter_component_updated,
                timeout=180,
                timeout_msg="TFVrouter components weren't updated",
            )
        LOG.info("TFVrouter components are updated:")
        for pod in self.get_vrouter_pods():
            containers = self._get(pod.data, path, default=[])
            container = self._get_item_by_attribute(containers, 'name', 'agent')
            LOG.info(f"{pod.name} image: {container['image']}")

    def get_analytics_nodes(self):
        nodes = []
        selectors = ['tfanalytics=enabled,tfanalyticsdb=enabled',
                     'tfanalytics=disabled,tfanalyticsdb=disabled']
        for selector in selectors:
            nodes.extend(self.api.nodes.list(
                namespace=self.tf_namespace,
                label_selector=selector
            ))
        return nodes

    def get_tf_services_cfgmap(self):
        return self.api.configmaps.get(name="tf-services-cfgmap",
                                       namespace=self.tf_namespace)

    @staticmethod
    def _is_k8s_res_not_exist(k8s_res: K8sNamespacedResource):
        return not utils.is_k8s_res_exist(k8s_res)

    @staticmethod
    def _is_ds_updated(k8s_obj: K8sDaemonSet):
        current_num = k8s_obj.data['status'].get('current_number_scheduled')
        updated_num = k8s_obj.data['status'].get('updated_number_scheduled')
        LOG.info(f"DaemonSet {k8s_obj} update status: {updated_num}/{current_num}")
        return updated_num == current_num

    @staticmethod
    def _is_ds_ready(k8s_obj: K8sDaemonSet):
        desired_num = k8s_obj.data['status'].get('desired_number_scheduled')
        ready_num = k8s_obj.data['status'].get('number_ready')
        LOG.info(f"DaemonSet {k8s_obj} ready status: {ready_num}/{desired_num}")
        return ready_num == desired_num

    @staticmethod
    def _is_sts_ready(k8s_obj: K8sStatefulSet):
        desired_num = k8s_obj.data['status'].get('replicas')
        ready_num = k8s_obj.data['status'].get('ready_replicas')
        LOG.info(f"StatefulSet {k8s_obj} ready status: {ready_num}/{desired_num}")
        return ready_num == desired_num

    @staticmethod
    def _get(dict_obj: dict, path, default=None):
        for key in path.split(':'):
            dict_obj = dict_obj.get(key, {})
        if dict_obj == {}:
            dict_obj = default
        return dict_obj

    @staticmethod
    def _get_item_by_attribute(list_obj: list, attr, name):
        for obj in list_obj:
            if obj.get(attr, None) == name:
                return obj
        return None

    def _get_env_var(self, k8s_obj: K8sNamespacedResource, path, env_name):
        envs = self._get(k8s_obj.data, path, default=[])
        env = list(filter(lambda x: x['name'] == env_name, envs))[0]['value']
        return env

    def get_storage_class(self):
        path = 'spec:dataStorageClass' if self.apiv2 else \
            'spec:settings:dataStorageClass'
        return self._get(self.tfoperator().data, path, default=False)

    def set_netns_availability_zone(self, az_name):
        if self.apiv2:
            patch = {
                "spec": {
                    "features": {
                        "config": {
                            "netnsAZ": az_name
                        }
                    }
                }
            }
        else:
            patch = {
                "spec": {
                    "controllers": {
                        "tf-config": {
                            "svc-monitor": {
                                "containers": [
                                    {
                                        "name": "svc-monitor",
                                        "env": [{"name": "NETNS_AVAILABILITY_ZONE",
                                                 "value": az_name}]
                                    }
                                ]
                            }
                        }
                    }
                }
            }
        self.tfoperator().patch(patch)

        def tf_config_netns_az():
            env_name = 'NETNS_AVAILABILITY_ZONE'
            if self.apiv2:
                path = 'spec:envSettings:svcMonitor'
                envs = self._get(self.tfconfig().data, path, default=[])
                env = self._get_item_by_attribute(envs, 'name', env_name)
            else:
                path = 'spec:svc-monitor:containers'
                containers = self._get(self.tfconfig().data, path, default=[])
                container = self._get_item_by_attribute(containers, 'name', 'svc-monitor')
                envs = container['env']
                env = self._get_item_by_attribute(envs, 'name', env_name)
            return env['value'] if env else env

        def ds_config_netns_az():
            env_name = 'NETNS_AVAILABILITY_ZONE'
            path = 'spec:template:spec:containers'
            containers = self._get(self.get_config_daemonset().data, path)
            container = self._get_item_by_attribute(containers, 'name', 'svc-monitor')
            env = self._get_item_by_attribute(container['env'], 'name', env_name)
            return env['value'] if env else env

        waiters.wait(lambda: tf_config_netns_az() == az_name, timeout=180, interval=10)
        waiters.wait(lambda: ds_config_netns_az() == az_name, timeout=180, interval=10)
        waiters.wait(self._is_ds_updated, predicate_args=[self.get_config_daemonset()],
                     timeout=180, interval=10)
        waiters.wait(self._is_ds_ready, predicate_args=[self.get_config_daemonset()],
                     timeout=180, interval=10)

    def is_tf_cli_enabled(self):
        path = 'spec:features:tfTools:tfCliEnabled' if self.apiv2 else \
            'spec:controllers:tf-tool:tf-cli:enabled'
        return self._get(self.tfoperator().data, path, default=False)

    def enable_tf_cli(self, state=True):
        LOG.info(f"Set TF cli to ${state}")
        tfoperator = self.tfoperator()
        current = self.is_tf_cli_enabled()
        if current == state:
            LOG.info(f"TF cli state is already: {current}")
        else:
            if self.apiv2:
                patch = {
                    "spec": {
                        "features": {
                            "tfTools": {
                                "tfCliEnabled": state
                            }
                        }
                    }
                }
            else:
                patch = {
                    "spec": {
                        "controllers": {
                            "tf-tool": {
                                "tf-cli": {
                                    "enabled": state
                                }
                            }
                        }
                    }
                }
            tfoperator.patch(patch)

    def is_ctools_enabled(self):
        path = 'spec:features:tfTools:tfToolsEnabled' if self.apiv2 else \
            'spec:controllers:tf-tool:tools:enabled'
        return self._get(self.tfoperator().data, path, default=False)

    def enable_ctools(self, state=True, labels=None):
        LOG.info(f"Set TF contrail-tools to ${state}")
        if not labels:
            labels = {'tfvrouter': 'enabled'}
        tfoperator = self.tfoperator()
        current = self.is_ctools_enabled()
        if current == state:
            LOG.info(f"TF contrail-tools (ctools) state is already: {current}")
        else:
            if self.apiv2:
                patch = {
                    "spec": {
                        "features": {
                            "tfTools": {
                                "tfToolsEnabled": state,
                                "labels": labels
                            }
                        }
                    }
                }
            else:
                patch = {
                    "spec": {
                        "controllers": {
                            "tf-tool": {
                                "tools": {
                                    "enabled": state,
                                    "labels": labels
                                }
                            }
                        }
                    }
                }
            tfoperator.patch(patch)

    def is_image_precaching_enabled(self):
        path = 'spec:features:imagePreCaching' if self.apiv2 else \
            'spec:settings:imagePreCaching'
        return self._get(self.tfoperator().data, path, default=True)

    def enable_image_precaching(self, state=True):
        LOG.info(f"Set TF imagePreCaching feature state: {state}")
        tfoperator = self.tfoperator()
        current = self.is_image_precaching_enabled()
        if current == state:
            LOG.info(f"TF imagePreCaching state is already: {current}")
        else:
            if self.apiv2:
                patch = {
                    "spec": {
                        "features": {
                            "imagePreCaching": state
                        }
                    }
                }
            else:
                patch = {
                    "spec": {
                        "settings": {
                            "imagePreCaching": state
                        }
                    }
                }
            tfoperator.patch(patch)

    def wait_tf_imageprecaching(self, exist=True, timeout=300, interval=10):
        LOG.info(f"Waiting until TF ImagePreCaching resource is existed == {exist}")
        waiters.wait(
            lambda: utils.is_k8s_res_exist(self.tfimageprecaching()) is exist,
            interval=interval,
            timeout=timeout,
        )

    def get_imageprecaching_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector='app=tf-image-pre-caching')

    def is_dbrepair_enabled(self):
        # Only apiv2 is supported
        path = 'spec:features:dbRepair:enabled'
        return self._get(self.tfoperator().data, path, default=False)

    def configure_dbrepair(self, state=True, suspend=None, schedule=None):
        patch = {
            "spec": {
                "features": {
                    "dbRepair": {
                        "enabled": state,
                    }
                }
            }
        }
        dbrepair = patch["spec"]["features"]["dbRepair"]
        if suspend:
            dbrepair["suspend"] = suspend
        if schedule:
            dbrepair["schedule"] = schedule
        self.tfoperator().patch(patch)

    def is_analytics_enabled(self):
        if version.parse(self.tfoperator_version()) >= version.parse("0.14"):
            default = False
        else:
            default = True
        if self.apiv2:
            path = 'spec:services:analytics:enabled'
            state = self._get(self.tfoperator().data, path, default=default)
        else:
            path = 'spec:settings:disableTFAnalytics'
            state = self._get(self.tfoperator().data, path, default=None)
            state = default if state is None else not state
        return state

    def enable_analytics(self, state=True):
        LOG.info(f"Set TF Analytics to ${state}")
        tfoperator = self.tfoperator()
        current = self.is_analytics_enabled()
        if current == state:
            LOG.info(f"TF Analytics state is already: {current}")
        else:
            if self.apiv2:
                patch = {
                    "spec": {
                        "services": {
                            "analytics": {
                                "enabled": state,
                            }
                        }
                    }
                }
            else:
                patch = {
                    "spec": {
                        "settings": {
                            "disableTFAnalytics": not state
                        }
                    }
                }
            tfoperator.patch(patch)
        self.wait_tf_analytics(exist=state)

    def wait_tf_analytics(self, exist=True, timeout=180, interval=10):
        tfanalytics = self.tfanalytics()
        if exist:
            LOG.info("Wait until TF Analytics resource is created")
            check_func = utils.is_k8s_res_exist
        else:
            LOG.info("Wait until TF Analytics resource is deleted")
            check_func = self._is_k8s_res_not_exist
        waiters.wait(
            check_func,
            predicate_args=[tfanalytics],
            interval=interval,
            timeout=timeout,
        )

    def is_gracefulRestart_enabled(self):
        path = 'spec:features:control:gracefulRestart:enabled' if self.apiv2 else \
            'spec:settings:gracefulRestart:enabled'
        return self._get(self.tfoperator().data, path, default=False)

    def db_backup(self, state=True):
        LOG.info(f"Set TF DB backup to ${state}")
        tfoperator = self.tfoperator()
        if self.apiv2:
            patch = {
                "spec": {
                    "features": {
                        "dbBackup": {
                            "enabled": state,
                        }
                    }
                }
            }
        else:
            patch = {
                "spec": {
                    "controllers": {
                        "tf-dbBackup": {
                            "enabled": state
                        }
                    }
                }
            }
        tfoperator.patch(patch)
        self.wait_db_backup_cronjob(exist=state)

    def wait_db_backup_cronjob(self, exist=True, timeout=180, interval=10):
        dbbackup_cronjob = self.tfdbbackup_cronjob()
        if exist:
            LOG.info("Wait until TF DB Backup cronjob is created")
            check_func = utils.is_k8s_res_exist
        else:
            LOG.info("Wait until TF DB Backup cronjob is deleted")
            check_func = self._is_k8s_res_not_exist
        waiters.wait(
            check_func,
            predicate_args=[dbbackup_cronjob],
            interval=interval,
            timeout=timeout,
        )

    def db_restore(self, backup_file=""):
        LOG.info("Enable TF DB Restore procedure")
        tfoperator = self.tfoperator()
        if self.apiv2:
            patch = {
                "spec": {
                    "features": {
                        "dbRestoreMode": {
                            "dbDumpName": backup_file,
                            "enabled": True,
                        }
                    }
                }
            }
        else:
            patch = {
                "spec": {
                    "settings": {
                        "dbRestoreMode": {
                            "enabled": True
                        }
                    }
                }
            }
        tfoperator.patch(patch)
        self.wait_db_restore()

    def wait_db_restore(self):
        tfdbrestore = self.tfdbrestore()
        waiters.wait(utils.is_k8s_res_exist, predicate_args=[tfdbrestore],
                     timeout=600, interval=10,
                     timeout_msg="Timeout wait for tfdbrestore object")
        waiters.wait(lambda: tfdbrestore.data['status']['health'] == "Ready",
                     timeout=7200, interval=30,
                     timeout_msg="Timeout wait for restoration status Ready")

    @property
    def _devOptions(self) -> dict:
        tf_operator_cr = self.tfoperator()
        spec = tf_operator_cr.data['spec']
        if self.apiv2:
            return spec.get('devOptions', {})
        else:
            return spec.get('settings', {}).get('devFeatures', {})

    def is_forceUpdate(self):
        return self._devOptions.get('forceUpdate', True)

    def is_vgw_disabled(self):
        return self._devOptions.get('disableVirtualGateway', False)

    def is_tftest_enabled(self):
        path = 'spec:features:tfTest:enabled' if self.apiv2 else \
            'spec:controllers:tf-test:tungsten-pytest:enabled'
        return self._get(self.tfoperator().data, path, default=False)

    def enable_tf_test(self, state=True):
        LOG.info(f"Set TF Test parameter enabled={state}")
        tfoperator = self.tfoperator()
        current = self.is_tftest_enabled()
        if current == state:
            LOG.info(f"TF Test (tungsten-pytest) state is already: {current}")
        else:
            if self.apiv2:
                patch = {
                    "spec": {
                        "features": {
                            "tfTest": {
                                "enabled": state,
                            }
                        }
                    }
                }
            else:
                patch = {
                    "spec": {
                        "controllers": {
                            "tf-test": {
                                "tungsten-pytest": {
                                    "containers": [{
                                        "name": "tungsten-pytest"
                                    }],
                                    "enabled": state
                                }
                            }
                        }
                    }
                }
            tfoperator.patch(patch)
        self.wait_tf_test(enabled=state)

    def wait_tf_test(self, enabled=True, timeout=300, interval=10):
        tftest = self.tftest()
        LOG.info(f"Wait until TF Test parameter enabled={enabled}")
        path = 'spec:enabled' if self.apiv2 else 'spec:tungsten-pytest:enabled'
        waiters.wait(
            lambda: self._get(tftest.data, path) == enabled,
            timeout=timeout, interval=interval,
            timeout_msg=f"Timeout wait for TF Test state {enabled}"
        )

    def run_pytest(self):
        # Run TF verification tests (tungsten-pytest) and return pod status.
        LOG.info("Run TF Test (tungsten-pytest)")
        tftest_pod_name = "tf-test-tungsten-pytest"
        if self.is_tftest_enabled():
            LOG.info("Disable tungsten-pytest before new run")
            self.enable_tf_test(state=False)
            # Wait until pod will be deleted
            waiters.wait(
                lambda: not self.api.pods.present(
                    tftest_pod_name, namespace=self.tf_namespace),
                timeout=180, interval=10
            )
        LOG.info("Enable tungsten-pytest")
        self.enable_tf_test(state=True)
        waiters.wait(
            lambda:
            self.api.pods.present(tftest_pod_name, namespace=self.tf_namespace),
            timeout=180, interval=10
        )
        test_pod = self.api.pods.get(tftest_pod_name,
                                     namespace=self.tf_namespace)
        LOG.info("Wait for pod to be Completed...")
        test_pod.wait_phase(['Succeeded', 'Failed'], timeout=900)
        return test_pod.read().status.phase

    @staticmethod
    def _check_health_status(k8s_resource: K8sNamespacedResource,
                             expected_status):
        return k8s_resource.read().status.get("health") == expected_status

    def _check_modules_health_status(self, k8s_tf_resources, expected_status):
        status = True
        for k8s_resource in k8s_tf_resources:
            if not self._check_health_status(k8s_resource, expected_status):
                status = False
        return status

    def is_vrouter_component_updated(self):
        return self.tfvrouter().read().status.get("updated") == "Yes"

    def _check_modules_updated(self):
        modules = self.tfoperator().read().status.get('modules')
        status = True
        for module in modules.keys():
            updated = modules[module].get('updated')
            if updated != "Yes":
                LOG.info(f"TF module {module} updated: {updated}")
                status = False
        return status

    @staticmethod
    def _is_pod_restarted(pod: K8sPod, init_restarts: int):
        return pod.get_restarts_number() != init_restarts

    def wait_pod_restarted(self, pod: K8sPod, timeout=180, interval=5):
        init_restarts = pod.get_restarts_number()
        waiters.wait(
            self._is_pod_restarted,
            predicate_args=[pod, init_restarts],
            interval=interval,
            timeout=timeout,
            timeout_msg=f"K8s {pod.resource_type} {pod.name} wasn't restarted",
        )
        LOG.info(f"Number of container restarts for {pod.resource_type}"
                 f" {pod.name} was changed from {init_restarts} to"
                 f" {pod.get_restarts_number()}")

    def wait_tfoperator_healthy(self, expected_status="Ready", timeout=900,
                                interval=10):
        LOG.info(f"Wait until TF operator health status becomes {expected_status}")
        waiters.wait(
            self._check_health_status,
            predicate_args=[self.tfoperator(), expected_status],
            interval=interval,
            timeout=timeout,
            timeout_msg=f"TFOperator doesn't reached status "
                        f"`{expected_status}`",
        )

    def wait_for_hold_healthy(self, expected_status="Ready", interval=5, hold_duration=120,  timeout=600):
        LOG.info(f"Wait until TF operator is healthy for {hold_duration} seconds:")

        waiters.wait_for_hold(
            self._check_health_status,
            predicate_args=[self.tfoperator(), expected_status],
            interval=interval,
            timeout=timeout,
            hold_duration=hold_duration,
            timeout_msg=f"TFOperator doesn't reached status "
                        f"`{expected_status}`",
        )
        LOG.info(f"TFOperator health status is {expected_status}.")

    def wait_tf_modules_updated(self, timeout=300, interval=10):
        LOG.info("Wait until TF operator modules to be updated:")
        waiters.wait(
            self._check_modules_updated,
            interval=interval,
            timeout=timeout,
            timeout_msg="TFOperator modules aren't updated",
        )
        LOG.info("TFOperator modules are updated.")

    def wait_tf_controllers_healthy(self, expected_status="Ready", timeout=180,
                                    interval=10):
        # TODO: implement a check of component is enabled and extend components
        #  when LCM API v2 is implemented (PRODX-17429)
        LOG.info("Wait TF controllers are healthy:")
        # Main components, enabled by default:
        resources = [self.tfconfig(), self.tfcontrol(),  self.tfvrouter()]
        if self.is_analytics_enabled():
            resources.append(self.tfanalytics())
            self.wait_tf_analytics(exist=True, timeout=450)
        waiters.wait(
            self._check_modules_health_status,
            predicate_args=[resources, expected_status],
            interval=interval,
            timeout=timeout,
            timeout_msg=f"TF controllers don't reached status: {expected_status}",
        )
        LOG.info(f"TF controllers have status: {expected_status}")

    def _are_pods_on_node_ready(self, node_name):
        pods = self.api.pods.list_raw(
            field_selector=f"spec.nodeName={node_name}")
        inactive = [p.metadata.name for p in pods.items if
                    p.status.phase.lower()
                    not in ('running', 'succeeded')]
        if inactive:
            LOG.info(f"Following pods are not ready: {inactive}")
            return False
        return True

    def wait_all_pods_on_node(self, node_name, timeout=300, interval=5):
        LOG.info(f"Wait all pods on node {node_name} are ready.")
        waiters.wait(
            self._are_pods_on_node_ready,
            predicate_args=[node_name],
            interval=interval,
            timeout=timeout,
            timeout_msg="Some pods aren't in running/succeeded phase",
        )

    def get_gr_job(self, read=False):
        return self.get_k8s_resource(
            "jobs", "gr-settings", read
        )

    def wait_gr_gob(self, timeout=120, interval=3):
        job = self.get_gr_job()
        LOG.info(f"Wait {job} is created.")
        waiters.wait(
            utils.is_k8s_res_exist,
            predicate_args=[job],
            interval=interval,
            timeout=timeout,
        )
        return job

    def get_api_conversion_job(self, read=False):
        return self.get_k8s_resource(
            "jobs", "tungstenfabric-operator-covert-to-v2", read
        )

    def wait_api_conversation_job(self, timeout=120, interval=3):
        job = self.get_api_conversion_job()
        LOG.info(f"Wait {job} is created.")
        waiters.wait(
            utils.is_k8s_res_exist,
            predicate_args=[job],
            interval=interval,
            timeout=timeout,
        )
        return job

    def get_confmap_apiv1_backup(self, read=False):
        return self.get_k8s_resource(
            "configmaps", "tfoperator-v1alpha1-copy", read
        )

    @property
    def pod_tf_cli(self):
        if ((self._pod_tf_cli and not self.api.pods.present(name=self._pod_tf_cli.name, namespace=self.tf_namespace))
                or not self._pod_tf_cli):
            pods = self.api.pods.list(namespace=self.tf_namespace, name_prefix="tf-tool-cli")
            self._pod_tf_cli = pods[0]
        return self._pod_tf_cli

    def exec_tf_cli_pod(self, cmd):
        return self.pod_tf_cli.exec(['/bin/sh', '-c', cmd])

    def exec_pod_cmd(self, node_name, cmd, verbose=True, timeout=600,
                     delete_pod=True):
        """Execute cmd in a privileged pod on the given node

        Doesn't require SSH access to the Node.

        * The {cmd} is executed in the privileged container from
          the chroot to the node's root filesystem /

        * A separated pod is started and removed for each {cmd}, so
          it will spend some time to pull the image and start the pod.

        * There is no way to interact with {cmd} from shell while it
          is executed. All required parameters must be set in {cmd}

        * There is no way to separate stdout from stderr, all the
          info is collected from the pod "logs".

        :rtype dict: {'logs': <str>, 'events': <str>, 'exit_code': <int>}
        """
        pod_name = "exec-pod-cmd-{}".format(utils.gen_random_string(6))
        pod_namespace = 'default'
        render_options = {
            'POD_NAME': pod_name.lower(),
            'POD_CMD': cmd,
            'NODE_NAME': str(node_name).lower(),
            'MCP_DOCKER_REGISTRY': "mirantis.azurecr.io",
        }
        pod_template = yaml.safe_load(template_utils.render_template(
            settings.MACHINE_PRIVELEGED_POD_YAML, options=render_options,
            log_env_vars=False, log_template=False))

        pod = self.api.pods.create(name=pod_name, namespace=pod_namespace,
                                   body=pod_template, log_body=False)
        if verbose:
            _LOG = LOG.info
            _ERRLOG = LOG.error
        else:
            _LOG = LOG.debug
            _ERRLOG = LOG.debug
        try:
            pod.wait_phase(['Succeeded', 'Failed'],
                           timeout=timeout, interval=1)
            pod_status = pod.read().status
            container_state = pod_status.container_statuses[0].state
            if container_state.terminated is not None:
                exit_code = container_state.terminated.exit_code
            else:
                exit_code = -1
                LOG.debug(f"Pod {pod.name} has wrong container status:\n"
                          f"{pod_status}")

            logs = pod.get_logs()
            events = None
            _LOG(f"[{node_name}] Output of the command "
                 f"\"{cmd}\":\n{logs}")
            _LOG(f"Events after command execution:\n{events}")
            _LOG(f"Exit code: {exit_code}")
            return {'logs': logs,
                    'events': events,
                    'exit_code': exit_code}
        except Exception as e:
            _ERRLOG(f"Error while running pod {pod_name}:\n{e}")
            raise (e)
        finally:
            if delete_pod:
                pod.delete()

    def get_rabbitmq_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector='app=rabbitmq')

    def wait_sts_redis_ready(self, timeout=180, interval=3):
        sts = self.api.statefulsets.get(name="tf-rabbitmq", namespace=self.tf_namespace)
        waiters.wait(self._is_sts_ready, predicate_args=[sts],
                     timeout=timeout, interval=interval)

    def get_redis_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector='redis=tf-redis')

    def wait_sts_rabbitmq_ready(self, timeout=180, interval=3):
        sts = self.api.statefulsets.get(name="redis-tf-redis", namespace=self.tf_namespace)
        waiters.wait(self._is_sts_ready, predicate_args=[sts],
                     timeout=timeout, interval=interval)

    def get_zookeeper_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector='app=tf-zookeeper')

    def wait_sts_zookeeper_ready(self, timeout=180, interval=3):
        sts = self.api.statefulsets.get(name="tf-zookeeper", namespace=self.tf_namespace)
        waiters.wait(self._is_sts_ready, predicate_args=[sts],
                     timeout=timeout, interval=interval)

    def get_cassandra_pods(self):
        return self.api.pods.list(namespace=self.tf_namespace,
                                  label_selector=f'cassandracluster={self._cassandra_cluster_tf_config}')

    def get_cassandra_pvcs(self):
        return self.api.pvolumeclaims.list(namespace=self.tf_namespace,
                                           label_selector=f'cassandracluster={self._cassandra_cluster_tf_config}')

    def get_cassandra_svc(self):
        return self.api.services.get(name=self._cassandra_cluster_tf_config, namespace=self.tf_namespace)

    def check_cassandra_cluster(self) -> bool:
        # Check cassandra cluster consistency and node statuses.
        healthy = True
        sts_name = f"{self._cassandra_cluster_tf_config}-dc1-rack1"
        sts = self.api.statefulsets.get(name=sts_name, namespace=self.tf_namespace)
        replicas = int(sts.read().spec.replicas)
        pods = self.api.pods.list_starts_with(sts_name, namespace=self.tf_namespace)
        LOG.info(f"Total {sts_name} pods: {len(pods)}/{replicas}")

        cmd = ['/bin/sh', '-c',
               "nodetool status 2>/dev/null | grep rack | awk '{print $1\";\"$2\";\"$7}'"]
        output = pods[0].exec(cmd, container='cassandra')
        if len(output) > 0:
            LOG.info(f"Status of cassandra nodes:\n{output}")
            cassandra_nodes = output.splitlines()
            if len(cassandra_nodes) != replicas:
                LOG.warning(f"Amount of cassandra nodes ({len(cassandra_nodes)}) in config isn't equal to the number "
                            f"of replicas")
                healthy = False
            for cassandra_node in cassandra_nodes:
                status = cassandra_node.split(";")[0]
                ip = cassandra_node.split(";")[1]
                node_id = cassandra_node.split(";")[2]
                if status != "UN":
                    LOG.warning(f"Unexpected cassandra node status: {status} {ip} {node_id}")
                    healthy = False
        else:
            LOG.error("Cant read cassandra nodes list, command returned empty data")
            return False

        cmd = ['/bin/sh', '-c', "nodetool describecluster 2>/dev/null | sed '1,/Schema versions:/d'"]
        output = pods[0].exec(cmd, container='cassandra')
        if len(output) > 0:
            LOG.info(f"Schema versions of cassandra cluster:\n{output}")
            schemas = [line.strip() for line in output.splitlines() if len(line.strip()) > 0]
            if len(schemas) != 1:
                LOG.warning(f"Cassandra cluster contains more than one schema version ({len(schemas)})")
                healthy = False
            schema = schemas[0]
            schema_list = schema.split(':')
            if "UNREACHABLE" in schema_list[0]:
                LOG.warning("Cassandra cluster contains UNREACHABLE nodes")
                healthy = False
            nodes = schema_list[1].split(',')
            if len(nodes) != replicas:
                LOG.warning(f"Amount of cassandra nodes ({len(nodes)}) in cluster schema isn't equal to the number "
                            f"of replicas")
                healthy = False
        else:
            LOG.error("Cant read cassandra cluster details, command returned empty data")
            return False

        return healthy
