#    Copyright 2023 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

import yaml

from si_tests import logger, settings
from si_tests.managers.bootstrap_manager import BootstrapManager
from si_tests.utils import packaging_version as version

LOG = logger.logger


class PDNSManager:

    def __init__(self, bootstrap_manager: BootstrapManager, verbose: bool = True):
        self.verbose = verbose
        self.bootstrap_manager = bootstrap_manager
        self.remote = bootstrap_manager.remote_seed()
        self.seed_node_ip = self.bootstrap_manager.get_seed_ip()

    def prepare_repository(self, server_version='4.6', recursor_version='4.6'):
        remote_distr_codename = self.remote.check_call("lsb_release -sc").stdout_str
        server_repo_code = ''.join(server_version.split('.')[:2])
        recursor_repo_code = ''.join(recursor_version.split('.')[:2])
        stable_versions_map = {
            'authserver':
                {'repo': f'deb [signed-by=/etc/apt/keyrings/auth-{server_repo_code}-pub.asc] '
                         f'http://repo.powerdns.com/ubuntu {remote_distr_codename}-auth-{server_repo_code} main'},
            'recursor':
                {'repo': f'deb [signed-by=/etc/apt/keyrings/rec-{recursor_repo_code}-pub.asc] '
                         f'http://repo.powerdns.com/ubuntu {remote_distr_codename}-rec-{recursor_repo_code} main'}
        }

        # Use official stable pdns server/recursor versions. These versions are compatible with 20 and 22 ubuntu
        # https://repo.powerdns.com
        prepare_cmds = [
            # Auth server 4.6
            # Recursor 4.6
            (f"cat << EOF > /etc/apt/sources.list.d/pdns.list\n"
             f"{stable_versions_map['authserver']['repo']}\n"
             f"{stable_versions_map['recursor']['repo']}\n"),

            (f"cat << EOF > /etc/apt/preferences.d/auth-{server_repo_code}\n"
             "Package: auth*\n"
             "Pin: origin repo.powerdns.com\n"
             "Pin-Priority: 600\n"
             "EOF\n"),

            (f"cat << EOF > /etc/apt/preferences.d/rec-{recursor_repo_code}\n"
             "Package: rec*\n"
             "Pin: origin repo.powerdns.com\n"
             "Pin-Priority: 600\n"
             "EOF\n"),

            ("install -d /etc/apt/keyrings; curl https://repo.powerdns.com/FD380FBB-pub.asc | "
             f"tee /etc/apt/keyrings/auth-{server_repo_code}-pub.asc"),

            ("install -d /etc/apt/keyrings; curl https://repo.powerdns.com/FD380FBB-pub.asc | "
             f"tee /etc/apt/keyrings/rec-{recursor_repo_code}-pub.asc"),
            "apt update",
        ]
        for cmd_p in prepare_cmds:
            with self.remote.sudo(enforce=True):
                self.remote.check_call(cmd_p, verbose=self.verbose)

    def install_actions(self):
        actions_cmds = [
            # Failing to start pdns.service during apt install is expected
            "apt install -y sqlite3 net-tools pdns-backend-sqlite3 pdns-server pdns-recursor",

            "systemctl stop pdns.service",
            "systemctl stop pdns-recursor.service",
            "rm -rf /etc/powerdns/pdns.d/* || true"]

        for action_cmd in actions_cmds:
            with self.remote.sudo(enforce=True):
                self.remote.check_call(action_cmd, verbose=self.verbose)

    def configure_server(self):
        config_server_cmds = [

            # Configure local pdns-server to use a simple SQLite3 backend
            # and default zone TTL 60 sec, to allow quick apply of the zone changes
            ("[ -f /var/lib/powerdns/pdns.sqlite3 ] || "
             "sqlite3 /var/lib/powerdns/pdns.sqlite3 < /usr/share/doc/pdns-backend-sqlite3/schema.sqlite3.sql"),

            ("cat << EOF > /etc/powerdns/pdns.d/local.conf\n"
             "launch=gsqlite3\n"
             "gsqlite3-database=/var/lib/powerdns/pdns.sqlite3\n"
             "default-ttl=60\n"
             "EOF\n"),

            # Change owner to pdns
            "chown pdns:pdns /var/lib/powerdns/pdns.sqlite3",

            # Listen address/port for the authoritative pdns server
            # which will keep our custom zones
            ("cat << EOF > /etc/powerdns/pdns.d/listen.conf\n"
             "local-address=127.0.0.1\n"
             "local-port=5300\n"
             "EOF\n")
        ]
        for config_server_cmd in config_server_cmds:
            with self.remote.sudo(enforce=True):
                self.remote.check_call(config_server_cmd, verbose=self.verbose)

    def _configure_recursor_old(self, zones, upstream_nameserver):
        if zones:
            forward_zones = [f"{zone}=127.0.0.1:5300" for zone in zones]
            forward_zones_str = "forward-zones=" + ','.join(forward_zones) + "\n"
        else:
            forward_zones_str = ""
        # Which servers the Recursor should ask for the requesting zone.
        # Firstly, ask the local pdns-server on 127.0.0.1:5300,
        # then ask the upstream nameserver
        configure_recursor_cmds = [
            (f"cat << EOF > /etc/powerdns/recursor.d/forwards.conf\n"
             f"{forward_zones_str}"
             f"forward-zones-recurse=.={upstream_nameserver}\n"
             f"trace=on\n"
             f"EOF\n"),

            # Addresses to listen port :53 for DNS requests
            (f"cat << EOF > /etc/powerdns/recursor.d/listen.conf\n"
             f"local-address=127.0.0.1, {self.seed_node_ip}\n"
             f"EOF\n")
        ]
        for configure_recursor_cmd in configure_recursor_cmds:
            with self.remote.sudo(enforce=True):
                self.remote.check_call(configure_recursor_cmd, verbose=self.verbose)

    def _configure_recursor_new(self, zones, upstream_nameserver):
        # Use yaml format for new versions
        recursor_conf = {
            'recursor': {
                'hint_file': '/usr/share/dns/root.hints',
                'forward_zones': [],
                'forward_zones_recurse': [
                    {'zone': '.',
                     'forwarders': [f'{upstream_nameserver}']
                     }
                ]
            },
            'incoming': {
                'listen': ['127.0.0.1', f'{self.seed_node_ip}']
            }
        }
        zones = zones or []
        for zone in zones:
            recursor_conf['recursor']['forward_zones'].append(
                {'zone': f'{zone}',
                 'forwarders': ['127.0.0.1:5300']}
            )
        with open('recursor.conf', 'w') as file:
            yaml.dump(recursor_conf, file, default_flow_style=False)
        LOG.info(f"Uploading to /etc/powerdns/recursor.conf\n{yaml.dump(recursor_conf)}")
        with self.remote.sudo(enforce=True):
            self.remote.upload(source='recursor.conf', target='/tmp/recursor.conf')
            self.remote.check_call('mv /tmp/recursor.conf /etc/powerdns/', verbose=self.verbose)

    def configure_recursor(self, zones, upstream_nameserver):
        recursor_version = self.remote.check_call(
            "pdns_recursor --version | grep 'PowerDNS Recursor' | awk '{print $3}'",
            verbose=self.verbose).stdout_str
        is_new_config = version.parse(recursor_version) >= version.parse('5.0.0')
        if is_new_config:
            LOG.info(f"Detected {recursor_version} recursor version. New config style will be used (YAML)")
            self._configure_recursor_new(zones=zones, upstream_nameserver=upstream_nameserver)
        else:
            LOG.info(f"Detected {recursor_version} recursor version. Using old config style")
            self._configure_recursor_old(zones=zones, upstream_nameserver=upstream_nameserver)

    def start_services(self):
        start_cmds = [
            "systemctl start pdns.service",
            "systemctl start pdns-recursor.service",

            'echo "nameserver 127.0.0.1" | tee /etc/resolv.conf',
            f'echo "nameserver {self.seed_node_ip}" >> /etc/resolv.conf',
            "sed -i '/127.0.1.1.*/d' /etc/hosts",
            'echo "127.0.1.1 $(hostname -s)" | tee -a /etc/hosts',
            "systemctl status pdns.service",
            "systemctl status pdns-recursor.service",
            "netstat -lpn|grep :53",
        ]
        for cmd in start_cmds:
            with self.remote.sudo(enforce=True):
                self.remote.check_call(cmd, verbose=self.verbose)

    def install_pdns(self, local_zones=None, upstream_nameserver=settings.DNS_NAMESERVER,
                     use_official_repositories=settings.SEED_PDNS_USE_OFFICIAL_REPOS,
                     server_version='4.6', recursor_version='4.6'):
        """Install PowerDNS server and recursor on the Seed node

        This turns Seed node into a DNS server which can manage own zones
        or forward requests recursively to upstream DNS

        :param local_zones: list of strings, zone names to forward to the Authoritative server
                            Do not use "local" zone because it is filtered as unsafe
                            in "systemd-resolved" service by default.
                            To check other filtered zones, please run on the node:
                                $ systemd-resolve --status
                                or
                                $ resolvectl nta
                            , and check for "DNSSEC NTA" entries (DNSSEC Negative Trust Anchor)
        :param upstream_nameserver: str, IP of upstream DNS server to forward all other queries
        :param use_official_repositories: bool, whether to use pdns from official repositories or not
        :param server_version: str, version of pdns server. Could be 4.6, 4.7 ... 4.9
        :param recursor_version: str, version of pdns recursor. Could be 4.3, 4.4 ... 5.2
        """

        if use_official_repositories:
            self.prepare_repository(server_version=server_version,
                                    recursor_version=recursor_version)

        LOG.info("Install packages and stop services")
        self.install_actions()
        LOG.info("Configure powerdns Authoritative Server")
        self.configure_server()
        LOG.info("Configure powerdns Recursor")
        self.configure_recursor(zones=local_zones, upstream_nameserver=upstream_nameserver)

        # Disable systemd-resolved
        # Service 'systemd-resolved' occupies port :53 and will conflict with pdns-recursor
        LOG.info("Disabling systemd-resolved because it occupies port :53")
        with self.remote.sudo(enforce=True):
            self.remote.check_call("systemctl disable --now systemd-resolved", verbose=self.verbose)

        LOG.info("Starting powerdns server and recursor")
        self.start_services()

    def restart_service(self):
        with self.remote.sudo(enforce=True):
            self.remote.check_call("systemctl restart pdns.service", verbose=True)

    def create_zone(self, zone_name, recreate=True):
        with self.remote.sudo(enforce=True):
            res = self.remote.execute(f"pdnsutil list-zone {zone_name}")
            if res.exit_code == 0:
                if recreate:
                    LOG.info(f"Delete old PDNS zone {zone_name}")
                    self.remote.check_call(f"pdnsutil delete-zone {zone_name}")
                else:
                    LOG.info(f"PDNS zone {zone_name} already exists, zone recreate is disabled, using existing")

            # Create zone with SOA and NS records
            self.remote.check_call(f"pdnsutil create-zone {zone_name} ns1.{zone_name}")

        # Create A records for the nameserver and zone name
        self.add_record(zone_name, 'ns1', 'A', self.seed_node_ip)
        self.add_record(zone_name, '.', 'A', self.seed_node_ip)

    def add_record(self, zone_name, record_name, record_type, record_content):
        with self.remote.sudo(enforce=True):
            self.remote.check_call(f"pdnsutil add-record {zone_name} {record_name} {record_type} {record_content}")

    def show_zone(self, zone_name):
        with self.remote.sudo(enforce=True):
            self.remote.check_call(f"pdnsutil list-zone {zone_name}", verbose=True)
