import os.path

from retry import retry
from time import sleep
from cached_property import cached_property, cached_property_with_ttl
import cachetools.func as cachetools_func
from urllib.parse import urlparse
import exec_helpers

from heatclient import client as heatclient
from heatclient import exc as heat_exceptions
from heatclient.common import template_utils
from keystoneauth1.identity import v3 as keystone_v3
from keystoneauth1 import session as keystone_session
from neutronclient.v2_0 import client as neutronclient
from novaclient import client as novaclient
from cinderclient import client as cinderclient
from octaviaclient.api.v2 import octavia
from neutronclient.common.exceptions import NotFound
from ironicclient import client as ironicclient

from si_tests import settings
from si_tests import exceptions
from si_tests import logger
from si_tests.utils import waiters, utils

EXPECTED_STACK_STATUS = "CREATE_COMPLETE"
BAD_STACK_STATUSES = ["CREATE_FAILED", "DELETE_FAILED"]
BAD_IRONIC_HW_NODE_STATUSES = ["error", "clean failed"]

LOG = logger.logger


class OpenstackClient:

    def __init__(self, auth):
        super().__init__()
        self.__auth = auth
        self.__token = {}
        self.__heatclient = None
        self.__neutronclient = None
        self.__novaclient = None

    def stack_create(self, template_path):
        stack_name = settings.ENV_NAME
        heat_env_path = settings.HEAT_ENV_PATH
        if settings.SI_HEAT_AUTO_GUESS_CONFIG:
            LOG.info(f'Looking heat configs at {settings.SI_HEAT_AUTO_GUESS_PATH}')
            _expected_template_path = os.path.join(settings.SI_HEAT_AUTO_GUESS_PATH, 'heat.hot')
            _postfix = ''
            LOG.debug(f'Looking for {_expected_template_path}')
            if os.path.exists(_expected_template_path):
                template_path = _expected_template_path
            if settings.MCC_AIRGAP:
                _postfix = '_airgapped'
            _expected_heat_env_path = os.path.join(settings.SI_HEAT_AUTO_GUESS_PATH, f'env_bootstrap{_postfix}.env')
            _expected_heat_env_path_child_bs = os.path.join(settings.SI_HEAT_AUTO_GUESS_PATH,
                                                            f'env_child_bootstrap{_postfix}.env')
            _expected_heat_env_path_child_upgrade = os.path.join(settings.SI_HEAT_AUTO_GUESS_PATH,
                                                                 f'env_child_upgrade{_postfix}.env')
            # main-fallback-default env file
            LOG.debug(f'Looking for {_expected_heat_env_path}')
            if os.path.exists(_expected_heat_env_path):
                LOG.info(f'Selecting {_expected_heat_env_path}')
                heat_env_path = _expected_heat_env_path
            if settings.BM_DEPLOY_CHILD:
                # env file for child deployment, bootstrap case
                LOG.debug(f'Looking for {_expected_heat_env_path_child_bs}')
                if os.path.exists(_expected_heat_env_path_child_bs):
                    LOG.info(f'Selecting {_expected_heat_env_path_child_bs}')
                    heat_env_path = _expected_heat_env_path_child_bs
            if settings.BM_DEPLOY_CHILD and settings.SI_BM_UPGRADE_CHILD:
                # env file for child deployment, upgrade case
                LOG.debug(f'Looking for {_expected_heat_env_path_child_upgrade}')
                if os.path.exists(_expected_heat_env_path_child_upgrade):
                    LOG.info(f'Selecting {_expected_heat_env_path_child_upgrade}')
                    heat_env_path = _expected_heat_env_path_child_upgrade
        LOG.info(f"Creating stack >{stack_name}< with next env file(s):"
                 f"\n{heat_env_path}"
                 f"\nand next template(s):"
                 f"\n{template_path}")
        tpl_files, template = template_utils.get_template_contents(template_path)
        env_files_list = []
        env_files, env = (template_utils.process_multiple_environments_and_files(
            env_paths=[heat_env_path],
            env_list_tracker=env_files_list))

        fields = {
            'stack_name': stack_name,
            'template': template,
            'files': dict(list(tpl_files.items()) + list(env_files.items())),
            'environment': env,
            'parameters': {}
        }
        if settings.CLOUD_KEY_NAME:
            fields['parameters']['key_pair'] = settings.CLOUD_KEY_NAME

        if settings.OS_AZ:
            fields['parameters']['availability_zone'] = settings.OS_AZ

        if settings.PHYS_NETWORK:
            fields['parameters']['phys_network'] = settings.PHYS_NETWORK

        if settings.PHYS_SUBNET:
            fields['parameters']['phys_subnet'] = settings.PHYS_SUBNET

        if settings.PHYS_SEED_FIXED_IP:
            fields['parameters']['phys_ip_address'] = \
                settings.PHYS_SEED_FIXED_IP

        if settings.PHYS_NETMASK:
            fields['parameters']['phys_netmask'] = settings.PHYS_NETMASK

        if settings.PHYS_GATEWAY:
            fields['parameters']['phys_gateway'] = settings.PHYS_GATEWAY

        if settings.DNS_NAMESERVER:
            fields['parameters']['nameservers'] = settings.DNS_NAMESERVER

        if settings.SI_HEAT_NODES_VOLUME_TYPE_NAME:
            fields['parameters']['nodes_volume_type_name'] = settings.SI_HEAT_NODES_VOLUME_TYPE_NAME

        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
            fields['parameters']['kaas_external_proxy_access_str'] = \
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR

        if settings.OS_FIP_RANGES:
            fields['parameters']['os_fip_ranges'] = \
                settings.OS_FIP_RANGES

        if settings.KAAS_OFFLINE_DEPLOYMENT:
            fields['parameters']['kaas_offline_deployment'] = \
                settings.KAAS_OFFLINE_DEPLOYMENT
            if settings.KAAS_CORE_REPO_URL:
                # If KAAS_CORE_REPO_URL is set, then configure additional rules on gate node
                fields['parameters']['kaas_core_repo_url'] = settings.KAAS_CORE_REPO_URL

        if settings.KAAS_DEPLOY_DEBUG_DEV_MODE:
            fields['parameters']['debug_dev_mode'] = settings.KAAS_DEPLOY_DEBUG_DEV_MODE

        if settings.MOSK_FLOATING_NETWORKS:
            fields['parameters']['mosk_floating_networks'] = \
                settings.MOSK_FLOATING_NETWORKS

        if settings.OFFLINE_CUSTOM_NETWORK_FORWARD:
            fields['parameters']['custom_network_forward'] = \
                settings.OFFLINE_CUSTOM_NETWORK_FORWARD

        if settings.METALLB_BGP_ENABLE:
            fields['parameters']['metallb_bgp_enable'] = settings.METALLB_BGP_ENABLE

        if settings.JENKINS_BUILD_URL:
            fields['parameters']['jenkins_build_url'] = settings.JENKINS_BUILD_URL

        if env_files_list:
            fields['environment_files'] = env_files_list

        @retry(heat_exceptions.HTTPBadGateway, delay=15, tries=20)
        def safe_heat_stack_create():
            self.__stacks.create(**fields)

        def safe_create():
            delay = 15
            tries = 3
            for tnum in range(1, tries + 2):
                try:
                    safe_heat_stack_create()
                    self.wait_of_stack_status(EXPECTED_STACK_STATUS,
                                              tries=140,
                                              delay=10)
                    LOG.info("Stack '%s' created", stack_name)
                    LOG.info('Double check, that stack has been created '
                             'w\\o failed resources')
                    fresources = self._verify_resources_status(
                        EXPECTED_STACK_STATUS)
                    if fresources:
                        for r in fresources:
                            LOG.error(
                                'The resource %s has status %s.'
                                'But it should be %s',
                                r.resource_name,
                                r.resource_status,
                                EXPECTED_STACK_STATUS)
                        raise exceptions.EnvironmentBadStatus(
                            stack_name,
                            EXPECTED_STACK_STATUS,
                            self._current_stack.stack_status,
                            fresources)
                    return
                except exceptions.EnvironmentBadStatus as e:
                    if tnum > tries:
                        # we need to return wrong_resources here as string,
                        # for proper parsing in testrail.
                        raise exceptions.EnvironmentBadStatus(
                            stack_name,
                            EXPECTED_STACK_STATUS,
                            e.env_actual_status,
                            self._get_resources_with_wrong_status())
                    LOG.error('Stack create failed.'
                              'Next attempt {}/{}.'
                              'after:{}s'.format(tnum, tries, delay))
                    LOG.error(
                        "Recreating the stack because some resources have "
                        "incorrect status:\n{}\n".format(e.wrong_resources))
                    sleep(delay)
                    self.delete_stack()

        LOG.info(f"Attempt to create heat stack - {stack_name}")
        if settings.BM_HALF_VIRTUAL_NODE and settings.BM_HALF_VIRTUAL_ENV:
            hwname = f"{settings.BM_HALF_VIRTUAL_NODE_PREFIX}" \
                     f"{settings.BM_HALF_VIRTUAL_NODE}"
            LOG.warning(f"Bm half-virtual node detected: {hwname}")
            # wait for ironic baremetal node cleaned
            self.wait_for_bm_node_status(status='available', hwname=hwname)
        safe_create()

    def delete_stack(self):
        stack_name = settings.ENV_NAME
        removed_retry = 10
        # HALF labs need some extra time, to be moved out
        wait_retry = 120 if settings.BM_HALF_VIRTUAL_ENV else 60
        LOG.debug(f"delete_stack: set wait_retry={wait_retry}")

        if list(self.__stacks.list(stack_name=stack_name)):
            LOG.info(f"Delete stack '{stack_name}'")

            @retry(heat_exceptions.HTTPBadGateway, delay=15, tries=20)
            def safe_heat_stack_delete():
                self.__stacks.delete(self._current_stack.id)

            safe_heat_stack_delete()
            self.wait_of_stack_status('DELETE_COMPLETE',
                                      delay=20, tries=wait_retry,
                                      wait_for_delete=True)
            # Status 'REMOVED' doesn't exists, so wait until the stack
            # is actually removed from the stack list with wait_for_delete=True
            self.wait_of_stack_status('REMOVED',
                                      delay=5, tries=removed_retry,
                                      wait_for_delete=True)
        else:
            LOG.warning(f"Can't delete stack {stack_name} due is absent")
        # Dont move it under "if stack exist"
        # there is might be a case, when stack deleted already, but ironic node
        # node cleaned yet.
        if settings.BM_HALF_VIRTUAL_NODE and settings.BM_HALF_VIRTUAL_ENV:
            hwname = f"{settings.BM_HALF_VIRTUAL_NODE_PREFIX}" \
                     f"{settings.BM_HALF_VIRTUAL_NODE}"
            LOG.warning(f"Bm half-virtual node detected:"
                        f"{hwname}")
            # wait for ironic baremetal node cleaned
            self.wait_for_bm_node_status(status='available', hwname=hwname)

    @cached_property_with_ttl(ttl=15 * 60)
    def _keystone_session(self):
        LOG.debug(f"Creating keystone session to "
                  f"{self.__auth['auth']['auth_url']}")
        keystone_auth = keystone_v3.Password(
            auth_url=self.__auth["auth"]["auth_url"],
            username=self.__auth["auth"]["username"],
            password=self.__auth["auth"]["password"],
            project_name=self.__auth["auth"]["project_name"],
            project_id=self.__auth["auth"]["project_id"],
            user_domain_name=self.__auth["auth"]["user_domain_name"],
            project_domain_name="Default")
        return keystone_session.Session(auth=keystone_auth, verify=False)

    @cached_property_with_ttl(ttl=15 * 60)
    def token(self):
        return self._keystone_session.get_token()

    @cached_property_with_ttl(ttl=15 * 60)
    def access(self):
        return self._keystone_session.auth.get_access(self._keystone_session)

    @property
    def public_endpoints(self):
        return self.access.service_catalog.get_urls(interface='public')

    @property
    def service_catalog(self):
        return self.access.service_catalog.catalog

    @cached_property_with_ttl(ttl=30 * 60)
    def heat(self):
        self.__init_heatclient()
        return self.__heatclient

    def __init_heatclient(self):
        endpoint_url = self._keystone_session.get_endpoint(
            service_type='orchestration', endpoint_type='publicURL')
        self.__heatclient = heatclient.Client(
            version=settings.OS_HEAT_VERSION,
            endpoint=endpoint_url,
            token=self.token,
            insecure=True)

    @cached_property_with_ttl(ttl=30 * 60)
    def cinder(self):
        self.__init_cinderclient()
        return self.__cinderclient

    def __init_cinderclient(self):
        self.__cinderclient = cinderclient.Client(
            service_type='volumev3',
            version="3",
            session=self._keystone_session
        )

    @cached_property_with_ttl(ttl=30 * 60)
    def octavia(self):
        self.__init_octaviaclient()
        return self.__octaviaclient

    def __init_octaviaclient(self):
        endpoint = self._keystone_session.get_endpoint(
            service_type='load-balancer', endpoint_type='publicURL')
        self.__octaviaclient = octavia.OctaviaAPI(
            endpoint=endpoint,
            session=self._keystone_session
        )

    @cached_property_with_ttl(ttl=30 * 60)
    def neutron(self):
        self.__init_neutronclient()
        return self.__neutronclient

    def __init_neutronclient(self):
        self.__neutronclient = neutronclient.Client(
            service_type='network',
            session=self._keystone_session
        )

    def get_routers(self, name_prefix=''):
        if name_prefix:
            return [
                rt for rt in self.neutron.list_routers().get(
                    'routers', []) if rt.get('name').startswith(name_prefix)]
        else:
            return [
                rt for rt in self.neutron.list_routers().get(
                    'routers', [])]

    def delete_router(self, router_id):
        # cleanup dependencies
        # Unset GW
        self.neutron.remove_gateway_router(router_id)
        # Cleanup interfaces
        interfaces = [i['id'] for i in self.neutron.list_ports()[
            'ports'] if router_id in i['device_id']
                      and 'HA port' not in i['name']]
        for interface in interfaces:
            try:
                self.neutron.remove_interface_router(
                    router=router_id, body={'port_id': interface})
            except NotFound:
                LOG.info("Interface {} has been already removed from "
                         "router {}".format(interface, router_id))

        self.neutron.delete_router(router_id)

    def get_servers(self, name_prefix=''):
        if name_prefix:
            return [server._info for server in self.server_list()
                    if server._info.get('name').startswith(name_prefix)]
        else:
            return self.server_list()

    def delete_server(self, server_id):
        self.nova.servers.delete(server_id)

    def get_loadbalancers(self, name_prefix=''):
        if name_prefix:
            return [lb for lb in self.octavia.load_balancer_list().get(
                'loadbalancers', []) if lb.get('name').startswith(name_prefix)]
        else:
            return [lb for lb in self.octavia.load_balancer_list().get(
                'loadbalancers', [])]

    def delete_loadbalancer(self, lb_id, cascade=True):
        self.octavia.load_balancer_delete(lb_id, cascade=cascade)

    def get_fips(self, tags_list=None):
        if tags_list:
            return [ip for ip in self.neutron.list_floatingips().get(
                'floatingips', []) if any(
                pref in ip.get('tags') for pref in tags_list)]
        else:
            return [ip for ip in self.neutron.list_floatingips().get(
                'floatingips', [])]

    def delete_fip(self, fip_id):
        self.neutron.delete_floatingip(fip_id)

    def get_networks(self, name_prefix=''):
        if name_prefix:
            return [net for net in self.neutron.list_networks().get(
                'networks', []) if net.get('name').startswith(name_prefix)]
        else:
            return [net for net in self.neutron.list_networks().get(
                'networks', [])]

    def delete_network(self, net_id):
        # Cleanup ports before network deletion
        network_ports = [
            p['id'] for p in self.neutron.list_ports()[
                'ports'] if p.get('network_id') == net_id]
        for port in network_ports:
            try:
                self.neutron.delete_port(port)
            except NotFound:
                LOG.info("Port {} has already been deleted".format(port))
        self.neutron.delete_network(net_id)

    def get_keypairs(self, name_prefix=''):
        if name_prefix:
            return [key._info['keypair'] for key in self.nova.keypairs.list()
                    if key.name.startswith(name_prefix)]
        else:
            return [key._info['keypair'] for key in self.nova.keypairs.list()]

    def delete_keypair(self, keypair_name):
        self.nova.keypairs.delete(keypair_name)

    def get_volumes(self, name_prefix=''):
        if name_prefix:
            return [
                vol._info for vol in self.cinder.volumes.list()
                if vol.name.startswith(name_prefix)]
        else:
            return [
                vol._info for vol in self.cinder.volumes.list()]

    def delete_volume(self, vol_id):
        self.cinder.volumes.delete(vol_id)

    def get_security_groups(self, name_prefix=''):
        if name_prefix:
            return [sg for sg in self.neutron.list_security_groups()[
                'security_groups'] if sg.get('name').startswith(name_prefix)]
        else:
            return [sg for sg in self.neutron.list_security_groups()[
                'security_groups']]

    def delete_security_group(self, sg_id):
        self.neutron.delete_security_group(sg_id)

    def get_stacks(self, name_prefix=''):
        if name_prefix:
            return [st._info for st in self.__stacks.list()
                    if st.stack_name.startswith(name_prefix)]
        else:
            return [st._info for st in self.__stacks.list()]

    def delete_stack_by_id(self, stack_id):
        self.__stacks.delete(stack_id)

    def add_sg_rule(self, id, ethertype="IPv4", **kwargs):
        sg_rule = {
            "security_group_rule": {
                "direction": kwargs['direction'],
                "ethertype": ethertype,
                "protocol": kwargs['proto'],
                "security_group_id": id,
                "remote_ip_prefix": kwargs['cidr'],
            }}
        if 'ports' in kwargs.keys():
            sg_rule['security_group_rule']['port_range_min'] = \
                kwargs['ports'].split("-")[0]
            sg_rule['security_group_rule']['port_range_max'] = \
                kwargs['ports'].split("-")[1]
        if 'port' in kwargs.keys():
            sg_rule['security_group_rule']['port_range_min'] = \
                kwargs['port']
            sg_rule['security_group_rule']['port_range_max'] = \
                kwargs['port']
        self.neutron.create_security_group_rule(sg_rule)

    def add_sg(self, name, description='offline SG'):
        body = {'name': name, 'description': description}
        sg = self.neutron.create_security_group({'security_group': body})
        return sg

    def get_sg_by_name(self, name):
        sg = [x for x in self.neutron.list_security_groups()['security_groups']
              if x['name'] == name]
        if sg:
            # we return only the first sg in list
            return sg[0]
        else:
            raise Exception("Security group with name {0} not "
                            "found".format(name))

    def get_os_cloud_endpoints(self):
        """Returns comma separated string of cloud endpoints"""
        ep = self.public_endpoints
        return ",".join([urlparse(e).netloc.split(':')[0] for e in ep])

    @cached_property_with_ttl(ttl=30 * 60)
    def nova(self):
        self.__init_novaclient()
        return self.__novaclient

    def __init_novaclient(self):
        self.__novaclient = novaclient.Client(
            service_type='compute',
            version='2',
            session=self._keystone_session
        )

    def server_list(self, **kwargs):
        return self.nova.servers.list(**kwargs)

    def get_server_by_id(self, server_id=None):
        return self.nova.servers.get(server_id)

    def get_server_by_name(self, server_name=None, verbose=True):
        """
        :param server_name: Openstack Instance Name
        :return: server object
        """
        servers = self.nova.servers.list(search_opts={'name': server_name})
        getserver = None
        for server in servers:
            if server.name == server_name:
                getserver = self.nova.servers.get(server.id)
                if verbose:
                    LOG.info("Server '%s' found with id: %s!", server.name,
                             server.id)
        if getserver is None:
            LOG.error("Server %s not found!", server_name)
        return getserver

    def get_server_by_fip(self, fip):
        lst = self.server_list()
        server = [x for x in lst if fip == self.__get_fip_only(x)]
        return server[0] if server else None

    @staticmethod
    def __get_fip_only(server):
        for network in server._info['addresses']:
            for address in server._info['addresses'][network]:
                addr_type = address['OS-EXT-IPS:type']
                if addr_type == 'floating':
                    LOG.debug("FIP for {0} is {1}".format(
                        server._info['name'], address['addr']))
                    return address['addr']
                continue

    def run_cmd(self, server, cmd, username, key_file, timeout=100):
        keys = utils.load_keyfile(key_file)
        pkey = utils.get_rsa_key(keys['private'])
        auth = exec_helpers.SSHAuth(username=username, password='', key=pkey)
        ssh = exec_helpers.SSHClient(
            host=self.__get_fip_only(server),
            port=22,
            auth=auth,
        )
        ssh.logger.addHandler(logger.console)
        return ssh.execute(cmd, verbose=True, timeout=timeout)

    def wait_server_vm_state(self, server, state='running',
                             timeout=120, interval=20):
        states = {'running': 1, 'shut down': 4}
        waiters.wait(
            lambda: self.nova.servers.get(server.id)._info[
                        'OS-EXT-STS:power_state'] == states[state],
            timeout=timeout, interval=interval
        )

    def wait_server_status(self):
        # Not implemented yet
        pass

    @property
    def __stacks(self):
        return self.heat.stacks

    @property
    def _current_stack(self):
        try:
            return self.__stacks.get(settings.ENV_NAME)
        except heat_exceptions.HTTPNotFound:
            return None

    def wait_of_stack_status(self, status, delay=30, tries=60,
                             wait_for_delete=False):

        @retry(exceptions.EnvironmentWrongStatus, delay=delay, tries=tries)
        def wait():
            # try:
            #     st = self._current_stack.stack_status
            # except heat_exceptions.HTTPNotFound as ex:
            #     if wait_for_delete is True:
            #         return
            #     raise ex

            stack = self._current_stack
            if stack is None:
                if wait_for_delete is True:
                    LOG.info(f"Stack '{settings.ENV_NAME}' not exist")
                    return
                raise Exception(f"Stack '{settings.ENV_NAME}' not found")
            st = stack.stack_status
            LOG.info(f"Stack '{settings.ENV_NAME}' status: <{st}>")
            if st == status:
                LOG.info(f"Waiting for stack '{settings.ENV_NAME}' <{status}> "
                         "reached")
                return
            elif st in BAD_STACK_STATUSES:
                raise exceptions.EnvironmentBadStatus(
                    settings.ENV_NAME,
                    status,
                    st,
                    self._get_resources_with_wrong_status())
            else:
                raise exceptions.EnvironmentWrongStatus(
                    settings.ENV_NAME,
                    status,
                    st)

        LOG.info(f"Waiting for stack '{settings.ENV_NAME}' status <{status}>")
        wait()

    def _get_resources_with_wrong_status(self):
        res = []
        for item in self.__nested_resources:
            if item.resource_status != EXPECTED_STACK_STATUS:
                res.append({
                    'resource_name': item.resource_name,
                    'resource_status': item.resource_status,
                    'resource_status_reason': item.resource_status_reason,
                    'resource_type': item.resource_type
                })
        wrong_resources = '\n'.join([
            "*** Heat stack resource '{0}' ({1}) has wrong status '{2}': {3}"
            .format(item['resource_name'],
                    item['resource_type'],
                    item['resource_status'],
                    item['resource_status_reason'])
            for item in res
        ])
        return wrong_resources

    @property
    def __nested_resources(self):
        resources = []
        stacks = [s for s in self.__stacks.list(show_nested=True)]
        current_stack_id = self._current_stack.id
        for stack in stacks:
            parent_stack_id = self.__get_stack_parent(
                stack.id, stacks) or stack.id
            if parent_stack_id == current_stack_id:
                # Add resources to list
                LOG.info("Get resources from stack {0}"
                         .format(stack.stack_name))
                resources.extend([
                    res for res in self.__resources.list(stack.id)
                ])
        LOG.info("Found {0} resources".format(len(resources)))
        return resources

    def _verify_resources_status(self, status):
        """Check that all resources have verified `status`

        In case when all resources have expected status return empty list,
            otherwise return a list with resources with incorrect status.
        """
        ret = [r for r in self.__nested_resources if
               r.resource_status != status]
        return ret

    def __get_stack_parent(self, stack_id, stacks):
        """Find the parent ID of the specified stack_id in the 'stacks' list"""
        for stack in stacks:
            if stack_id == stack.id:
                if stack.parent:
                    return self.__get_stack_parent(
                        stack.parent, stacks) or stack.id
                else:
                    return stack.id
        LOG.warning("Stack with ID {} not found!".format(stack_id))
        return None

    @property
    def __resources(self):
        return self.heat.resources

    @cached_property
    def _nodes(self):
        """Get list of nodenames from heat

        Returns list of dicts.
        Example:
        - name: seed-node
          addresses:  # Optional. Maybe an empty dict
          - { "fixed" : "p.p.p.202" }
          - { "floating" : "p.p.x.201" }

        'name': taken from heat template resource's ['name'] parameter
        'roles': a list taken from resource's ['metadata']['roles'] parameter
        """
        # address_pools = self._address_pools
        nodes = []
        for heat_node in self._get_resources_by_type("OS::Nova::Server"):
            addresses = {}
            for network in heat_node.attributes['addresses']:
                fixed = None
                floating = None
                for address in heat_node.attributes['addresses'][network]:
                    addr_type = address['OS-EXT-IPS:type']
                    if addr_type == 'fixed':
                        fixed = address['addr']
                    elif addr_type == 'floating':
                        floating = address['addr']
                    else:
                        LOG.error("Unexpected OS-EXT-IPS:type={0} "
                                  "in node '{1}' for network '{2}'"
                                  .format(addr_type,
                                          heat_node.attributes['name'],
                                          network))
                if fixed is None and floating is None:
                    LOG.error("Unable to determine the correct IP address "
                              "in node '{0}' for network '{1}'"
                              .format(heat_node.attributes['name'], network))
                    continue

                addresses["fixed"] = fixed
                addresses["floating"] = floating
            nodes.append({
                "name": heat_node.attributes["name"],
                "addresses": addresses,
            })
        return nodes

    def _stack_resources_data(self) -> dict:
        """Get list of resources, that could be reused.
        Generally cover: nodes,disks,flavor data.

        Returns list of dicts.
        TODO(alexz): to be updated for logic in ansible
        """
        nodes = []
        subnets = []
        for heat_node in self._get_resources_by_type_with_ttl('OS::Nova::Server'):
            _node = {'hostname': heat_node.attributes['OS-EXT-SRV-ATTR:hostname'],
                     'name': heat_node.attributes['name'],
                     'cpu': heat_node.attributes['flavor']['vcpus'],
                     'ram': heat_node.attributes['flavor']['ram'],
                     'server_id': heat_node.attributes['id'],
                     'logical_resource_id': heat_node.logical_resource_id or '',
                     'addresses': heat_node.attributes['addresses']}
            nodes.append(_node)
        for heat_subnet in self._get_resources_by_type_with_ttl('OS::Neutron::Subnet'):
            _subnet = {'resource_name': heat_subnet.resource_name or '',
                       'logical_resource_id': heat_subnet.logical_resource_id or '',
                       'cidr': heat_subnet.attributes.get('cidr', ''),
                       'gateway_ip': heat_subnet.attributes.get('gateway_ip', ''),
                       }
            subnets.append(_subnet)
        return {'nodes': nodes,
                'subnets': subnets}

    def _get_resources_by_type(self, resource_type):
        res = []
        for item in self.__nested_resources:
            if item.resource_type == resource_type:
                resource = self.__resources.get(
                    item.stack_name,
                    item.resource_name)
                res.append(resource)
        return res

    @cachetools_func.ttl_cache(ttl=5 * 60)
    def _get_resources_by_type_with_ttl(self, resource_type):
        return self._get_resources_by_type(resource_type)

    def get_all_resources(self):
        resources = dict()
        resources['vms'] = [server._info for server in self.server_list()]
        resources['volumes'] = [
            vol._info for vol in self.cinder.volumes.list()]
        resources['lbs'] = self.octavia.load_balancer_list().get(
            'loadbalancers', [])
        resources['subnets'] = self.neutron.list_subnets().get('subnets', [])
        resources['routers'] = self.neutron.list_routers().get('routers', [])
        resources['sgs'] = self.neutron.list_security_groups().get(
            'security_groups', [])
        resources['fips'] = self.neutron.list_floatingips().get(
            'floatingips', [])
        resources['nets'] = self.neutron.list_networks().get('networks', [])
        return resources

    @cached_property_with_ttl(ttl=30 * 60)
    def baremetal(self):
        self.__init_baremetal()
        return self.__baremetalclient

    def __init_baremetal(self):
        self.__baremetalclient = ironicclient.Client(
            session=self._keystone_session,
            version="1",
            insecure=True)

    def wait_for_bm_node_status(self, status, hwname, delay=10, tries=120):

        @retry(exceptions.EnvironmentWrongStatus, delay=delay, tries=tries)
        def wait():
            hwnode_id = [i.uuid for i in self.baremetal.node.list() if
                         i.name == hwname]
            if len(hwnode_id) == 0:
                raise Exception(f"Ironic HW node with name:{hwname} not found")
            elif len(hwnode_id) > 1:
                raise Exception(f"Detected more than one ironic node for"
                                f"HW name {hwname}, IDs: {hwnode_id}")
            st = self.baremetal.node.get(hwnode_id[0]).provision_state
            LOG.info(f"Current Ironic node '{hwname}' status: <{st}>")
            if st == status:
                LOG.info(f"Waiting Ironic node status '{hwname}' <{status}> "
                         "reached")
                return
            elif st in BAD_IRONIC_HW_NODE_STATUSES:
                raise Exception(f"ironic node  {hwname}/{hwnode_id},"
                                f"in incorrect status:{st}")
            else:
                raise exceptions.EnvironmentWrongStatus(
                    hwname,
                    status,
                    st)

        LOG.info(f"Waiting for ironic HW node '{hwname}' status <{status}>")
        wait()

    def get_seed_ip(self, fip='seed_floating_ip'):
        if settings.SEED_STANDALONE_EXTERNAL_IP and fip == 'seed_floating_ip':
            LOG.info("SSH into seed using SEED_STANDALONE_EXTERNAL_IP")
            return settings.SEED_STANDALONE_EXTERNAL_IP

        LOG.debug("Getting seed ip via openstack client")
        stack_name = settings.ENV_NAME
        stack = self._current_stack
        if stack:
            outputs = stack.outputs
            key = next(
                (o for o in outputs if o["output_key"] == fip),
                {})
            if key:
                return key['output_value']
            else:
                LOG.warning(f"Stack: {stack_name} doesnt contain requested"
                            f"key: {fip}")
                return None
        LOG.warning("Stack {0} isn't created".format(stack_name))
        return None

    @utils.log_method_time()
    def create_seed_node(self, save_artifacts=True):
        if self._current_stack and settings.KEEP_ENV_BEFORE:
            LOG.warning(f"Stack {settings.ENV_NAME} already created. reusing the seed node")
        else:
            stack = self._current_stack
            assert not stack
            if settings.KAAS_OFFLINE_DEPLOYMENT:
                if not settings.KAAS_EXTERNAL_PROXY_ACCESS_STR and not settings.MCC_AIRGAP:
                    LOG.error("KAAS_EXTERNAL_PROXY_ACCESS_STR is not set, but "
                              "KAAS_OFFLINE_DEPLOYMENT is. "
                              "It makes no sense to deploy mgmt cluster "
                              "in offline mode, but w/o proxy. Exiting..")
                    raise RuntimeError('KAAS_OFFLINE_DEPLOYMENT is '
                                       'set, but KAAS_EXTERNAL_PROXY_ACCESS_STR '
                                       'is empty. Please fill out '
                                       'KAAS_EXTERNAL_PROXY_ACCESS_STR')
            elif settings.KAAS_EXTERNAL_PROXY_ACCESS_STR and settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
                LOG.info("Using proxy with certificate and not in offline mode")
            else:
                if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
                    LOG.error("KAAS_EXTERNAL_PROXY_ACCESS_STR is set, but "
                              "KAAS_OFFLINE_DEPLOYMENT is disabled. "
                              "It makes no sense to use proxy on online "
                              "management cluster. Exiting..")
                    raise RuntimeError('KAAS_OFFLINE_DEPLOYMENT is disabled, but '
                                       'KAAS_EXTERNAL_PROXY_ACCESS_STR is set. '
                                       'Please enable KAAS_OFFLINE_DEPLOYMENT')
            self.stack_create(settings.HEAT_TEMPLATE_PATH)

        if save_artifacts:
            self.save_seed_node_info()

        return self._current_stack.outputs

    def save_seed_node_info(self):
        seed_ip = self.get_seed_ip()
        LOG.info(f"Seed node ip address is {seed_ip}")
        assert seed_ip
        with open('{0}/seed_node_ip'.format(settings.ARTIFACTS_DIR),
                  mode='w') as f:
            # TODO: use yaml editor to store the seed info
            f.write(str(seed_ip))
        with open('{0}/env_name'.format(settings.ARTIFACTS_DIR),
                  mode='w') as f:
            # TODO: use yaml editor to store the seed info
            f.write(settings.ENV_NAME)

    def get_seed_instance(self):
        resources = self._get_resources_by_type("OS::Nova::Server")
        if not resources:
            return None

        resource = resources[0]
        instance = self.get_server_by_id(resource.attributes['id'])

        return instance
