#    Copyright 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations

from kubernetes.client.rest import ApiException
from urllib3.exceptions import MaxRetryError
from websocket._exceptions import WebSocketBadStatusException

import si_tests.utils.waiters as helpers
from si_tests import logger
from si_tests.clients.k8s.base import K8sNamespacedResource
from si_tests.clients.k8s.base import K8sBaseManager
from si_tests.utils.helpers import skip_rc_retry

LOG = logger.logger


class K8sConfigMap(K8sNamespacedResource):
    resource_type = 'configmap'

    def _read(self, **kwargs):
        return self._manager.api.read_namespaced_config_map(
            self.name, self.namespace, **kwargs)

    def _create(self, body, **kwargs):
        return self._manager.api.create_namespaced_config_map(
            self.namespace, body, **kwargs)

    def _patch(self, body, **kwargs):
        return self._manager.api.patch_namespaced_config_map(
            self.name, self.namespace, body, **kwargs)

    def _replace(self, body, **kwargs):
        return self._manager.api.replace_namespaced_config_map(
            self.name, self.namespace, body, **kwargs)

    def _delete(self, async_del=False, timeout=60, **kwargs):
        uid = self.uid
        LOG.info(f"Deleting configmap {self.namespace}/{self.name} with UID: {uid}")
        self._manager.api.delete_namespaced_config_map(
            self.name, self.namespace, **kwargs)
        if not async_del:
            helpers.wait(lambda: not self.exists(),
                         timeout=timeout,
                         timeout_msg=f"Timeout deleting cm {self.namespace}/{self.name}")

    @skip_rc_retry((ApiException, MaxRetryError, WebSocketBadStatusException), delay=20, tries=8)
    def exists(self):
        cms = self._manager.list_raw(self.namespace).to_dict()['items']
        cm = [x for x in cms if x['metadata']['uid'] == self.uid]
        return True if cm else False


class K8sConfigMapManager(K8sBaseManager):
    resource_class = K8sConfigMap

    @property
    def api(self):
        return self._cluster.api_core

    def _list(self, namespace, **kwargs):
        return self.api.list_namespaced_config_map(namespace, **kwargs)

    def _list_all(self, **kwargs):
        return self.api.list_config_map_for_all_namespaces(**kwargs)

    def present(self, name, namespace):
        return any([cm.name for cm in self.list(namespace=namespace)if cm.name == name])
