#    Copyright 2013 - 2017 Mirantis, Inc.
#
#    Licensed under the Apache License, Version 2.0 (the "License"); you may
#    not use this file except in compliance with the License. You may obtain
#    a copy of the License at
#
#         http://www.apache.org/licenses/LICENSE-2.0
#
#    Unless required by applicable law or agreed to in writing, software
#    distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
#    WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
#    License for the specific language governing permissions and limitations
#    under the License.

from __future__ import print_function

import argparse
import sys
import os
import yaml

from si_runscenario import templates
from si_runscenario import logger


LOG = logger.logger


def option_render_scenario(opts, args):
    scenario_file_path = os.path.abspath(opts.scenario_file_path)
    steps_directory = os.path.abspath(opts.steps_directory)

    os.path.join
    if not os.path.isfile(scenario_file_path):
        LOG.error(f"--scenario-file-path argument not set or contains wrong path: '{scenario_file_path}'")
        return 11

    if not os.path.isdir(steps_directory):
        LOG.error(f"--steps-directory argument not set or contains wrong path: '{steps_directory}'")
        return 12

    render_options = {}
    scenario_steps_yaml = templates.render_template(scenario_file_path, options=render_options)
    scenario_steps = yaml.load(scenario_steps_yaml, Loader=yaml.Loader)

    rendered_scenario_yaml = ""
    for section in ['workflow', 'finally']:
        if section not in scenario_steps:
            continue
        rendered_scenario_yaml += f"\n{section}:\n"

        for step in scenario_steps[section]:
            if type(step) is str:
                step_name = step
                render_options = {}
            elif type(step) is dict:
                _step_names = list(step.keys())
                assert len(_step_names) == 1, f"Expect 1 step name per scenario step, got {_step_names}"

                step_name = _step_names[0]
                render_options = step[step_name]
                assert type(render_options) is dict, (
                    f"Expect step options for {step_name} as a dict object, got {type(render_options)}")

            step_filename = f"{step_name}.yaml"
            step_filepath = os.path.join(steps_directory, step_filename)
            assert os.path.isfile(step_filepath), f"Missing scenario chunk file {step_filepath} for step {step_name}"

            step_yaml = templates.render_template(step_filepath, options=render_options)
            rendered_scenario_yaml += f"{step_yaml}\n"
    # Scenario output to 'stdout'
    print(rendered_scenario_yaml)


def load_params(args, defaults=None):
    """
    Parse CLI arguments and environment variables

    Returns: ArgumentParser instance
    """
    defaults = defaults or {}

    render_parser = argparse.ArgumentParser(add_help=False)
    render_parser.add_argument('--scenario-file-path',
                               help='Path to the scenario YAML file with steps file names',
                               default=defaults.get('SCENARIO_FILE_PATH', ''),
                               type=str)
    render_parser.add_argument('--steps-directory',
                               help='Directory with steps YAML files for the scenario',
                               default=defaults.get('STEPS_DIRECTORY', ''),
                               type=str)

    parser = argparse.ArgumentParser(
        description="Execute a shell command locally or on a remote host.\n"
                    "For additional help, use with -h/--help option")
    subparsers = parser.add_subparsers(title="Options",
                                       help='Available options',
                                       dest='option')

    subparsers.add_parser('render-scenario',
                          parents=[render_parser],
                          help="Render a si-config YAML to stdout from defaults and provided arguments")

    if len(args) == 0:
        args = ['-h']
    return parser.parse_args(args)


OPTIONS = {
    'render-scenario': option_render_scenario,
}


def main():
    args = sys.argv[1:]

    opts = load_params(args)
    option = opts.option

    if option not in OPTIONS.keys():
        print(f"Unsupported option {option}, please use one of: "
              f"{OPTIONS.keys()}")
        return 12

    method = OPTIONS[option]
    exit_code = method(opts, args)

    return exit_code


if __name__ == '__main__':
    sys.exit(main())
