import http
from kubernetes.client.rest import ApiException
from retry import retry
from si_tests import logger
import time

LOG = logger.logger


def get_chart_version(chart_spec):
    if 'version' in chart_spec:
        return chart_spec['version']
    f = chart_spec['chartURL'].split('/')[-1]
    name = chart_spec['name']
    return f[f.find(name)+len(name)+1:f.rfind('.tgz')]


def skip_rc_retry(exceptions=ApiException, delay=20, tries=8, rcs=None):
    if not rcs:
        rcs = [http.HTTPStatus.NOT_FOUND]

    def skip(func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except exceptions as e:
                if type(e) is ApiException and e.status in rcs:
                    LOG.debug("Return code %s. Won't retry", e.status)
                    raise e
                else:
                    LOG.warning(f"{e}, retrying in {delay} seconds...")
                    time.sleep(delay)
                    return do_retry(*args, **kwargs)

        @retry(exceptions, delay=delay, tries=tries, logger=LOG)
        def do_retry(*args, **kwargs):
            return func(*args, **kwargs)
        return wrapper
    return skip


def relogin_401(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except ApiException as e:
            if e.status == 401 and e.reason == 'Unauthorized':
                LOG.debug(f"Kubernetes 'Unauthorized' error:\n{e}")
                LOG.info("Kubernetes 'Unauthorized' error, "
                         "trying to refresh authorization")
                if len(args) == 0:
                    LOG.warning("Skipping refresh authorization: "
                                "not a class instance was decorated")
                elif hasattr(args[0], '_manager'):
                    # Assuming instance of 'K8sBaseResource'
                    cluster = args[0]._manager._cluster
                    cluster.login()
                    cluster.init_apis()
                    return func(*args, **kwargs)
                elif hasattr(args[0], '_cluster'):
                    # Assuming instance of 'K8sBaseManager'
                    cluster = args[0]._cluster
                    cluster.login()
                    cluster.init_apis()
                    return func(*args, **kwargs)
                else:
                    LOG.warning("Skipping refresh authorization: "
                                "no suitable class instance provided")
            raise e
    return wrapper


def retry_for_rcs(exceptions=ApiException, delay=10, tries=4, rcs=None):
    if not rcs:
        rcs = [http.HTTPStatus.NOT_FOUND]

    def retry_rc(func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except exceptions as e:
                if e.status in rcs:
                    LOG.warning(f"Exception {e}, retrying in {delay} seconds...")
                    time.sleep(delay)
                    return do_retry(*args, **kwargs)
                else:
                    LOG.debug(f"Return code {e.status}. Won't retry")
                    raise e

        @retry(exceptions, delay=delay, tries=tries, logger=LOG)
        def do_retry(*args, **kwargs):
            return func(*args, **kwargs)
        return wrapper
    return retry_rc
