from si_tests import settings
from si_tests import logger

LOG = logger.logger


def test_servicetemplate(kcm_manager, show_step):
    """Create and apply a ServiceTemplate to the specified cluster and check the conditions
    All objects (ServiceTemplate, HelmRepository, ConfigMap) are created in the
    same namespace as for the ClusterDeployment.

    Scenario:
        1. Access target cluster and check that cluster is Ready and accessible
        2. Create the specified service template
        3. Check that desired template is valid and available in namespace
        4. Prepare service name/namespace/template parameters
        5. Get service values from a text file and create ConfigMap, if path to values is specified
        6. Patch cluster deployment to apply service templates
        7. Wait for the service's Sveltos Helm release readiness
        8. Wait that Helm status is 'Provisioned'
        9. Wait for cluster deployment became ready
        10. Check expected pods
    """
    show_step(1)
    ns = kcm_manager.get_namespace(settings.TARGET_NAMESPACE)
    cld = ns.get_cluster_deployment(settings.TARGET_CLD)
    cld.check.check_actual_expected_pods()
    # Limit readiness check since we should have already deployed working cluster
    cld.check.check_cluster_readiness(timeout=600)
    # assert clusterdeployment.k8sclient.componentstatuses.describe_fixed_resources
    LOG.info(f"Clusterdeployment {cld.namespace}/{cld.name} checked and ready.")

    show_step(2)
    # To create a new ServiceTemplate, please use the CUSTOM_SERVICETEMPLATE_PATH to the object yaml
    # To use an existing ServiceTemplate, please use the KCM_SERVICE_TEMPLATE_NAME with name of the template
    if settings.CUSTOM_SERVICETEMPLATE_PATH:
        servicetemplate = ns.create_service_template_from_file(settings.CUSTOM_SERVICETEMPLATE_PATH)
        servicetemplate_name = str(servicetemplate.name)
    elif settings.KCM_SERVICE_TEMPLATE_NAME:
        servicetemplate_name = settings.KCM_SERVICE_TEMPLATE_NAME
    else:
        raise Exception("Neither CUSTOM_SERVICETEMPLATE_PATH nor KCM_SERVICE_TEMPLATE_NAME is set")

    show_step(3)
    template = ns.get_service_template_by_name(servicetemplate_name)
    assert template.is_valid, f"Template {template.namespace}/{template.name} is not valid"

    show_step(4)
    template_data = template.data
    service_name = settings.CUSTOM_CLUSTERSERVICE_NAME or template_data['spec']['helm']['chartSpec']['chart']
    service_namespace = settings.CUSTOM_CLUSTERSERVICE_NAMESPACE or service_name
    svc = {
        'name': service_name,
        'namespace': service_namespace,
        'template': servicetemplate_name
    }

    show_step(5)
    if settings.CUSTOM_SERVICETEMPLATE_VALUES_PATH:
        configmap = ns.create_configmap_from_values(name=f"service-{settings.TARGET_CLD}-{servicetemplate_name}",
                                                    file_path=settings.CUSTOM_SERVICETEMPLATE_VALUES_PATH)
        svc['valuesFrom'] = [
            {
                'kind': 'ConfigMap',
                'name': configmap.name
            }
        ]

    show_step(6)
    LOG.info(f"Applying service templates to cluster deployment {cld.namespace}/{cld.name}")
    cld.apply_service_templates(svc_list=[svc])

    show_step(7)
    # wait for service condition type {service_namespace}.{service_name} status "True":
    #      - lastTransitionTime: "..."
    #        message: Release my-space/my-name
    #        reason: Managing
    #        status: "True"
    #        type: my-space.my-name/SveltosHelmReleaseReady    # noqa
    cld.check.wait_service_condition(service_type=f'{service_namespace}.{service_name}/SveltosHelmReleaseReady',
                                     expected_status='Managing')

    show_step(8)
    cld.check.wait_service_condition(service_type='Helm', expected_status='Provisioned')

    show_step(9)
    cld.check.check_cluster_readiness(timeout=1800)

    show_step(10)
    cld.check.check_actual_expected_pods()
