import pytest

import json
from si_tests import settings
from si_tests import logger
from si_tests.utils import utils, k8s_utils
from si_tests.managers.kcm_manager import Manager

LOG = logger.logger


@pytest.mark.os
@pytest.mark.usefixtures('log_step_time')
@pytest.mark.usefixtures('log_method_time')
@pytest.mark.usefixtures('store_cluster_kubeconfig')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.KCM_CLUSTER_DEPLOYMENT_NAME}"])
def test_create_os_hosted_standalone_cluster(kcm_manager, show_step, _):
    """Deploy hosted clusterdeployment on OpenStack

    Scenario:
        1. Collect env data
        2. Create or get NS and check cluster templates presence
        3. Create auth
        4. Prepare clusterdeployment config
        5. Create clusterdeployment
        6. Wait for clusterdeployment progress to finish
        7. Check expected pods, its statuses and k8s nodes
        8. Set default storageclass if not present
        9. Sync CoreDNS Corefile with management cluster
    """
    show_step(1)
    # TODO(va4st): Move to separated test with auth actions, determine auth parameters dynamically
    cluster_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    namespace_name = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE
    host_cluster_name = settings.KSI_HOST_CLUSTER_NAME
    host_cluster_ns = settings.KSI_HOST_CLUSTER_NAMESPACE
    assert host_cluster_name, 'KSI_HOST_CLUSTER_NAME is required for this test'
    auth_suffix = f"ksi-openstack-auth-{utils.gen_random_string(4)}"
    # https://github.com/kubernetes/cloud-provider-openstack/blob/master/charts/
    # cinder-csi-plugin/templates/storageclass.yaml#L5
    # https://github.com/k0rdent/kcm/blob/main/templates/cluster/openstack-standalone-cp/
    # templates/k0scontrolplane.yaml#L149
    storageclass = 'csi-cinder-sc-delete'
    auth_secret_name = f"{auth_suffix}-secret"
    cred_name = f"{auth_suffix}-cred"
    # Gathering network config of the host cluster for hosted deployment using mgmt cluster
    LOG.info(f"Gathering network config of the host cluster {host_cluster_ns}/{host_cluster_name} ")
    host_ns = kcm_manager.get_or_create_namespace(host_cluster_ns)
    host_cluster = host_ns.get_cluster_deployment(name=host_cluster_name)
    infracluster = host_cluster.clusterobject.infracluster
    network_name = infracluster.data['status']['network']['name']
    # The first (0) subnet will be used. https://docs.k0rdent.io/latest/admin/hosted-control-plane/hcp-openstack/
    subnet_name = infracluster.data['status']['network']['subnets'][0]['name']
    router_name = infracluster.data['status']['router']['name']
    # Initialise host cluster as manager
    LOG.info(f"Initialise cluster {host_cluster_ns}/{host_cluster_name} as kcm manager instance")
    _, auth_data = host_cluster.get_kubeconfig_from_secret()
    host_kcm_manager = Manager(kubeconfig=None, auth_data=auth_data)

    show_step(2)
    ns = host_kcm_manager.get_or_create_namespace(namespace_name)
    clustertemplate = (settings.KCM_CLUSTER_TEMPLATE_NAME or
                       ns.get_latest_template_by_prefix('openstack-hosted', strategy='chart'))
    available_templates = [tmpl.name for tmpl in ns.get_cluster_templates()]
    assert clustertemplate in available_templates, (f"Clustertemplate {clustertemplate} not found in namespace "
                                                    f"{ns.name}. Available templates: {available_templates}")
    utils.check_expected_pods_template_exists(clustertemplate)

    show_step(3)
    auth_sec = ns.create_os_auth_secret(name=auth_secret_name)
    # NOTE(va4st): For OpenStack there no separated auth. Only secret required
    cred = ns.create_credential(name=cred_name, cred_object=auth_sec)
    ns.create_sveltos_resource_template_cm_using_secret(auth_sec, provider='openstack')

    show_step(4)
    # TODO(va4st): make configurable through settings and add parameters verification before testrun
    config = ns.get_cluster_config(provider='openstack-hosted',
                                   os_auth_secret=auth_sec,
                                   network_name=network_name,
                                   router_name=router_name,
                                   subnet_name=subnet_name)

    child_labels = None

    if host_kcm_manager.kof.kof_ms_deployed():
        child_labels = {
            'k0rdent.mirantis.com/kof-cluster-role': 'child'
        }
        if host_kcm_manager.kof.kof_is_deployed():
            # update specific labels for Istio regional
            labels = {
                'k0rdent.mirantis.com/istio-role': 'child'
            }
        else:
            labels = {
                'k0rdent.mirantis.com/kof-storage-secrets': 'true'
            }
        utils.merge_dicts(child_labels, labels)

        if not host_kcm_manager.kof.kof_is_deployed() and host_kcm_manager.source == 'custom-enterprise':
            annotations = {
                'k0rdent.mirantis.com/kof-collectors-values': json.dumps({
                    "kof": {
                        "logs": {
                            "tls_options": {
                                "insecure_skip_verify": True
                            }
                        },
                        "metrics": {
                            "tls_options": {
                                "insecure_skip_verify": True
                            }
                        },
                        "traces": {
                            "tls_options": {
                                "insecure_skip_verify": True
                            }
                        }
                    }
                }, indent=2)
            }
            config['clusterAnnotations'] = annotations

    show_step(5)
    cld = ns.create_cluster_deployment(name=cluster_name,
                                       template=clustertemplate,
                                       credential=cred,
                                       config=config,
                                       labels=child_labels)
    LOG.info(f"Created clusterdeployment {cld.namespace}/{cld.name}")

    show_step(6)
    cld.check.check_cluster_readiness()

    show_step(7)
    cld.check.check_actual_expected_pods()
    cld.check.check_k8s_pods()
    cld.check.check_k8s_nodes()

    show_step(8)
    if not cld.is_default_storageclass_present():
        sc = cld.get_storageclass_by_name(storageclass)
        sc.make_default()
        assert sc.is_default, f"StorageClass {storageclass} is not default"

    show_step(9)
    # TODO(va4st): Need a better way to determine the needs for CoreDNS sync (e.g some marker in mgmt config maybe?)
    if (not host_kcm_manager.kof.kof_is_deployed() and host_kcm_manager.kof.kof_ms_deployed()
            and host_kcm_manager.source == 'custom-enterprise'):
        corefile = k8s_utils.get_corefile(host_kcm_manager.api)
        k8s_utils.override_coredns_configmap(cld.k8sclient, corefile)
        coredns_deployment = cld.k8sclient.deployments.get(name='coredns', namespace='kube-system')
        coredns_deployment.rollout_restart()
