import pytest

import kubernetes
import os
import base64
import exec_helpers

from si_tests import settings
from si_tests.managers.kcm_manager import Manager
from si_tests.managers.bootstrap_manager import KCMBootstrapManager
from si_tests import logger
from si_tests.utils import utils

LOG = logger.logger


class ComponentTestResources(object):
    """Manage resources for component tests"""

    def __init__(self, kaas_manager, pod_name, namespace,
                 target_namespace, target_cluster):

        self.kaas_manager = kaas_manager
        self.pod_name = pod_name
        self.ns = namespace
        self.target_namespace = target_namespace
        self.target_cluster = target_cluster
        self.conformance_version = settings.K8S_CONFORMANCE_IMAGE_VERSION
        self.secret_name = '-'.join(('kubeconfig', target_cluster))

    @property
    def cluster(self):
        ns = self.kaas_manager.get_namespace(
            namespace=self.target_namespace
        )
        return ns.get_cluster(self.target_cluster)

    @property
    def parent_api(self):
        """Returns k8sclient of parent cluster where all component
        test resources (including pods) will be created
        """
        # returns target cluster's api instead of parent's in case of True
        if self.run_on_target_cluster:
            return self.cluster.k8sclient
        if not self.cluster.is_management:
            return self.cluster.get_parent_cluster().k8sclient
        else:
            return self.kaas_manager.api

    def setUp(self):
        # create Namespace
        if not self.parent_api.namespaces.present(name=self.ns):
            LOG.info("Creating Namespace '{0}'".format(self.ns))
            ns_body = kubernetes.client.V1Namespace(api_version='v1',
                                                    kind='Namespace',
                                                    metadata={'name': self.ns})
            self.parent_api.api_core.create_namespace(body=ns_body)
        else:
            LOG.info("Namespace '{0}' already exists".format(self.ns))

        # create ServiceAccount
        if not self.parent_api.serviceaccounts.present(name=self.pod_name,
                                                       namespace=self.ns):
            LOG.info("Creating ServiceAccount '{0}'".format(self.pod_name))
            sa_body = kubernetes.client.V1ServiceAccount(
                api_version='v1',
                kind='ServiceAccount',
                metadata={'name': self.pod_name}
            )
            self.SA = self.parent_api.serviceaccounts.create(
                namespace=self.ns,
                body=sa_body
            )
            LOG.debug(self.SA)
        else:
            LOG.info("ServiceAccount '{0}' already exists"
                     .format(self.pod_name))
            self.SA = self.parent_api.serviceaccounts.get(
                name=self.pod_name, namespace=self.ns)

        # create ClusterRole
        api_rbac = self.parent_api.api_rbac_auth
        cr_present = any([
            cr for cr in api_rbac.list_cluster_role().items
            if cr.metadata.name == self.pod_name])
        if not cr_present:
            LOG.info("Creating ClusterRole '{0}'".format(self.pod_name))
            pr = kubernetes.client.V1PolicyRule(api_groups=['*'],
                                                resources=['*'],
                                                verbs=['*'])
            # skipping api_version='rbac.authorization.k8s.io / v1beta1'
            cr_body = kubernetes.client.V1ClusterRole(
                kind='ClusterRole',
                metadata={'name': self.pod_name, 'namespace': self.ns},
                rules=[pr]
            )
            CR = api_rbac.create_cluster_role(body=cr_body)
            LOG.debug(CR)
        else:
            LOG.info("ClusterRole '{0}' already exists".format(self.pod_name))

        # create ClusterRoleBinding
        crb_present = any([
            crb for crb in api_rbac.list_cluster_role_binding().items
            if crb.metadata.name == self.pod_name])

        if not crb_present:
            LOG.info("Creating ClusterRoleBinding '{0}'"
                     .format(self.pod_name))
            rr = kubernetes.client.V1RoleRef(
                kind="ClusterRole",
                api_group="rbac.authorization.k8s.io",
                name=self.pod_name)
            sb = kubernetes.client.V1Subject(name=self.pod_name,
                                             kind="ServiceAccount",
                                             namespace=self.ns)
            # skipping api_version='rbac.authorization.k8s.io / v1beta1'
            crb_template = kubernetes.client.V1ClusterRoleBinding(
                metadata={'name': self.pod_name},
                kind='ClusterRoleBinding',
                role_ref=rr,
                subjects=[sb]
            )
            CRB = api_rbac.create_cluster_role_binding(
                body=crb_template
            )
            LOG.debug(CRB)
        else:
            LOG.info("ClusterRoleBinding '{0}' already exists"
                     .format(self.pod_name))

        if self.parent_api.secrets.present(name=self.secret_name,
                                           namespace=self.ns):
            LOG.warning("Secret '{0}' already exists. Deleting.."
                        .format(self.secret_name))
            self.parent_api.api_core.delete_namespaced_secret(
                self.secret_name, self.ns
            )

        # create kubeconfig secret
        LOG.info("Creating Secret '{0}'".format(self.secret_name))
        target_kubeconfig = ''
        LOG.info("TARGET_NAMESPACE is {0}, TARGET_CLUSTER is {1}. "
                 "Using this cluster kubeconfig and running tests "
                 "against it.".format(self.target_namespace,
                                      self.target_cluster))

        name, target_kubeconfig = self.cluster.get_kubeconfig_from_secret()
        self.secret = create_secret(self.parent_api, target_kubeconfig,
                                    self.ns, self.secret_name)
        LOG.debug(self.secret)

    def _get_conformance_version(self, k8s_version_obj):
        try:
            status = self.cluster.data['status']
            if status:
                lcmType = status.get('providerStatus', {}).get(
                    'releaseRefs', {}).get('current', {}).get('lcmType', '')
                if lcmType and lcmType == 'ucp':
                    LOG.info("Cluster under testing has lcmType=ucp")
                    k8s_version = \
                        k8s_version_obj.git_version.split("-docker")[0]
                    k8s_version = k8s_version.split(".")[0] + \
                        "." + k8s_version.split(".")[1]

                    if settings.K8S_CONFORMANCE_IMAGE_VERSION:
                        LOG.info(f"Conformance version manually set to {settings.K8S_CONFORMANCE_IMAGE_VERSION}.")
                        conformance_version = settings.K8S_CONFORMANCE_IMAGE_VERSION
                    else:
                        LOG.info('Trying to detect latest suitable conformance version.')
                        conformance_version = utils.get_latest_k8s_image_version(
                            settings.K8S_CONFORMANCE_IMAGE_VERSION_CHECK_PATH,
                            k8s_version
                        )
        except Exception as e:
            LOG.error(e)
            LOG.error("Cannot dynamically identify conformance image version. ")
            raise e

        return conformance_version

    def get_conformance_version(self):
        target_kubectl_client = self.cluster.k8sclient

        k8s_version_obj = target_kubectl_client.api_version.get_code()
        self.conformance_version = self._get_conformance_version(k8s_version_obj)
        LOG.info(f"K8s conformance image version for this ucp cluster is: {self.conformance_version}")
        LOG.warning("For ucp child we need to set special environment variables")
        LOG.info("*Set K8S_CONFORMANCE_RUN_STORAGE_TESTS to False")
        os.environ["K8S_CONFORMANCE_RUN_STORAGE_TESTS"] = "False"
        LOG.info("*Set K8S_CONFORMANCE_NON_BLOCKING_TAINTS to "
                 "com.docker.ucp.manager,"
                 "com.docker.ucp.orchestrator.kubernetes")
        os.environ["K8S_CONFORMANCE_NON_BLOCKING_TAINTS"] = \
            "com.docker.ucp.manager,"\
            "com.docker.ucp.orchestrator.kubernetes"

        exec_helpers.Subprocess().check_call(
            "echo '{0}' > {1}/conformance_image_version".format(
                self.conformance_version, settings.ARTIFACTS_DIR)
        )

        k8s_version = "{0}.{1}".format(
            k8s_version_obj.major,
            k8s_version_obj.minor).replace("+", ".x")

        LOG.info("Target cluster K8s version: {}".format(k8s_version))

        exec_helpers.Subprocess().check_call(
            "echo '{0}' > {1}/k8s_version".format(
                k8s_version, settings.ARTIFACTS_DIR)
        )

    def tearDown(self):
        LOG.info("teardown prerequisites")
        self.SA.delete()
        self.parent_api.api_rbac_auth.delete_cluster_role(
            name=self.pod_name, body=kubernetes.client.V1DeleteOptions()
        )
        self.parent_api.api_rbac_auth.delete_cluster_role_binding(
            name=self.pod_name, body=kubernetes.client.V1DeleteOptions()
        )
        self.secret.delete()
        self.parent_api.api_core.delete_namespace(
            name=self.ns, body=kubernetes.client.V1DeleteOptions()
        )


def create_secret(kubectl_client, content, namespace, name):
    config = base64.b64encode(content.encode('utf-8'))
    secret_body = kubernetes.client.V1Secret(
        api_version='v1',
        kind='Secret',
        metadata={'name': name, 'namespace': namespace},
        data={name: config.decode('utf-8')}
    )
    secret = kubectl_client.secrets.create(namespace=namespace,
                                           body=secret_body)
    LOG.debug(secret)
    return secret


@pytest.fixture(scope='session')
def kcm_manager():
    kcm_manager = Manager(kubeconfig=settings.KUBECONFIG_PATH)
    if not kcm_manager.api.k0rdent_cluster_deployments.available:
        LOG.error(f"Kubeconfig {settings.KUBECONFIG_PATH} is not a KCM cluster config or KCM is not yet installed")
        raise Exception("Incorrect kubeconfig settings")
    return kcm_manager


@pytest.fixture(scope='session')
def kcm_bootstrap_manager():
    kcm_bootstrap_manager = KCMBootstrapManager(kubeconfig=settings.KUBECONFIG_PATH)
    try:
        kcm_bootstrap_manager.api.pods.list()
        LOG.info('Provided cluster available')
    except Exception as e:
        LOG.error(f"Provided kubeconfig ({settings.KUBECONFIG_PATH}) navigates to unavailable cluster")
        raise e
    return kcm_bootstrap_manager


@pytest.fixture(scope='function')
def store_cluster_kubeconfig(kcm_manager):

    yield

    mgr = kcm_manager

    if settings.KSI_HOST_CLUSTER_NAME and settings.KSI_HOST_CLUSTER_NAMESPACE:
        LOG.info('Hosted cluster deployment detected.')
        host_ns = kcm_manager.get_or_create_namespace(settings.KSI_HOST_CLUSTER_NAMESPACE)
        host_cluster = host_ns.get_cluster_deployment(name=settings.KSI_HOST_CLUSTER_NAME)
        _, auth_data = host_cluster.get_kubeconfig_from_secret()
        mgr = Manager(kubeconfig=None, auth_data=auth_data)

    cl_name = settings.KCM_CLUSTER_DEPLOYMENT_NAME
    cl_ns = settings.KCM_CLUSTER_DEPLOYMENT_NAMESPACE
    mgrclient = mgr.api
    if (not mgrclient.k0rdent_cluster_deployments.present(cl_name, cl_ns) or
            not mgrclient.secrets.present(f"{cl_name}-kubeconfig", cl_ns)):
        LOG.info('Not able to store kubeconfig. Cluster does not exist or kubeconfig secret not present')
        return

    ns = mgr.get_namespace(cl_ns)
    cluster = ns.get_cluster_deployment(cl_name)

    kubeconfig_name, kubeconfig = cluster.get_kubeconfig_from_secret()
    kubeconfig_path = os.path.join(
        settings.ARTIFACTS_DIR, kubeconfig_name)

    LOG.info(f"Saving cluster kubeconfig to {kubeconfig_path}")
    with open(kubeconfig_path, 'w') as f:
        f.write(kubeconfig)


@pytest.fixture(scope='module',
                params=settings.TARGET_CLDS)
def target_clusters_logs(request):
    yield request.param


@pytest.fixture(scope='function')
def component_test_resources():
    """Returns the class ComponentTestResources"""
    return ComponentTestResources


# TODO(va4st): Will be restored later
# @pytest.fixture(scope="module")
# def k8s_prerequisites(kcm_manager):
#     resources = ComponentTestResources(
#         kcm_manager,
#         settings.K8S_CONFORMANCE_POD_NAME,
#         settings.K8S_CONFORMANCE_NAMESPACE,
#         settings.TARGET_NAMESPACE,
#         settings.TARGET_CLUSTER,
#     )
#     resources.setUp()
#     resources.get_conformance_version()
#     yield resources
#     resources.tearDown()


# @pytest.fixture(scope="module")
# def target_kubectl(kcm_manager):
#     try:
#         kcm_manager.get_kaasreleases()
#     except ApiException:
#         LOG.info("Cannot use kcm_manager for this cluster. "
#                  "target_kubectl will be from KUBECONFIG_PATH")
#         yield K8sCluster(kubeconfig=settings.KUBECONFIG_PATH)
#     else:
#         namespace = kcm_manager.get_namespace(settings.TARGET_NAMESPACE)
#         cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
#         LOG.info("KaaS cluster detected")
#         yield cluster.k8sclient
#
#
# @pytest.fixture(scope="module")
# def target_cluster(kcm_manager: Manager):
#     namespace = kcm_manager.get_namespace(settings.TARGET_NAMESPACE)
#     cluster = namespace.get_cluster(settings.TARGET_CLUSTER)
#     yield cluster
