import http
from kubernetes.client.rest import ApiException
from retry import retry
from functools import wraps
from si_tests import logger
import time

LOG = logger.logger


def get_chart_version(chart_spec):
    if 'version' in chart_spec:
        return chart_spec['version']
    f = chart_spec['chartURL'].split('/')[-1]
    name = chart_spec['name']
    return f[f.find(name)+len(name)+1:f.rfind('.tgz')]


def skip_rc_retry(exceptions=(ApiException), delay=20, tries=8, rcs=None):
    if not rcs:
        rcs = [http.HTTPStatus.NOT_FOUND]

    def skip(func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except exceptions as e:
                if type(e) == ApiException and e.status in rcs:
                    LOG.debug("Return code %s. Won't retry", e.status)
                    raise e
                else:
                    LOG.warning(f"{e}, retrying in {delay} seconds...")
                    time.sleep(delay)
                    return do_retry(*args, **kwargs)

        @retry(exceptions, delay=delay, tries=tries, logger=LOG)
        def do_retry(*args, **kwargs):
            return func(*args, **kwargs)
        return wrapper
    return skip


def relogin_401(func):
    def wrapper(*args, **kwargs):
        try:
            return func(*args, **kwargs)
        except ApiException as e:
            if e.status == 401 and e.reason == 'Unauthorized':
                LOG.debug(f"Kubernetes 'Unauthorized' error:\n{e}")
                LOG.info("Kubernetes 'Unauthorized' error, "
                         "trying to refresh authorization")
                if len(args) == 0:
                    LOG.warning("Skipping refresh authorization: "
                                "not a class instance was decorated")
                elif hasattr(args[0], '_manager'):
                    # Assuming instance of 'K8sBaseResource'
                    cluster = args[0]._manager._cluster
                    cluster.login()
                    cluster.init_apis()
                    return func(*args, **kwargs)
                elif hasattr(args[0], '_cluster'):
                    # Assuming instance of 'K8sBaseManager'
                    cluster = args[0]._cluster
                    cluster.login()
                    cluster.init_apis()
                    return func(*args, **kwargs)
                else:
                    LOG.warning("Skipping refresh authorization: "
                                "no suitable class instance provided")
            raise e
    return wrapper


def retry_for_rcs(exceptions=(ApiException), delay=10, tries=4, rcs=None):
    if not rcs:
        rcs = [http.HTTPStatus.NOT_FOUND]

    def retry_rc(func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except exceptions as e:
                if e.status in rcs:
                    LOG.warning(f"Exception {e}, retrying in {delay} seconds...")
                    time.sleep(delay)
                    return do_retry(*args, **kwargs)
                else:
                    LOG.debug(f"Return code {e.status}. Won't retry")
                    raise e

        @retry(exceptions, delay=delay, tries=tries, logger=LOG)
        def do_retry(*args, **kwargs):
            return func(*args, **kwargs)
        return wrapper
    return retry_rc


# PRODX-36530
# Check exception body and if it's related to auth service - retry few times due highly likely
# this happening when auth leader was upgraded/changed/removed first and we need some time to move leader
# Expecting to use this wrapper only on read ops due getting this usually during upgrade waiters
def retry_auth_500(delay=40, tries=3):
    def lookup(func):
        def wrapper(*args, **kwargs):
            try:
                return func(*args, **kwargs)
            except ApiException as e:
                mark_list = ['unexpected response code from token endpoint',
                             'unable to authenticate user with refresh token on auth provider: http client error']
                mark_condition = any(mark in e.body for mark in mark_list)
                if e.status == 500 and mark_condition:
                    LOG.warning('Possible temporal auth issues detected.'
                                '(PRODX-36530, PRODX-40918)')
                    LOG.warning(f"{e}, retrying in {delay} seconds...")
                    time.sleep(delay)
                    return do_retry(*args, **kwargs)
                else:
                    LOG.debug("Exception details does not match temporal auth issues."
                              "(PRODX-36530, PRODX-40918)"
                              "Won't retry as auth retry strategy.")
                    raise e

        @retry(ApiException, delay=delay, tries=tries, logger=LOG)
        def do_retry(*args, **kwargs):
            return func(*args, **kwargs)
        return wrapper
    return lookup


# PRODX-51739
def retry_429(delay=40, tries=3):
    def lookup(func):
        @wraps(func)
        def wrapper(self, *args, **kwargs):
            try:
                return func(self, *args, **kwargs)
            except ApiException as e:
                error_msg = ['storage is (re)initializing']
                err_condition = any(err in e.body for err in error_msg)
                if e.status == 429 and err_condition:
                    LOG.banner('Possible temporal PRODX-51739 issues detected.')
                    time.sleep(delay)
                    return do_retry(self, *args, **kwargs)
                else:
                    LOG.debug("Exception details does not match temporal issues."
                              "(PRODX-51739). Won't apply retry strategy.")
                    raise e

        @retry(ApiException, delay=delay, tries=tries, logger=LOG)
        def do_retry(self, *args, **kwargs):
            return func(self, *args, **kwargs)
        return wrapper
    return lookup
