from si_tests import logger
from si_tests import settings
import requests
import json
import os
import re

LOG = logger.logger

TYPE = 'json'
NUM_JOBS = settings.RALLY_JOBS_TO_COMPARE
DEVIATION = settings.RALLY_NIGHTLY_DEVIATION_PERCENT
ARTIFACTS_DIR = settings.ARTIFACTS_DIR
WORKFLOW_NAME = settings.RALLY_NIGHTLY_WORKFLOW_NAME

JENKINS_URL = 'https://ci.mcp.mirantis.net/job'
ARTIFACTORY_URL = ''.join(['https://artifactory.mcp.mirantis.net:443/',
                           'artifactory/si-local/jenkins-job-artifacts'])
RALLY_JOB = 'scale-rally-testing-kubernetes'
RALLY_ARTIFACT_PATH = 'artifacts/rally-report.json'
SCENARIO_NAME = "Kubernetes.create_and_delete_pod"
SCENARIOS = ['create_pod', 'delete_pod']
EPS = 10 ** -7


def get_rally_data(file, scenario, metric=r'95%ile'):
    try:
        for tasks in file['tasks']:
            for subtasks in tasks['subtasks']:
                for workloads in subtasks['workloads']:
                    if SCENARIO_NAME in workloads['scenario'].keys():
                        for atomics in workloads['statistics']['durations']['atomics']:
                            atomic_name = re.search('[^.]+$', atomics['name']).group()
                            if atomic_name == scenario:
                                data = float(atomics['data'][metric])
    except KeyError as k:
        LOG.error(f'JSON parsing error: {k}')
    except Exception as e:
        LOG.error(e)
    return data if data else None


def get_json(url):
    file = requests.get(url=url)
    if file.status_code == 404:
        LOG.error(f"URL does not exist! {file.url}")
        return None
    file.raise_for_status()
    return file.json()


def timer_trigger(url):
    causes = "causes"
    class_cause = "hudson.triggers.TimerTrigger$TimerTriggerCause"
    build = get_json(url=url)
    if build:
        try:
            for action in build['actions']:
                if causes in action and len(action["causes"]) == 1:
                    return action['causes'][0]['_class'] == class_cause
        except KeyError as k:
            LOG.error(f'JSON parsing error: {k}')
        except Exception as e:
            LOG.error(e)
    return False


def get_upstream_wf(job):
    wf_number = 0
    description = 'shortDescription'
    upstream = 'upstreamProject'
    causes = "causes"
    for action in job['actions']:
        if causes not in action:
            break
        for cause in action["causes"]:
            if description in cause and cause[description].startswith('Rebuilds'):
                return 0
            if upstream in cause and cause[upstream] == WORKFLOW_NAME:
                upstream_build_number = cause["upstreamBuild"]
                upstream_build_url = f'{JENKINS_URL}/{WORKFLOW_NAME}/{upstream_build_number}/api/json'
                if timer_trigger(upstream_build_url):
                    wf_number = upstream_build_number
    return wf_number


def test_compare_rally_results():
    # current results
    LOG.info('Fetching current rally results')
    rally_report_artifact = settings.get_var('RALLY_REPORT_ARTIFACT', None)
    if not rally_report_artifact:
        LOG.error('Rally artifacts are not defined')

    rally_current = get_json(url=rally_report_artifact)
    if not rally_current:
        LOG.error('Could not fetch current rally results!')
        return False
    else:
        LOG.info(f'Got rally results from {rally_report_artifact}')

    job_number = int(requests.get(
        url=JENKINS_URL + '/' + RALLY_JOB + '/lastSuccessfulBuild/buildNumber').text)
    results = {scenario: [] for scenario in SCENARIOS}

    while (len(results[SCENARIOS[0]]) < NUM_JOBS):
        try:
            # was the job triggered by workflow?
            url = f'{JENKINS_URL}/{RALLY_JOB}/{job_number}/api/json'
            job = get_json(url=url)
            if not job:
                break
            wf_number = get_upstream_wf(job=job)
            if job['result'] != "SUCCESS" or wf_number == 0:
                continue

            # get rally report
            url = f'{ARTIFACTORY_URL}/{RALLY_JOB}/{job_number}/{RALLY_ARTIFACT_PATH}'
            if url == rally_report_artifact:
                continue
            LOG.info(f'Fetching data from {url}')
            rally = get_json(url=url)
            if rally:
                LOG.info(f'Got data from workflow {WORKFLOW_NAME} number {wf_number}')
                for scenario in SCENARIOS:
                    results[scenario].append(
                        get_rally_data(
                            file=rally,
                            scenario=scenario))
            else:
                LOG.warning(f'Could not retrieve data from {RALLY_JOB} number {job["number"]}')
                LOG.debug(f'URL: {url}')
        except Exception as e:
            LOG.error(e)
        finally:
            # previous workflow
            try:
                job_number = job['previousBuild']['number']
            except TypeError:
                LOG.error('No more jobs left')
                break

    # get statistics
    averages = {}
    stat = {}
    for scenario in SCENARIOS:
        try:
            averages[scenario] = sum(results[scenario]) / len(results[scenario])
        except ZeroDivisionError:
            averages[scenario] = -1.0
        stat[scenario] = {}
        LOG.info(f'Comparing {scenario} scenario')
        current = get_rally_data(file=rally_current,
                                 scenario=scenario)
        if current > EPS and averages[scenario] > EPS:
            diff = 100.0 * current / averages[scenario] - 100
            if diff > DEVIATION:
                LOG.warning(f'Current results are {abs(diff):.1f}% slower than average!')
            if diff < -DEVIATION:
                LOG.warning(f'Current results are {abs(diff):.1f}% faster than average!')
            if abs(diff) < DEVIATION:
                LOG.info('Current results are within the deviation range')
        else:
            diff = 0.0
            LOG.warning('There is nothing to compare with!')
        LOG.debug(f'Current result: {current:.3f}\nAverage result: {averages[scenario]:.3f}')

        # dump
        stat[scenario]['current'] = current
        stat[scenario]['average'] = averages[scenario]
        stat[scenario]['diff'] = diff
        stat[scenario]['length'] = len(results[scenario])

    # dump results
    with open(os.path.join(ARTIFACTS_DIR, 'previous_results.json'), 'w') as f:
        json.dump(results, f)
    with open(os.path.join(ARTIFACTS_DIR, 'averages.json'), 'w') as f:
        json.dump(averages, f)
    with open(os.path.join(ARTIFACTS_DIR, 'results.json'), 'w') as f:
        json.dump(stat, f)
