# Copyright 2025 Mirantis, Inc.
#
# Licensed under the Apache License, Version 2.0 (the "License"); you may
# not use this file except in compliance with the License. You may obtain
# a copy of the License at
#
#      http://www.apache.org/licenses/LICENSE-2.0
#
# Unless required by applicable law or agreed to in writing, software
# distributed under the License is distributed on an "AS IS" BASIS, WITHOUT
# WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied. See the
# License for the specific language governing permissions and limitations
# under the License.

import pytest
import random
from si_tests import logger, settings
from si_tests.utils import waiters
from si_tests.managers.netchecker_manager import NetcheckerManager

LOG = logger.logger


def _is_machine_in_expected_status(icm, machine_name, expected_source='inventoryConfigStatus', expected_status='ok'):
    icm_inv_status = icm.data.get('status', {}).get(expected_source, {})
    for m in icm_inv_status.get('nodes', []):
        m_name = m.get('machineName', '')
        m_status = m.get('status', '')
        if m_name == machine_name:
            if m_status == expected_status:
                return True
            else:
                LOG.error(f"Machine {machine_name} has wrong status under 'status.{expected_source}.nodes' field:\n"
                          f"Expected status: {expected_status}\n"
                          f"Actual status: {m_status}")
                break
    return False


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_create_infraconnectivitymonitor_bm(kaas_manager, _, show_step):
    """This test should verify Netchecker feature functionality.

    Scenario:
        1. Create a InfraConnectivityMonitor object
        2. Check that corresponding NetCheckerTargetsConfig object has been created
        3. Check that corresponding CheckerInventoryConfig has been created
        4. Check the status of InfraConnectivityMonitor object
        5. Maintenance mode test (child cluster only)
           Put cluster and selected machines (one-by-one) to maintenance mode
           Verify that node in maintenance mode is not processed
           Disable maintenance mode for node
           Verify that all nodes are processed again. Repeat for all chosen nodes
        6. machineSelector test
           Assign a specific label to few machines
           Set machineSelector in InfraConnectivityMonitor object to this label
           Verify that only selected machines have been processed in all objects
        7. Delete InfraConnectivityMonitor object and verify that it does not exist
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    netchecker = NetcheckerManager(cluster)

    if not cluster.netchecker_enabled:
        skip_msg = (f"Netchecker is disabled in Cluster object {cluster_name}."
                    f"Check 'spec.providerSpec.value.disableInfraConnectivityMonitor'")
        pytest.skip(skip_msg)

    icm_name = "test-create-infraconnectivitymonitor" + str(random.randint(10, 99))
    icm_machine_label = {"test-create-infraconnectivitymonitor" + str(random.randint(10, 99)): "enabled"}

    icm_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "InfraConnectivityMonitor",
        "metadata": {
            "name": icm_name,
            "namespace": namespace_name,
        },
        "spec": {
            "targetCluster": cluster_name,
        }
    }

    machines_to_label = []
    machines_to_maintain = []
    mgmt_machines = cluster.get_machines(machine_type='control')
    assert len(mgmt_machines) > 1, f"We need at least 2 mgmt machines in cluster: {cluster_name}"
    machines_to_label.append(mgmt_machines[0])
    if cluster.is_child:
        worker_machines = cluster.get_machines(machine_type='worker')
        assert len(worker_machines) > 2, f"We need at least 3 worker machine in child cluster: {cluster_name}"
        machines_to_label.append(worker_machines[0])
        machines_to_maintain.append(worker_machines[-1])
        machines_to_maintain.append(worker_machines[-2])

    show_step(1)
    LOG.info(f"Create InfraConnectivityMonitor object {icm_data}")
    icm = ns.create_infraconnectivitymonitor_raw(icm_data)
    icm_list = ns.get_infraconnectivitymonitors()
    assert len(icm_list) == 1, (f"There is more than one InfraConnectivityMonitor "
                                f"for {cluster_name} cluster: {icm_list}. Validation webhook does not work")
    show_step(2)
    nctc_namespace = 'netchecker'
    nctc_name = 'mcc-netchecker-targets-config'
    nctc_creation_timeout = 120
    nctc_creation_interval = 5
    nctc_creation_timeout_msg = (f"NetCheckerTargetsConfig object {nctc_name} in "
                                 f"namespace {nctc_namespace} has not been created")
    waiters.wait(lambda: bool(cluster.k8sclient.cnnc_netcheckertargetsconfigs.present(name=nctc_name,
                                                                                      namespace=nctc_namespace)),
                 timeout=nctc_creation_timeout,
                 interval=nctc_creation_interval,
                 timeout_msg=nctc_creation_timeout_msg)
    show_step(3)
    cic_namespace = 'netchecker'
    cic_name = 'mcc-netchecker-inventory-config'
    cic_creation_timeout = 120
    cic_creation_interval = 5
    cic_creation_timeout_msg = (f"CheckerInventoryConfig object {cic_name} in "
                                f"namespace {cic_namespace} has not been created")
    waiters.wait(lambda: bool(cluster.k8sclient.cnnc_checkerinventoryconfigs.present(name=cic_name,
                                                                                     namespace=cic_namespace)),
                 timeout=cic_creation_timeout,
                 interval=cic_creation_interval,
                 timeout_msg=cic_creation_timeout_msg)
    show_step(4)
    netchecker.wait_compare_targets(netchecker_obj_name=icm_name)
    netchecker.wait_infraconnectivitymonitor_status(icm)

    show_step(5)
    if cluster.is_child:
        icm_machine_in_maint_timeout = 120
        icm_machine_in_maint_interval = 5
        cluster.cluster_maintenance_start()
        for m_maint in machines_to_maintain:
            LOG.info(f"Putting machine {m_maint.name} into Maintenance mode")
            m_maint.machine_maintenance_start()
            netchecker.wait_compare_targets(netchecker_obj_name=icm_name)
            icm_maint_timeout_msg = (f"Machine {m_maint.name} did not appear in InfraConnectivityMonitor "
                                     f"'status.inventoryConfigStatus.nodes' field with "
                                     f"'Maintenance' status after {icm_machine_in_maint_timeout} seconds")
            waiters.wait(lambda: _is_machine_in_expected_status(icm, m_maint.name, expected_status='Maintenance'),
                         timeout=icm_machine_in_maint_timeout,
                         interval=icm_machine_in_maint_interval,
                         timeout_msg=icm_maint_timeout_msg)
            m_maint.machine_maintenance_stop()
        cluster.cluster_maintenance_stop()
        netchecker.wait_compare_targets(netchecker_obj_name=icm_name)
        netchecker.wait_infraconnectivitymonitor_status(icm)
    else:
        LOG.info("Skipping maintenance mode test on mgmt cluster as it does not have worker nodes")

    show_step(6)
    for machine in machines_to_label:
        LOG.info(f"Add label {icm_machine_label} to machine: {machine.name} "
                 f"in cluster: {cluster_name}")
        machine.add_machine_labels(icm_machine_label)
    icm_patch = {
        "spec": {
            "machineSelector": {
                "matchLabels": icm_machine_label,
            }
        }
    }
    icm_old_generation = icm.generation
    LOG.info(f"InfraConnectivityMonitor object {icm_name} generation before patching: {icm_old_generation}")
    LOG.info(f"Patching InfraConnectivityMonitor object: setting machineSelector.matchLabels to "
             f"'{icm_machine_label}'")
    ns.update_infraconnectivitymonitor_raw(name=icm_name, data=icm_patch)
    icm.wait_for_new_generation(old_generation=icm_old_generation)
    LOG.info(f"InfraConnectivityMonitor object {icm_name} generation after patching: {icm.generation}")
    netchecker.wait_compare_targets(netchecker_obj_name=icm_name)
    netchecker.wait_infraconnectivitymonitor_status(icm)
    show_step(7)
    LOG.info(f"Deleting InfraConnectivityMonitor '{icm.name}'")
    icm.delete(async_req=True)
    icm_deletion_timeout = 30
    icm_deletion_interval = 5
    icm_deletion_timeout_msg = f"Deletion failed, icm with name '{icm_name}' still exists"
    waiters.wait(lambda: len([icm for icm in ns.get_infraconnectivitymonitors() if icm_name in icm.name]) == 0,
                 timeout=icm_deletion_timeout,
                 interval=icm_deletion_interval,
                 timeout_msg=icm_deletion_timeout_msg)
    for machine in cluster.get_machines():
        machine.remove_machine_labels(list(icm_machine_label.keys()))


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_infraconnectivitymonitor_poweroff_node_bm(kaas_manager, _, show_step):
    """This test should verify InfraConnectivityMonitor object status when some nodes are powered off.

    Scenario:
        1. Create a InfraConnectivityMonitor object
        2. Check the status of InfraConnectivityMonitor object
        3. Power off one node
        4. Ensure errors in inventory and targets configs
        5. Power on the node, wait for its availability
        6. Ensure all machines are part of inventory and targets configs, no errors in statuses.
           Repeat for all nodes selected to power cycle
        7. Delete InfraConnectivityMonitor object
    """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    netchecker = NetcheckerManager(cluster)

    if not cluster.netchecker_enabled:
        skip_msg = (f"Netchecker is disabled in Cluster object {cluster_name}."
                    f"Check 'spec.providerSpec.value.disableNetChecker'")
        pytest.skip(skip_msg)

    machines_to_poweroff = []
    if cluster.is_child:
        worker_machines = cluster.get_machines(machine_type='worker')
        assert len(worker_machines) > 2, f"We need at least 3 worker machine in child cluster: {cluster_name}"
        machines_to_poweroff.append(worker_machines[0])
        machines_to_poweroff.append(worker_machines[1])
    else:
        skip_msg = (f"Skipping test for the non-child cluster {cluster_name}, as a power-cycle of "
                    f"'control' node may break the whole cluster")
        pytest.skip(skip_msg)

    icm_name = "test-infraconnectivitymonitor-poweroff-node" + str(random.randint(10, 99))

    icm_data = {
        "apiVersion": "kaas.mirantis.com/v1alpha1",
        "kind": "InfraConnectivityMonitor",
        "metadata": {
            "name": icm_name,
            "namespace": namespace_name,
        },
        "spec": {
            "targetCluster": cluster_name,
        }
    }

    show_step(1)
    LOG.info(f"Create InfraConnectivityMonitor object {icm_data}")
    icm = ns.create_infraconnectivitymonitor_raw(icm_data)
    icm_list = ns.get_infraconnectivitymonitors()
    assert len(icm_list) == 1, (f"There is more than one InfraConnectivityMonitor "
                                f"for {cluster_name} cluster: {icm_list}. Validation webhook does not work")

    show_step(2)
    netchecker.wait_compare_targets(netchecker_obj_name=icm_name)
    netchecker.wait_infraconnectivitymonitor_status(icm)

    icm_machine_in_error_timeout = 300
    icm_machine_in_error_interval = 10
    for machine in machines_to_poweroff:
        show_step(3)
        machine.set_baremetalhost_power(online=False)

        show_step(4)
        LOG.info(f"Checking that powered-off machine {machine.name} has 'error' statuses "
                 f"in InfraConnectivityMonitor object {icm_name}")
        icm_error_timeout_msg = (f"Machine {machine.name} did not appear in InfraConnectivityMonitor "
                                 f"'status.inventoryConfigStatus.nodes' field with"
                                 "'error' status after {icm_machine_in_error_timeout} seconds")
        waiters.wait(lambda: _is_machine_in_expected_status(icm, machine.name, expected_status='error'),
                     timeout=icm_machine_in_error_timeout,
                     interval=icm_machine_in_error_interval,
                     timeout_msg=icm_error_timeout_msg)
        icm_error_timeout_msg = (f"Machine {machine.name} did not appear in InfraConnectivityMonitor "
                                 f"'status.targetsConfigStatus.nodes' field with"
                                 "'error' status after {icm_machine_in_error_timeout} seconds")
        waiters.wait(lambda: _is_machine_in_expected_status(icm, machine.name,
                                                            expected_source='targetsConfigStatus',
                                                            expected_status='error'),
                     timeout=icm_machine_in_error_timeout,
                     interval=icm_machine_in_error_interval,
                     timeout_msg=icm_error_timeout_msg)
        netchecker.wait_compare_targets(netchecker_obj_name=icm_name)

        show_step(5)
        machine.set_baremetalhost_power(online=True)

        show_step(6)
        netchecker.wait_compare_targets(netchecker_obj_name=icm_name, timeout=600)
        netchecker.wait_infraconnectivitymonitor_status(icm, timeout=600)

    show_step(7)
    LOG.info(f"Deleting InfraConnectivityMonitor '{icm.name}'")
    icm.delete(async_req=True)
    icm_deletion_timeout = 30
    icm_deletion_interval = 5
    icm_deletion_timeout_msg = f"Deletion failed, icm with name '{icm_name}' still exists"
    waiters.wait(lambda: len([icm for icm in ns.get_infraconnectivitymonitors() if icm_name in icm.name]) == 0,
                 timeout=icm_deletion_timeout,
                 interval=icm_deletion_interval,
                 timeout_msg=icm_deletion_timeout_msg)
