import json
import kubernetes
import pytest
import yaml

from kubernetes.client.rest import ApiException

from si_tests import logger
from si_tests import settings
from si_tests.utils import templates, utils
from si_tests.managers.openstack_manager import OpenStackManager

LOG = logger.logger


def create_test_pod(cluster, machine, base_image_repo=None):
    base_image_repo = base_image_repo or cluster.determine_mcp_docker_registry()
    LOG.info(f"Create pod on {machine.name}")
    machine_type = machine.machine_type
    node_labels = machine.get_k8s_node().read().metadata.labels
    node_name = node_labels['kubernetes.io/hostname']
    pod_name = f"dummy-pod-{utils.gen_random_string(6)}"
    template = templates.render_template(settings.DUMMY_TEST_POD_YAML,
                                         {'POD_NAME': pod_name, 'NODE_NAME': node_name,
                                          'IMAGE_BASE_REPO': base_image_repo})
    json_body = json.dumps(yaml.load(template, Loader=yaml.SafeLoader))
    pod = cluster.k8sclient.pods.create(name='test-pod', body=json.loads(json_body))
    pod.wait_phase('Succeeded')
    LOG.info(f"Start Maintenance for {machine_type} machine {machine.name}")
    machine.machine_maintenance_start()
    cluster.check.wait_machine_status_by_name(
        machine_name=machine.name, expected_status='Maintenance')
    dedicated_control_plane_status = cluster.get_cluster_dedicated_control_plane_status()
    if dedicated_control_plane_status and machine_type == 'control':
        LOG.info("Control node should not be drained for child cluster with dedicatedControlPlane=True")
        assert pod.exists(), "Node should not be drained"
    else:
        LOG.info("Check Kubernetes drain for node")
        node_pods = cluster.get_pods_for_node(machine.get_k8s_node_name())
        remaining_pods = {}
        for p in node_pods:
            try:
                if p.namespace == 'node-feature-discovery' and cluster.workaround.prodx_52237():
                    LOG.warning(
                        f"Skipping pod in node-feature-discovery namespace: {p.namespace}/{p.name}")
                    continue
                if not p.data.get('metadata', {}).get('owner_references', [{}])[0].get('kind') == 'DaemonSet':
                    try:
                        remaining_pods[p.namespace].append(p.name)
                    except KeyError:
                        remaining_pods[p.namespace] = [p.name]
            except ApiException as ex:
                LOG.warning(f"Raised exception while reading pod {p.namespace}/{p.name}:\n{ex}")
                continue

        assert not remaining_pods, \
            "Unexpectedly pods exists on node when is Maintenance mode enabled. Node should be drained\n" \
            f"Remaining pods: \n{yaml.dump(remaining_pods)}"
    if pod.exists():
        pod.delete()


def check_node_maintenance(cluster, control_plane_status, validate_flag=False):
    """
    Turns on the machines into Maintenance mode depending on the type of cluster
    validate_flag: handles the negative scenario by enabling on additional machines the maintenance mod
                   and checks for validation messages, when is True
    :return: None or raises Exception
    """
    LOG.info(
        f"Test node maintenance at dedicatedControlPlane {control_plane_status} on the {cluster._cluster_type} cluster"
    )
    worker_machines_ready = [
        machines for machines in cluster.get_machines(machine_type='worker') if machines.machine_status == 'Ready']
    control_machines_ready = [
        machines for machines in cluster.get_machines(machine_type='control') if machines.machine_status == 'Ready']
    if cluster.is_child:
        control_machines_ready[0].machine_maintenance_start()
        cluster.check.wait_machine_status_by_name(
            machine_name=control_machines_ready[0].name, expected_status='Maintenance')
        control_machines_ready[0].wait_condition_cluster()
        if not control_plane_status:
            if validate_flag:
                # we expect to receive a response without a machine type
                try:
                    worker_machines_ready[0].machine_maintenance_start()
                except kubernetes.client.rest.ApiException as e:
                    message = json.loads(e.body)['message']
                    LOG.info(message)
                    if e.status == 400 and worker_machines_ready[0].machine_type in message:
                        raise Exception(f"Incorrect error message. Expected message without machine type.\n {e}")
                try:
                    control_machines_ready[1].machine_maintenance_start()
                except kubernetes.client.rest.ApiException as e:
                    message = json.loads(e.body)['message']
                    LOG.info(message)
                    if e.status == 400 and control_machines_ready[1].machine_type in message:
                        # we get a response without a machine type
                        raise Exception(f"Incorrect error message. Expected message without machine type.\n {e}")
        else:
            worker_machines_ready[0].machine_maintenance_start()
            cluster.check.wait_machine_status_by_name(
                machine_name=worker_machines_ready[0].name, expected_status='Maintenance')
            worker_machines_ready[0].wait_condition_cluster()
        if control_plane_status and validate_flag:
            # we expect to receive a response with a machine type
            try:
                control_machines_ready[1].machine_maintenance_start()
            except kubernetes.client.rest.ApiException as e:
                message = json.loads(e.body)['message']
                LOG.info(message)
                if e.status == 400 and not control_machines_ready[1].machine_type in message:
                    raise Exception(f"Incorrect error message. Expected message with machine type.\n  {e}")
            try:
                worker_machines_ready[1].machine_maintenance_start()
            except kubernetes.client.rest.ApiException as e:
                message = json.loads(e.body)['message']
                LOG.info(message)
                if e.status == 400 and not worker_machines_ready[1].machine_type in message:
                    raise Exception(f"Incorrect error message. Expected message with machine type.\n  {e}")
            try:
                cluster.cluster_maintenance_stop()
            except kubernetes.client.rest.ApiException as e:
                message = json.loads(e.body)['message']
                LOG.info(message)
                if e.status != 400:
                    raise Exception(f"Incorrect error message\n {e}")
            else:
                raise Exception(
                    "Impossible to turn off the Maintenance cluster mod when the worker node has the status=Maintenance"
                )
        if control_plane_status:
            worker_machines_ready[0].machine_maintenance_stop()
            cluster.check.wait_machine_status_by_name(
                machine_name=worker_machines_ready[0].name, expected_status='Ready')
        control_machines_ready[0].machine_maintenance_stop()
        cluster.check.wait_machine_status_by_name(
            machine_name=control_machines_ready[0].name, expected_status='Ready')
    elif not control_plane_status and (cluster.is_management or cluster.is_regional):
        control_machines_ready[0].machine_maintenance_start()
        cluster.check.wait_machine_status_by_name(
            machine_name=control_machines_ready[0].name, expected_status='Maintenance')
        control_machines_ready[0].wait_condition_cluster()
        if validate_flag:
            # we expect to receive a response with a machine type
            try:
                control_machines_ready[1].machine_maintenance_start()
            except kubernetes.client.rest.ApiException as e:
                message = json.loads(e.body)['message']
                LOG.info(message)
                if e.status == 400 and not control_machines_ready[1].machine_type in message:
                    raise Exception(f"Incorrect error message. Expected message with machine type.\n  {e}")
            try:
                cluster.cluster_maintenance_stop()
            except kubernetes.client.rest.ApiException as e:
                message = json.loads(e.body)['message']
                LOG.info(message)
                if e.status != 400:
                    raise Exception(f"Incorrect error message. Expected message with machine type.\n  {e}")
            else:
                raise Exception(
                    "Impossible to turn off the Maintenance cluster mod when the contlol node has"
                    " the status=Maintenance"
                )
        control_machines_ready[0].machine_maintenance_stop()
        cluster.check.wait_machine_status_by_name(
            machine_name=control_machines_ready[0].name, expected_status='Ready')


@pytest.mark.usefixtures("introspect_distribution_not_changed")
@pytest.mark.usefixtures("collect_downtime_statistics")     # Should be used if ALLOW_WORKLOAD == True
@pytest.mark.usefixtures('post_action_stop_mm_mode')
@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
@pytest.mark.usefixtures('mcc_loadtest_prometheus')
@pytest.mark.usefixtures('mcc_loadtest_grafana')
@pytest.mark.usefixtures('mcc_loadtest_alerta')
@pytest.mark.usefixtures('mcc_loadtest_kibana')
@pytest.mark.usefixtures('mcc_loadtest_alertmanager')
@pytest.mark.usefixtures('mcc_loadtest_keycloak')
def test_positive_maintenance_mod(kaas_manager, _, show_step):
    """Verify Maintenance mode for Cluster/Machine
    Scenario:
        1. Check init state cluster
        2. Cluster maintenance mod on
        3. Check create node maintenance
        4. Test create custom pod and check node status and drain
        5. Stop cluster maintenance mod
        6. Check return cluster to init state
     """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    control_plane_status = cluster.get_cluster_dedicated_control_plane_status()
    worker_machines = cluster.get_machines(machine_type='worker')
    control_machines = cluster.get_machines(machine_type='control')
    show_step(1)
    LOG.info(f"Check init state on the {cluster._cluster_type} cluster {cluster.name} with dedicatedControlPlane ="
             f" {control_plane_status} on {cluster.provider.name} povider")
    assert ((cluster.is_management or cluster.is_regional) and not control_plane_status) or cluster.is_child,\
        "Incorrect {0} cluster status with dedicatedControlPlane = {1} : " \
        "Possible variations:\n" \
        "1) Mgmt cluster with dedicatedControlPlane = False\n" \
        "2) Regional cluster with dedicatedControlPlane = False\n".format(cluster._cluster_type, control_plane_status)
    # Check init state
    cluster.check.check_machines_status()
    cluster.check.wait_cluster_maintenance_status(expected_status=False)
    cluster.check.check_cluster_readiness()

    cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')
    cluster.check.check_k8s_nodes()
    show_step(2)
    LOG.info(f"Cluster {cluster.name} maintenance mod on")
    cluster.cluster_maintenance_start()
    show_step(3)
    LOG.info("Check create node maintenance")
    check_node_maintenance(cluster, control_plane_status)
    show_step(4)
    LOG.info("Test create custom pod and check 'control' node status and drain")
    if cluster.is_child and cluster.provider == utils.Provider.equinixmetalv2:
        create_test_pod(cluster, control_machines[0], base_image_repo='127.0.0.1:44301')
    else:
        create_test_pod(cluster, control_machines[0])
    LOG.info(f"Stop Maintenance for {control_machines[0].machine_type} machine {control_machines[0].name}")
    control_machines[0].machine_maintenance_stop()
    if cluster.is_child:
        LOG.info("Test create custom pod and check 'worker' node status and drain")
        if cluster.provider == utils.Provider.equinixmetalv2:
            create_test_pod(cluster, worker_machines[0], base_image_repo='127.0.0.1:44301')
        else:
            create_test_pod(cluster, worker_machines[0])
        LOG.info(f"Stop Maintenance for {worker_machines[0].machine_type} machine {worker_machines[0].name}")
        worker_machines[0].machine_maintenance_stop()
    show_step(5)
    LOG.info(f"Stop cluster {cluster.name} maintenance mod")
    LOG.info('Stop maintenance for all the nodes if any')
    for ctl in control_machines:
        try:
            ctl.machine_maintenance_stop()
        except Exception as e:
            LOG.warning(e)

    for wrk in worker_machines:
        try:
            wrk.machine_maintenance_stop()
        except Exception as e:
            LOG.warning(e)
    cluster.cluster_maintenance_stop()
    show_step(6)
    LOG.info(f"Check return cluster {cluster.name} to init state")
    cluster.check.check_machines_status()
    cluster.check.wait_cluster_maintenance_status(expected_status=False)
    cluster.check.check_cluster_readiness()

    if cluster.is_mosk:
        LOG.info("Check OpenStack resources")
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open('child_conf', 'w') as f:
            f.write(child_kubeconfig)
        os_manager = OpenStackManager(kubeconfig='child_conf')
        if cluster.tf_enabled():
            cluster.mos_check.check_cassandra_nodes_config(
                os_manager=os_manager,
                actualize_nodes_config=settings.TF_CASSANDRA_NODES_CLENAUP)

    cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')


@pytest.mark.parametrize("_", [f"CLUSTER_NAME={settings.TARGET_CLUSTER}"])
def test_validate_maintenance_mod(kaas_manager, _, show_step):
    """Check validation on maintenance mod for Cluster/Machine
    Scenario:
        1. Check init state cluster
        2. Check validation on cluster maintenance mod off
        3. Cluster maintenance cluster mod on
        4. Check create machine on maintenance cluster mod on
        5. Check validation on node maintenance
        6. Test create custom pod and check node status and drain
        7. Check node health
        8. Stop cluster maintenance mod
        9. Check return cluster to init state
        10. Check that cluster maintenance mode can be enabled during cluster upgrade
     """
    cluster_name = settings.TARGET_CLUSTER
    namespace_name = settings.TARGET_NAMESPACE
    ns = kaas_manager.get_namespace(namespace_name)
    cluster = ns.get_cluster(cluster_name)
    region = kaas_manager.get_mgmt_cluster().region_name
    if cluster.provider != utils.Provider.openstack:
        msg = "Negative mm tests designed to be run only on openstack Provider"
        LOG.warning(msg)
        pytest.skip(msg)
    control_plane_status = cluster.get_cluster_dedicated_control_plane_status()
    worker_machines = cluster.get_machines(machine_type='worker')
    control_machines = cluster.get_machines(machine_type='control')
    show_step(1)
    LOG.info(f"Check init state on the {cluster._cluster_type} cluster {cluster.name} with dedicatedControlPlane ="
             f" {control_plane_status} on {cluster.provider.name} povider")
    assert ((cluster.is_management or cluster.is_regional) and not control_plane_status) or cluster.is_child, \
        "Incorrect {0} cluster status with dedicatedControlPlane = {1} : " \
        "Possible variations:\n" \
        "1) Mgmt cluster with dedicatedControlPlane = False\n" \
        "2) Regional cluster with dedicatedControlPlane = False\n".format(cluster._cluster_type, control_plane_status)
    # Check init state
    cluster.check.check_machines_status()
    cluster.check.wait_cluster_maintenance_status(expected_status=False)
    cluster.check.check_cluster_readiness()
    cluster.check.check_k8s_nodes()

    cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')

    show_step(2)
    LOG.info(f"Validation check on cluster {cluster.name} maintenance mod off")
    try:
        control_machines[1].machine_maintenance_start()
    except kubernetes.client.rest.ApiException as e:
        message = json.loads(e.body)['message']
        LOG.info(message)
        if e.status != 400:
            raise Exception(f"Incorrect error message\n {e}")
    else:
        raise Exception(f"Control machine {control_machines[1].name} should not start at cluster maintenance mod")

    if cluster.is_child:
        try:
            worker_machines[1].machine_maintenance_start()
        except kubernetes.client.rest.ApiException as e:
            message = json.loads(e.body)['message']
            LOG.info(message)
            if e.status != 400:
                raise Exception(f"Incorrect error message\n {e}")
        else:
            raise Exception(f"Worker machine {worker_machines[1].name} should not start at cluster maintenance mod")
    show_step(3)
    LOG.info(f"Cluster {cluster.name} maintenance mod on")
    cluster.cluster_maintenance_start()
    show_step(4)
    existing_master_node = cluster.get_machines(machine_type='control')[0]
    machine_flavor_name = \
        existing_master_node.data['spec']['providerSpec']['value']['flavor']

    machine_image_name = \
        existing_master_node.data['spec']['providerSpec']['value']['image']

    machine_az_name = \
        existing_master_node.data['spec']['providerSpec']['value'][
            'availabilityZone']
    LOG.info("Trying to add 1 master node")
    try:
        cluster.create_os_machine(
            node_type="master",
            region=region,
            node_flavor=machine_flavor_name,
            node_image=machine_image_name,
            node_az=machine_az_name)
    except Exception as e:
        message = json.loads(e.body)['message']
        if e.status != 400 or ("Cannot create machine" not in message or "is in maintenance" not in message):
            raise e
    else:
        raise Exception("Machine should not be create at cluster maintenance mod")
    show_step(5)
    LOG.info("Check validation on node maintenance")
    check_node_maintenance(cluster, control_plane_status, validate_flag=True)
    show_step(6)
    LOG.info("Test create custom pod and check 'control' node status and drain")
    if cluster.provider == utils.Provider.equinixmetalv2:
        create_test_pod(cluster, control_machines[0], base_image_repo='127.0.0.1:44301')
    else:
        create_test_pod(cluster, control_machines[0])
    LOG.info(f"Stop Maintenance for {control_machines[0].machine_type} machine {control_machines[0].name}")
    control_machines[0].machine_maintenance_stop()
    if cluster.is_child:
        LOG.info("Test create custom pod and check 'worker' node status and drain")
        if cluster.provider == utils.Provider.equinixmetalv2:
            create_test_pod(cluster, worker_machines[0], base_image_repo='127.0.0.1:44301')
        else:
            create_test_pod(cluster, worker_machines[0])
        LOG.info(f"Stop Maintenance for {worker_machines[0].machine_type} machine {worker_machines[0].name}")
        worker_machines[0].machine_maintenance_stop()
    show_step(7)
    LOG.info("Check node health")
    cluster.check.check_machines_status()
    cluster.check.check_k8s_nodes()
    if cluster.provider not in utils.Provider.with_ceph():
        cluster.check.check_cluster_readiness()
    show_step(8)
    LOG.info(f"Stop cluster {cluster.name} maintenance mod")
    cluster.cluster_maintenance_stop()
    show_step(9)
    LOG.info(f"Check return cluster {cluster.name} to init state")
    cluster.check.check_machines_status()
    cluster.check.wait_cluster_maintenance_status(expected_status=False)
    cluster.check.check_cluster_readiness()

    if cluster.is_mosk:
        LOG.info("Check OpenStack resources")
        child_kubeconfig_name, child_kubeconfig = cluster.get_kubeconfig_from_secret()
        with open('child_conf', 'w') as f:
            f.write(child_kubeconfig)
        os_manager = OpenStackManager(kubeconfig='child_conf')
        if cluster.tf_enabled():
            cluster.mos_check.check_cassandra_nodes_config(
                os_manager=os_manager,
                actualize_nodes_config=settings.TF_CASSANDRA_NODES_CLENAUP)

    cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')

    # TODO(slalov) Uncomment after fix PRODX-22061
    # show_step(10)
    # # Create proxy and add it to cluster
    # LOG.info("Create proxy object")
    # proxy_object = ns.create_proxyobject(
    #     name=f"{cluster_name}-{settings.KAAS_PROXYOBJECT_NAME}",
    #     region=mgmt_cluster.region_name,
    #     proxy_str=settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
    # proxy_child_name = proxy_object.data['metadata']['name']
    # LOG.info(f"Proxy object {proxy_child_name} created")
    #
    # # Add proxy
    # LOG.info("Add proxy to child")
    # cluster.update_cluster_proxy(proxy_child_name)
    #
    # # Waiting for the transition of one node to the Deploy status
    # cluster.check.check_any_machine_exp_status(
    #     expected_status='Deploy')
    # LOG.info(f"Cluster {cluster.name} maintenance mod on while the cluster is in the upgrade state")
    # cluster.cluster_maintenance_start()
    # LOG.info("Test create custom pod and check node status and drain")
    # create_test_pod(cluster, control_machines[0])
    # LOG.info(f"Stop Maintenance for {control_machines[0].machine_type} machine {control_machines[0].name}")
    # control_machines[0].machine_maintenance_stop()
    # LOG.info(f"Check return cluster {cluster.name} to init state")
    # cluster.check.check_machines_status()
    # cluster.check.wait_cluster_maintenance_status(expected_status=False)
    # cluster.check.check_cluster_readiness()
