import random
import string
import os
import pytest

from si_tests import logger
from si_tests.lcm.openstack_lcm.policies import Policy
from si_tests.utils.utils import verify
from si_tests.deployments.utils import (file_utils, namespace, kubectl_utils)


LOG = logger.logger
kubectl = kubectl_utils.Kubectl()
target_template = "/tmp/empty_stack.yaml"
letters = string.ascii_lowercase
# creating an unique part of a stack name to delete it after the test
name_postfix = ''.join(random.choice(letters) for i in range(5))


@pytest.fixture(scope='function')
def delete_resources(openstack_client_manager):

    # executing test_crud_operations_stack_policy
    yield

    # stacks deletion
    LOG.info("Teardown: Deleting created resources")
    stack_list = openstack_client_manager.stack.list([])
    for stack in stack_list:
        if name_postfix in stack.get('Stack Name'):
            openstack_client_manager.stack.delete([stack.get('ID')])


@pytest.mark.usefixtures('delete_resources')
def test_crud_operations_stack_policy(openstack_client_manager, policy_user):
    """Test CRUD operations for stack policy
       https://mirantis.testrail.com/index.php?/cases/view/4963693

       Parameters required for test execution:
         - KUBECONFIG

       Setup environment:
       Create user with role member
    """
    user_name, password = policy_user
    http_exception = "ERROR: You are not authorized to use stacks:"
    os_service = 'heat'
    os_pod = 'heat-api'
    policy = Policy(os_service, os_pod)
    client_name = openstack_client_manager.client.name
    src = file_utils.join(os.path.dirname(os.path.abspath(__file__)), "templates/empty_stack.yaml")
    destination = (f"{namespace.NAMESPACE.openstack}/{client_name}:{target_template}")
    check_policy_file = ['/bin/sh', '-c', ' cat /etc/heat/policy.d/02-custom.yaml']

    try:
        LOG.info("Setup: Copying heat template")
        # copying Heat template file from the repository to Keystone client pod
        kubectl.cp(src, destination)

        # checking than the policy defining file is not changed before the test
        get_policy_file_before_update = ['/bin/sh', '-c',
                                         r' grep -irn "stacks:index:\|stacks:create:"'
                                         ' /etc/heat/policy.d/02-custom.yaml']
        policy.check_policy_absent_before_update(get_policy_file_before_update)

        LOG.info("Step 1: Checking that policies are absent. Admin and user can create stack and get stack list")
        step = 1

        # creating a stack with the admin credentials
        stack = openstack_client_manager.stack.create([f"test_stack_{name_postfix}", "-t", target_template],
                                                      combined_output=True)
        verify(http_exception not in stack["stderr"],
               f"[Step {step}]: Stack was not created with admin credentials without policies",
               f"[Step {step}]: Policies are absent. Admin can create a new stack")

        # checking the possibility to get the stack details with the admin credentials
        stack_list = openstack_client_manager.stack.list([])
        verify(f"test_stack_{name_postfix}" in str(stack_list),
               f"[Step {step}]: It was impossible to get stack list with admin credentials without policies",
               f"[Step {step}]: Policies are absent. Admin can obtain the stack list")

        # creating a stack with the user credentials
        stack = openstack_client_manager.stack.create([f"test_stack_user_no_policy_{name_postfix}", "-t",
                                                       target_template, f"--os-username={user_name}",
                                                       f"--os-password={password}"],
                                                      combined_output=True)
        verify(http_exception not in stack["stderr"],
               f"[Step {step}]: Stack was not created with user credentials without policies",
               f"[Step {step}]: Policies are absent. User can create a new stack")

        # checking the possibility to get the stack details with the user credentials
        stack_list = openstack_client_manager.stack.list(
            [f"--os-username={user_name}", f"--os-password={password}"])
        verify(f"test_stack_user_no_policy_{name_postfix}" in str(stack_list),
               f"[Step {step}]: It was impossible to get stack list with user credentials without policies",
               f"[Step {step}]: Policies are absent. User can get the stack list")

        LOG.info("Step 2: Adding new rules to OSDPL file")
        step = 2

        # changing the policy defining file, no one will be able to create a stack, no one will be able to watch
        # stack details
        policy_add_role = {'stacks:index': '!', 'stacks:create': '!'}
        policy.update_policy_osdpl(policy_add_role, step)

        LOG.info("Step 3: Checking that the policies were added to Heat API pod")
        step = 3

        # checking that the policies were updated
        policy.wait_policy_updated_on_pod(check_policy_file, policy_add_role, step)

        LOG.info("Step 4: The policies were added. Checking that admin and user cannot create a new stack and cannot"
                 " obtain the stack list")
        step = 4

        # checking the impossibility to create a stack with the admin credentials
        error_output = openstack_client_manager.stack.create(
            [f"test_stack_admin_policy_{name_postfix}", "-t", target_template], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Stack was created with admin credentials and with '!' policies",
               f"[Step {step}]: Policies disallow everyone to create a new stack. Admin cannot create a new stack as"
               f" well")

        # checking the impossibility to get stack details with the admin credentials
        error_output = openstack_client_manager.stack.list([], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: It was possible to get stack list with admin credentials and with '!' policies",
               f"[Step {step}]: Policies disallow everyone to get the stack list. Admin cannot obtain the stack list as"
               f" well")

        # checking the impossibility of stack creating with the user credentials
        error_output = openstack_client_manager.stack.create(
            [f"test_stack_user_policy_{name_postfix}", "-t", target_template, f"--os-username={user_name}",
             f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Stack was created with user credentials and with '!' policies",
               f"[Step {step}]: Policies disallow everyone to create a new stack. User cannot create a new stack as"
               f" well")

        # checking the impossibility to get stack details with the user credentials
        error_output = openstack_client_manager.stack.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: It was possible to get stack list with user credentials and with '!' policies",
               f"[Step {step}]: Policies disallow everyone to get the stack list. User cannot obtain the stack list as"
               f" well")

        LOG.info("Step 5: Updating policies. Checking that the policies were added to Heat API pod")
        step = 5

        # changing the policy defining file, only admin will be able to create a stack and watch stack details
        policy_update_role = {'stacks:index': 'rule:project_admin', 'stacks:create': 'rule:project_admin'}
        policy.update_policy_osdpl(policy_update_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, policy_update_role, step)

        LOG.info("Step 6: The policies were updated. Checking that only admin can create a new stack and obtain the"
                 " stack list")
        step = 6

        # creating a stack with the admin credentials
        stack = openstack_client_manager.stack.create(
            [f"test_stack_admin_only_policy_{name_postfix}", "-t", target_template], combined_output=True)
        verify(http_exception not in stack["stderr"],
               f"[Step {step}]: Stack was not created with admin credentials and with 'rule:project_admin' policies",
               f"[Step {step}]: Policies allow admin to create a new stack, admin can create a new stack")

        # checking the possibility to get stack details with the admin credentials
        stack_list = openstack_client_manager.stack.list([])
        verify(f"test_stack_admin_only_policy_{name_postfix}" in str(stack_list),
               f"[Step {step}]: It was impossible to get stack list with admin credentials and with"
               f" 'rule:project_admin' policies",
               f"[Step {step}]: Policies allow admin to get the stack list, admin can obtain the stack list")

        # checking the impossibility of stack creating with the user credentials
        error_output = openstack_client_manager.stack.create(
            [f"test_stack_user_admin_only_policy_{name_postfix}", "-t", target_template,
             f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: Stack was created with user credentials and with 'rule:project_admin' policies",
               f"[Step {step}]: Policies allow only admin to create a new stack, user cannot create a new stack")

        # checking the impossibility to get stack details with the user credentials
        error_output = openstack_client_manager.stack.list(
            [f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception in error_output["stderr"],
               f"[Step {step}]: It was possible to get stack list with user credentials and with"
               f" 'rule:project_admin' policies",
               f"[Step {step}]: Policies allow only admin to get the stack list, user cannot obtain the stack list")

    finally:
        LOG.info("Step 7: Deleting policies. Checking that the policies were deleted and"
                 " the result is the same as in the first step")
        step = 7

        # changing the policy defining file to the pre-test condition
        policy_not_role = None
        policy.update_policy_osdpl(policy_not_role, step)
        policy.wait_policy_updated_on_pod(check_policy_file, {}, step)

        # creating a stack with the admin credentials
        stack = openstack_client_manager.stack.create(
            [f"test_stack_admin_delete_policy_{name_postfix}", "-t", target_template], combined_output=True)
        verify(http_exception not in stack["stderr"],
               f"[Step {step}]: Stack was not created with admin credentials without policies",
               f"[Step {step}]: Policies are absent. Admin can create a new stack")

        # checking the possibility to get stack details with the admin credentials
        stack_list = openstack_client_manager.stack.list([])
        verify(f"test_stack_admin_delete_policy_{name_postfix}" in str(stack_list),
               f"[Step {step}]: It was impossible to get stack list with admin credentials without policies",
               f"[Step {step}]: Policies are absent. Admin can obtain the stack list")

        # creating a stack with the user credentials
        stack = openstack_client_manager.stack.create(
            [f"test_user_delete_policy_{name_postfix}", "-t", target_template,
             f"--os-username={user_name}", f"--os-password={password}"], combined_output=True)
        verify(http_exception not in stack["stderr"],
               f"[Step {step}]: Stack was not created with user credentials without policies",
               f"[Step {step}]: Policies are absent. User can create a new stack")

        # checking the possibility to get stack details with the user credentials
        stack_list = openstack_client_manager.stack.list(
            [f"--os-username={user_name}", f"--os-password={password}"], yaml_output=False)
        verify(f"test_user_delete_policy_{name_postfix}" in str(stack_list),
               f"[Step {step}]: It was impossible to get stack list with user credentials without policies",
               f"[Step {step}]: Policies are absent. User can get the stack list")
