import ipaddress
import os
from typing import Dict
import paramiko
import re
import json
import unittest
import urllib.parse
import yaml
import copy

from datetime import datetime
from exec_helpers import SSHClient, SSHAuth, \
    CalledProcessError, ExecHelperTimeoutError, Subprocess
from si_tests.utils import packaging_version as version
from retry import retry
from time import sleep

from si_tests import logger
from si_tests import settings
from si_tests.clients.iam import keycloak_client
from si_tests.exceptions import UnknownProviderException
from si_tests.managers.kaas_manager import Manager
from si_tests.managers.aws_manager import AwsManager
from si_tests.managers import si_config_manager
from si_tests.utils import templates, utils, waiters, helpers, templates_adaptor
from si_tests.utils.utils import Provider
from si_tests.utils.certs import CertManager
from si_tests.deployments.utils import kubectl_utils
from kubernetes.client.rest import ApiException
from urllib3.exceptions import MaxRetryError, ProtocolError

LOG = logger.logger


# Example versions are 1.20.1 (release) or 1.20.1-158-d2cd4a8 (master)
kaas_binary_version_regex = re.compile(r"Git Version: (.*)")


def get_kubeconfig_path(kubeconfig_filename="management_kubeconfig", check_only=False):
    """Determine the path to management kubeconfig for mgmt or regional post-checks"""

    kubeconfig_path = f"{settings.ARTIFACTS_DIR}/{kubeconfig_filename}"
    if os.path.isfile(kubeconfig_path):
        return kubeconfig_path

    if settings.KUBECONFIG_PATH is not None:
        if os.path.isfile(settings.KUBECONFIG_PATH):
            return settings.KUBECONFIG_PATH

    if check_only:
        return None
    else:
        raise Exception(f"Management cluster KUBECONFIG not found: "
                        f"environment variable 'KUBECONFIG' is empty, "
                        f"file '{kubeconfig_path}' is missing")


class BootstrapManager(unittest.TestCase):
    def __init__(self, *args, seed_ip=None, priv_key_file=None, **kwargs):
        self.__seed_ip = seed_ip
        self.__priv_key_file = priv_key_file or settings.SEED_SSH_PRIV_KEY_FILE
        self.__bootstrap_target_dir = None
        self.__bm_ansible_env_config_file = '.ansible_state/env-config.yaml'
        super().__init__(*args, **kwargs)

    @staticmethod
    def get_si_config_bootstrap_manager():
        """Init bootstrap manager using seed IP address and SSH private key from SI_CONFIG

        Requires SI_CONFIG environment variable that contains patch to the correct si_config YAML
        """
        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        seed_ip = si_config.data.get('run_on_remote', {}).get('SEED_STANDALONE_EXTERNAL_IP')
        assert seed_ip, f'Seed node IP address not found in {settings.SI_CONFIG_PATH}'
        LOG.info(f"Seed node IP address {seed_ip}")

        key = si_config.data.get('run_on_remote', {}).get('SEED_SSH_PRIV_KEY')
        assert key, f'Seed node SEED_SSH_PRIV_KEY not found in {settings.SI_CONFIG_PATH}'
        with open('seed-key-file', mode='w') as f:
            f.write(key)
        bootstrap_manager = BootstrapManager(seed_ip=seed_ip, priv_key_file='seed-key-file')

        return bootstrap_manager

    def get_seed_ip(self):
        return self.__seed_ip

    def get_kaas_release_file_name(self):
        kaas_release_file = settings.KAAS_RELEASE_FILE
        if kaas_release_file is None:
            kaas_release_file = ''

        if kaas_release_file and kaas_release_file.lower() != 'stable':
            LOG.info(f'KAAS_RELEASE_FILE has been passed:{kaas_release_file}')
            return kaas_release_file

        def ver(n):
            return version.Version(os.path.splitext(n)[0])
        if settings.KAAS_RELEASES_FOLDER and \
                os.path.isdir(settings.KAAS_RELEASES_FOLDER):
            kaas_releases_folder = os.path.join(
                settings.KAAS_RELEASES_FOLDER, 'kaas')
            list_of_kaas_yamls = os.listdir(kaas_releases_folder)
            # remove .bak files
            list_of_kaas_yamls = [x for x in list_of_kaas_yamls if
                                  not x.endswith('.bak')]
            list_of_kaas_yamls.sort(key=ver)
            LOG.debug("Found kaas releases files in {0} "
                      "- {1}".format(settings.KAAS_RELEASES_FOLDER,
                                     list_of_kaas_yamls))
        else:
            raise NameError(
                "Folder {} from variable"
                " settings.KAAS_RELEASES_FOLDER not exist".format(
                    settings.KAAS_RELEASES_FOLDER))

        if kaas_release_file.lower() == 'stable':
            latest_file = None
            for filename in reversed(list_of_kaas_yamls):
                if '-rc.' in filename:
                    continue
                latest_file = filename
                break
        else:
            latest_file = list_of_kaas_yamls[-1]

        LOG.info('Latest kaas release file is {}'.format(latest_file))
        return latest_file

    def is_bootstrap_version_dev(self, bs_version):
        """ check for sha in version, like dev identification"""
        # https://miracloud.slack.com/archives/C11HWTVJ5/p1627549478188000
        if '-' in bs_version and len(bs_version.split('-')[-1]) > 4:
            LOG.info(f"Looks like version {bs_version} is 'dev'")
            return True
        else:
            LOG.info(f"Looks like version {bs_version} is 'release'")
            return False

    def get_bootstrap_version(self):
        kaas_release_filename = self.get_kaas_release_file_name()
        kaas_release_yml_path = os.path.join(settings.KAAS_RELEASES_FOLDER,
                                             'kaas',
                                             kaas_release_filename)
        with open(kaas_release_yml_path, 'r') as f:
            kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)

        bootstrap_version = kaas_release_yml['spec']['bootstrap']['version']
        LOG.info(f"Bootstrap version fetched from "
                 f"file:{kaas_release_yml_path}\n"
                 f"version: {bootstrap_version}")
        return bootstrap_version

    def get_release_templates_version(self):
        kaas_release_filename = self.get_kaas_release_file_name()
        kaas_release_yml_path = os.path.join(settings.KAAS_RELEASES_FOLDER,
                                             'kaas',
                                             kaas_release_filename)
        with open(kaas_release_yml_path, 'r') as f:
            kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)

        release_templates_version = kaas_release_yml['metadata']['name']
        LOG.info(f"release_templates_version:  "
                 f"file:{kaas_release_yml_path}\n"
                 f"version: {release_templates_version}")
        return release_templates_version

    def get_bootstrap_ssh_keyfile_path(self, remote):
        # deprecated path
        r_base = "kaas-bootstrap/"
        if remote.isfile("bootstrap/dev/container-cloud"):
            # baremetal seed custom dir
            r_base = "bootstrap/dev/"
        key_path = "{0}ssh_key".format(r_base)

        # current rc solution
        key_proposal = os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR,
                                    "ssh_key")
        # can't rely on kaas version here because
        # upgraded cluster may still use old key
        if remote.isfile(key_proposal):
            key_path = key_proposal
        return key_path

    def store_cluster_artifacts(self,
                                cluster_namespace,
                                cluster_name,
                                mgmt_kubeconfig_path=None,
                                cluster_kubeconfig_path=None,
                                cluster_ssh_key_path=None):
        '''
        Store all MCC operator output artifacts in one
        place at the seed node.

        :param openstack_client:
        :param cluster_namespace:
        :param cluster_name:
        :param mgmt_kubeconfig_path: used to fetch MCC cluster secrets
        :param cluster_kubeconfig_path: used to avoid touch MCC mgmt API
        :param cluster_ssh_key_path: used to avoid touch MCC mgmt API
        :return:
        '''
        if cluster_kubeconfig_path and cluster_ssh_key_path:
            self.copy_user_file(
                "{kubeconfig} {ssh_key}"
                .format(kubeconfig=cluster_kubeconfig_path,
                        ssh_key=cluster_ssh_key_path),
                settings.KAAS_MGMT_CLUSTER_BOOTSTRAP_USERSPACE,
                "{artifacts_home}/{namespace}/{cluster}/"
                .format(artifacts_home=settings.SEED_NODE_MCC_ARTIFACTS_BASE_DIR,
                        namespace=cluster_namespace,
                        cluster=cluster_name))
        else:
            # TODO add children fetch logic here via mgmt_kubeconfig_path
            raise Exception('there is no way to identify cluster artifacts')

    def get_regional_artifacts(self, cluster_name):
        remote = self.remote_seed()
        meta = {
            "keyfile_path": os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR,
                                         "ssh_key"),
            "kubeconfig_path": f"kubeconfig-{cluster_name}"
        }
        artifacts_home = settings.SEED_NODE_MCC_ARTIFACTS_BASE_DIR
        if remote.isdir(f"{artifacts_home}/default/{cluster_name}"):
            base_path = f"{artifacts_home}/default/{cluster_name}"
            LOG.info("Regional cluster clean seed scenario detected"
                     f" artifacts from {base_path} will be used")

            meta["keyfile_path"] = f"{base_path}/ssh_key"
            meta["kubeconfig_path"] = f"{base_path}/kubeconfig-{cluster_name}"

        return meta

    def get_remote_default_kaas_data(self, remote):
        version_cmd = (
            "grep KAAS_RELEASE_YAML {}/bootstrap.env| "
            "cut -d '=' -f2 ".format(settings.KAAS_BOOTSTRAP_TARGET_DIR))
        LOG.info('Plan to execute {}'.format(version_cmd))
        release_file_path = remote.execute(version_cmd).stdout[0].rstrip()
        LOG.info("Kaas file from boot.env {}".format(
            release_file_path.decode("utf-8")))
        LOG.info("Check founded file exists")

        release_content = self.get_remote_file_yaml(
            filepath=release_file_path.decode("utf-8"))
        LOG.info("Content on kaas release:\n{0}".format(release_content))
        return release_content

    def get_remote_kaas_version(self, remote):
        data = self.get_remote_default_kaas_data(remote)
        return data['spec']['version']

    def get_bootstrap_templates_path(self, remote=None):
        bootstrap_version = self.get_bootstrap_version()
        if settings.BOOTSTRAP_WITH_DEFAULTS:
            bootstrap_version = self.get_remote_default_kaas_data(
                remote)['spec']['bootstrap']['version']
            LOG.info(f"Get default version {bootstrap_version}")
        elif settings.KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE == 'kaas_core' \
                and self.is_bootstrap_version_dev(bootstrap_version):
            # cover case, when we test upgrades in kaas/core repo
            bootstrap_version = 'master'

        bootstrap_templates_path = os.path.expanduser(
            settings.KAAS_BOOTSTRAP_TEMPLATES_DIR)
        bootstrap_templates_path = os.path.join(
            bootstrap_templates_path,
            f'templates_{bootstrap_version}')

        LOG.info(f"Bootstrap template folder placed here: "
                 f"{bootstrap_templates_path}")
        assert os.path.isdir(bootstrap_templates_path), \
            f"Path {bootstrap_templates_path} not found"
        return bootstrap_templates_path

    def get_release_templates_path(self, remote=None):
        if settings.KAAS_CUSTOM_DEPLOYMENT_BOOTSTRAP_TEMPLATES_TARBALL:
            LOG.warning("KAAS_CUSTOM_DEPLOYMENT_BOOTSTRAP_TEMPLATES_TARBALL is specified, "
                        "fallback to bootstrap-based templates path")
            return self.get_bootstrap_templates_path(remote)
        release_version = self.get_release_templates_version()

        if settings.BOOTSTRAP_WITH_DEFAULTS:
            release_version = self.get_remote_default_kaas_data(
                remote)['metadata']['name']
            LOG.info(f"Get default version {release_version}")
        elif settings.KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE == 'kaas_core' \
                and self.is_bootstrap_version_dev(self.get_bootstrap_version()):
            # during kaas/core tests, there might be 2 cases:
            # First: kaas/core master, where we use templates/bootstrap/master
            # Second: kaas/core release/xxx branch, where  we must use templates/bootstrap/kaas-xxx
            if settings.FEATURE_FLAGS.enabled("si-guess-bootstrap-templates"):
                release_version = f"kaas-{self.get_kaas_version().replace('.', '-')}"
            else:
                release_version = 'kaas-master'
            LOG.warning(f"Guessed release_version={release_version}")

        release_templates_path = os.path.expanduser(settings.KAAS_BOOTSTRAP_TEMPLATES_DIR)
        release_templates_path = os.path.join(release_templates_path, f'{release_version}')
        if release_templates_path[-2:] == "rc":
            release_templates_path = release_templates_path[:-3]

        LOG.info(f"Release template folder placed here: "
                 f"{release_templates_path}")
        assert os.path.isdir(release_templates_path), \
            f"Path {release_templates_path} not found"
        return release_templates_path

    def get_kaas_release_yaml_path(self):
        kaas_release_filename = self.get_kaas_release_file_name()
        return os.path.join(settings.KAAS_RELEASES_FOLDER, 'kaas', kaas_release_filename)

    def get_kaas_release_name(self) -> str:
        path = self.get_kaas_release_yaml_path()
        with open(path, 'r') as f:
            kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)
        kaas_name = kaas_release_yml['metadata']['name']
        LOG.info("KaaS name fetched from {} - {}".format(
            path, kaas_name))
        return kaas_name

    def get_kaas_version(self) -> str:
        path = self.get_kaas_release_yaml_path()
        with open(path, 'r') as f:
            kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)
        kaas_version = kaas_release_yml['spec']['version']
        LOG.info("KaaS version fetched from {} - {}".format(
            path, kaas_version))
        return kaas_version

    def get_clusterrelease_from_kaas(self) -> str:
        path = self.get_kaas_release_yaml_path()
        with open(path, 'r') as f:
            kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)
        kaas_version = kaas_release_yml['spec']['version']
        clusterrelease_from_yaml = kaas_release_yml['spec']['clusterRelease']
        LOG.info("Clusterrelease version {} fetched from {}".format(
            clusterrelease_from_yaml, kaas_version))
        return clusterrelease_from_yaml

    def get_clusterrelease_yaml_path(self):
        path = self.get_kaas_release_yaml_path()
        cl_release_version = self.get_clusterrelease_from_kaas()
        with open(path, 'r') as f:
            kaas_release_yml = yaml.load(f.read(), Loader=yaml.SafeLoader)
        supported_v = [s['version'] for s in kaas_release_yml['spec']['supportedClusterReleases'] if
                       s['name'] == cl_release_version]
        assert supported_v, f"Version not found in supported clusterreleases for {cl_release_version} clusterrelease"
        return os.path.join(settings.KAAS_RELEASES_FOLDER, 'cluster', supported_v[0] + '.yaml')

    def get_allowed_distributions(self):
        clusterrelease_path = self.get_clusterrelease_yaml_path()
        with open(clusterrelease_path, 'r') as f:
            cluster_release_yaml = yaml.load(f.read(), Loader=yaml.SafeLoader)
        return utils.get_distribution_relevance(cluster_release_yaml)

    def get_kaas_binary_version(self):
        remote = self.remote_seed()
        bootstrap_target_dir = self.get_bootstrap_target_dir(remote)
        ret = remote.check_call(f"{bootstrap_target_dir}/container-cloud version")
        if len(ret.stderr_lines) == 0:
            raise RuntimeError("kaas binary version output is empty")
        match = kaas_binary_version_regex.search(ret.stderr_lines[0])
        if match is None:
            raise RuntimeError("Failed to parse kaas binary version")
        kaas_binary_version = match.group(1)
        return kaas_binary_version

    @retry(Exception, delay=1, tries=3, jitter=1, logger=LOG)
    def remote_seed(self):

        key = utils.load_keyfile(self.__priv_key_file)
        key = utils.get_rsa_key(key["private"])
        seed_ip = self.get_seed_ip()
        remote = SSHClient(
            host=seed_ip,
            port=22,
            auth=SSHAuth(
                username=settings.SEED_SSH_LOGIN,
                key=key))
        remote.logger.addHandler(logger.console)
        return remote

    def get_bootstrap_target_dir(self, remote):
        if not self.__bootstrap_target_dir:
            if remote.isfile(f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/container-cloud"):
                self.__bootstrap_target_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR
            elif remote.isfile("kaas-bootstrap/container-cloud"):
                self.__bootstrap_target_dir = "kaas-bootstrap"
            elif remote.isfile("bootstrap/dev/container-cloud"):
                self.__bootstrap_target_dir = "bootstrap/dev"
            else:
                raise Exception("Binary 'container-cloud' not found on the seed node")
        return self.__bootstrap_target_dir

    def get_remote_file_yaml(self, filepath):
        key = utils.load_keyfile(settings.SEED_SSH_PRIV_KEY_FILE)
        seed_ip = self.get_seed_ip()
        with templates.YamlEditor(
                file_path=filepath,
                host=seed_ip,
                port=22,
                username=settings.SEED_SSH_LOGIN,
                password="",
                private_keys=[key["private"]]) as editor:
            content = editor.content or {}
        return content

    def put_remote_file_yaml(self, content, filepath):
        '''
        Save content as yaml file on remote seed node
        :param content:
        :param filepath:
        :return:
        '''
        key = utils.load_keyfile(settings.SEED_SSH_PRIV_KEY_FILE)
        seed_ip = self.get_seed_ip()
        with templates.YamlEditor(
                file_path=filepath,
                host=seed_ip,
                port=22,
                username=settings.SEED_SSH_LOGIN,
                password="",
                private_keys=[key["private"]]) as editor:
            editor.content = content

    def copy_user_file(self,
                       source_path,
                       dest_user,
                       dest_path):
        '''
        Copy file between different users based on the single node
        source user should have sudo

        :param source_path:
        :param dest_user:
        :param dest_path:
        :param openstack_client:
        :return:
        '''

        remote = self.remote_seed()
        dest_dir_name = os.path.split(dest_path)[0]

        remote.check_call(f"sudo mkdir -p {dest_dir_name}")
        remote.check_call(f"sudo cp {source_path} {dest_path}")
        remote.check_call("sudo chown {user}:{user} -R {directory}"
                          .format(user=dest_user, directory=dest_dir_name))

    def wait_for_seed_node_ssh(self):
        # wait for ssh on seed node is up
        seed_ip = self.get_seed_ip()
        seed_port = 22
        timeout = 1800
        msg = ("Seed node with IP {0} didn't open SSH in {1} sec"
               .format(seed_ip, timeout))
        waiters.wait_tcp(seed_ip, seed_port, timeout, msg)
        # Let cloud-init to finish "modules-final" section
        # to avoid errors like "E: Unable to locate package docker.io"
        sleep(10)

    def is_kind_running(self,
                        kind_kubeconfig_remote_path=".kube/kind-config-clusterapi",
                        kind_container_name="clusterapi-control-plane"):
        """Check that KinD container is running and the KinD kubeconfig exists"""
        remote = self.remote_seed()
        ret = remote.execute(f"docker ps | grep {kind_container_name}")
        return (len(ret.stdout) > 0 and
                kind_container_name in ret.stdout_str and
                remote.isfile(kind_kubeconfig_remote_path))

    def get_remote_kubeconfig_path(self, kubeconfig_remote_name="kubeconfig"):
        """Get remote kubeconfig path

        :return: path to the local kubeconfig, or None if wasn't downloaded
        """
        remote = self.remote_seed()
        if remote.isfile(kubeconfig_remote_name):
            return kubeconfig_remote_name
        elif remote.isfile(f"bootstrap/dev/{kubeconfig_remote_name}"):
            return f"bootstrap/dev/{kubeconfig_remote_name}"
        elif remote.isfile(f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/{kubeconfig_remote_name}"):
            return f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/{kubeconfig_remote_name}"
        else:
            return None

    def download_remote_kubeconfig(self,
                                   kubeconfig_remote_name="kubeconfig",
                                   kubeconfig_local_name="management_kubeconfig"):
        """Download remote kubeconfig to the local directory

        :return: path to the local kubeconfig, or None if wasn't downloaded
        """
        kubeconfig_remote_path = self.get_remote_kubeconfig_path(kubeconfig_remote_name)
        if kubeconfig_remote_path:
            remote = self.remote_seed()
            remote.download(kubeconfig_remote_path, kubeconfig_local_name)
            return kubeconfig_local_name, kubeconfig_remote_path
        else:
            return None, None

    @retry(tries=10, delay=5, jitter=1, logger=LOG)
    def expose_kind_cluster(self,
                            kind_kubeconfig_remote_path=".kube/kind-config-clusterapi",
                            kind_container_name="clusterapi-control-plane",
                            run_on_remote=False):
        """Expose KinD API on the external seed IP address and patch KinD kubeconfig

        If run_on_remote is True, then assume that the test is running on the same seed node
        where KinD cluster is running. Do not modify KinD kubeconfig and iptables

        return: kind_kubeconfig_local_path, or None if KinD cluster is not running or kubeconfig is not created
        """
        remote = self.remote_seed()
        seed_ip = self.get_seed_ip()
        kind_kubeconfig_local_path = f"{settings.ARTIFACTS_DIR}/kubeconfig-kind"

        if not self.is_kind_running(kind_kubeconfig_remote_path, kind_container_name):
            return None

        LOG.info("Exposing KinD cluster API from the seed node...")
        # 1. Download kind kubeconfig
        remote.download(kind_kubeconfig_remote_path, kind_kubeconfig_local_path)

        if run_on_remote:
            LOG.info("RUN_ON_REMOTE is TRUE, assuming that the KinD cluster is available locally "
                     "on seed node. Skipping KinD cluster kubeconfig modification")
            return kind_kubeconfig_local_path

        # 2. Patch kind kubeconfig
        with templates.YamlEditor(file_path=kind_kubeconfig_local_path) as editor:
            current_content = editor.content
            current_content['clusters'][0]['cluster']['insecure-skip-tls-verify'] = 'true'
            server = current_content['clusters'][0]['cluster']['server']
            server_port = urllib.parse.urlparse(server).port
            assert server_port, f"KinD cluster port is not specified in 'server' object: '{server}'"
            current_content['clusters'][0]['cluster']['server'] = server.replace("127.0.0.1", seed_ip)
            current_content['clusters'][0]['cluster']['insecure-skip-tls-verify'] = True
            # CA is not used when 'insecure' flag is specified, so removing it from kubeconfig
            current_content['clusters'][0]['cluster'].pop('certificate-authority-data', None)
            editor.content = current_content

        with remote.sudo(enforce=True):
            # 3. Get default route device
            ret = remote.check_call("ip -4 route show default")
            assert len(ret.stdout) == 1, (f"Seed node default route configuration contains unexpected data:"
                                          f"'{ret.stdout_str}'")
            # split("dev ") == ['default via 1.1.1.1 ', 'bond0 onlink']
            device = ret.stdout_str.split("dev ")[1].split(" ")[0]

            # 4. Enable route to localnet for the default route device
            remote.check_call(f"sysctl -w net.ipv4.conf.{device}.route_localnet=1")
            # 5. Enable DNAT to KinD port
            remote.execute(f"iptables -t nat -D PREROUTING -i {device} -p tcp --dport {server_port} "
                           f"-j DNAT --to-destination 127.0.0.1")
            remote.check_call(f"iptables -t nat -A PREROUTING -i {device} -p tcp --dport {server_port} "
                              f"-j DNAT --to-destination 127.0.0.1")

        LOG.info(f"KinD cluster API is exposed to {seed_ip}:{server_port}. "
                 f"Kubeconfig is stored to file '{kind_kubeconfig_local_path}'")
        return kind_kubeconfig_local_path

    def is_kind_exposed(self, kind_kubeconfig_path):
        kind_mgr = Manager(kubeconfig=kind_kubeconfig_path)
        try:
            LOG.info("Get clusterrelease names using provided kubeconfig to check if KinD exposed")
            kind_mgr.get_clusterrelease_names()
            return True
        except (ApiException, MaxRetryError, ProtocolError):
            LOG.warning("Could not connect to KinD. It is exposed?")
            return False

    def reboot_seed_node(self):
        remote = self.remote_seed()
        remote.check_call('touch /run/user/$(id -u)/si_rebooted')
        LOG.warning('Rebooting node..')
        with remote.sudo(enforce=True):
            remote.check_call(
                'sync;'
                'nohup bash -c "sleep 2;'
                'iptables -A INPUT -p tcp --dport 22 -j REJECT;'
                'shutdown -r -f now" &>/dev/null & exit',
                verbose=True)
            # just to be sure
            sleep(2)
        self.wait_for_seed_node_ssh()
        remote.reconnect()
        if remote.isfile('/run/user/$(id -u)/si_rebooted'):
            raise Exception('Node has not been rebooted!')

    def _generate_ca_cert(self, ca_path=None):
        """Generate CA key and certificate on seed node

        Return content of ca.pem and ca-key.pem files on seed node,
        and the path on the seed node where these files were stored.
        """
        if not ca_path:
            ca_dir_suffix = datetime.now().strftime('%d-%-H-%-M')
            ca_path = f'si-tests-ca-{ca_dir_suffix}'
        remote = self.remote_seed()
        bootstrap_dir = self.get_bootstrap_target_dir(remote)
        remote.check_call(f"rm -rf {ca_path}",
                          raise_on_err=True,
                          verbose=True)
        remote.check_call(f"mkdir -p {ca_path}; "
                          f"./{bootstrap_dir}/container-cloud get ca "
                          f"--path {ca_path}/ "
                          f"--cn si-tests.test",
                          raise_on_err=True,
                          verbose=True)
        ca_pem = self.get_cert_from_seed(path=os.path.join(ca_path, "ca.pem"))
        ca_key_pem = self.get_cert_from_seed(path=os.path.join(ca_path, "ca-key.pem"))
        return ca_pem, ca_key_pem, ca_path

    def _generate_app_cert(self, cn=None, ips=None, dns_names=None, ca_path=None, cert_dir=None):
        """Generate an application certificate using provided CA on seed node

        Can be provided lists of either 'dns_names' or 'ips', or both, to set into
        Subject Alternative Name (SAN) extension field.

        :param cn: str, CN for the generating certificate. If not set, then a provided DNS or IP will be used
        :param ips: list of str, IP addresses to add into SAN extension
        :param dns_names: list of str, DNS names to add into SAN extension
        :param ca_path: str, directory on the seed node where ca.pem and ca-key.pem are placed
        :param cert_dir: str, sub-directory name to place the generated cert.pem and key.pem: {ca_path}/{cert_dir}/

        Return content of the application certificate and key: cert.pem and key.pem
        """
        remote = self.remote_seed()
        bootstrap_dir = self.get_bootstrap_target_dir(remote)

        if not cert_dir:
            cert_dir = 'certs'

        # Usage:
        #   container-cloud get certificate [flags]
        #
        # Flags:
        #       --ca-path string     Full path to directory that containts CA certificate
        #       --cn string          Common name to put
        #       --dir string         Subdirectory to put requested certificate
        #       --dnsNames strings   Comma separated DNS names to include as SAN
        #   -h, --help               help for certificate
        #       --ips strings        Comma separated IP addresses to include as SAN
        cmd = f"./{bootstrap_dir}/container-cloud get certificate " \
              f"--ca-path {ca_path} " \
              f"--dir {cert_dir}"

        if not (dns_names or ips):
            raise ValueError("Neither hostname nor ip address were provided to generate certificate")

        dns_names = dns_names or []
        ips = ips or []

        if not cn:
            # Use first of dns_names or ips as a CN
            cn = (dns_names + ips)[0]
        cmd += f" --cn {cn}"

        if dns_names:
            cmd += " --dnsNames " + ",".join(dns_names)

        if ips:
            cmd += " --ips " + ",".join(ips)

        remote.check_call(cmd, raise_on_err=True, verbose=True)

        cert_pem = self.get_cert_from_seed(path=os.path.join(ca_path, cert_dir, "cert.pem"))
        key_pem = self.get_cert_from_seed(path=os.path.join(ca_path, cert_dir, "key.pem"))
        return cert_pem, key_pem

    def generate_cert(self, ips=None, dns_names=None, ca_path=None, cert_dir=None):
        """Generate CA and an application certificate with this CA"""
        ca_pem, ca_key_pem, ca_path = self._generate_ca_cert(ca_path=ca_path)
        cert_pem, key_pem = self._generate_app_cert(ips=ips, dns_names=dns_names, ca_path=ca_path, cert_dir=cert_dir)
        return cert_pem, key_pem, ca_pem

    def get_cert_from_seed(self, path):
        remote = self.remote_seed()
        cmd = f"cat {path}"
        cert = remote.check_call(cmd).stdout_str
        return cert

    @utils.log_method_time()
    def step_001_erase_env_before(self):
        """Erase Management/Regional cluster using the seed node with bootstrap binaries"""
        skip_clean = settings.KAAS_SKIP_BOOTSTRAP_CLEANUP
        stack_name = settings.ENV_NAME
        LOG.info(f"Cleanup KaaS cluster from env {stack_name}")
        seed_ip = None if skip_clean else self.get_seed_ip()
        run_envs = {
            "CLUSTER_NAME": settings.CLUSTER_NAME,
            "REGIONAL_CLUSTER_NAME": settings.REGIONAL_CLUSTER_NAME,
            "KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL
        }
        if seed_ip:
            try:
                # handle clean seed regional specific scenario
                regional_artifacts = self.get_regional_artifacts(settings.REGIONAL_CLUSTER_NAME)
                run_envs["REGIONAL_KUBECONFIG"] = \
                    regional_artifacts["kubeconfig_path"]
            except Exception as e:
                LOG.error(e)
        else:
            LOG.warning("No seed node(or it has been skipped)."
                        "Skipping get_regional_artifacts")

        envs_string = utils.make_export_env_strting(run_envs)

        if seed_ip and not skip_clean:
            LOG.info(f"IP address {seed_ip}")
            try:
                remote = self.remote_seed()

                kind_container_name = "clusterapi-control-plane"
                ret = remote.execute(f"docker ps | grep {kind_container_name}")
                if len(ret.stdout) > 0:
                    if settings.MCC_BOOTSTRAP_VERSION != "v2":
                        LOG.info("Found running KinD cluster, trying to cleanup Machine objects from it")
                        if settings.VSPHERE_USE_VVMT_OBJECTS:
                            cmd = ("export KUBECONFIG=~/.kube/kind-config-clusterapi;"
                                   "PATH=${PATH}:~/kaas-bootstrap/bin:/home/ubuntu/bootstrap/dev/bin;"
                                   "kubectl delete vvmt,machines --all")
                        else:
                            cmd = ("export KUBECONFIG=~/.kube/kind-config-clusterapi;"
                                   "PATH=${PATH}:~/kaas-bootstrap/bin:/home/ubuntu/bootstrap/dev/bin;"
                                   "kubectl delete machines --all")
                        remote.execute(cmd, verbose=True, timeout=1800)

                LOG.info("Run 'bootstrap.sh cleanup'")
                bootstrap_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR
                if remote.isfile(f"./{bootstrap_dir}/bootstrap.sh"):
                    try:
                        ret = remote.execute("{env}; bash -x ./{bdir}/bootstrap.sh"
                                             " cleanup".format(
                                                env=envs_string,
                                                bdir=bootstrap_dir),
                                             verbose=True)
                        if ret.exit_code == 0:
                            LOG.info("Cleanup has been finished successfully")
                        else:
                            raise Exception('KaaS cleanup procedure failed')
                    except Exception:

                        # If cleanup fails, KinD cluster may contain
                        # useful logs. Also this method seems to be used only
                        # for management cluster cleanup, so uses its name
                        try:
                            kind_kubeconf = settings.KUBECONFIG_KIND_PATH
                            self.step_collect_logs(
                                remote,
                                cluster_name=settings.CLUSTER_NAME,
                                cluster_namespace=settings.CLUSTER_NAMESPACE,
                                management_kubeconfig_path=kind_kubeconf,
                                kubeconfig_path="",
                                cluster_type="bootstrap")
                        except Exception as e:
                            LOG.warning(
                                "Unable to collect KinD logs: {}".format(e))

                        try:
                            # TODO use step_collect_logs in case of clean seed
                            # because of different ssh_key for extra region
                            # also need to improve step_collect_logs to sniff
                            # kind kubeconfig as well
                            self.step_bs_collect_logs(remote)
                        except Exception as e:
                            LOG.warning("Unable to collect logs: {}".format(e))

                        if settings.KAAS_FORCE_FAIL_ON_BOOTSTRAP_CLEANUP:
                            raise Exception('KaaS cleanup procedure failed')

                        LOG.warning("""
                                Cleanup hasn't been finished successfully
                                Test failure was suppressed
                                set environment variable/Jenkins parameter
                                KAAS_FORCE_FAIL_ON_BOOTSTRAP_CLEANUP: True
                                to force test failure""")
                else:
                    LOG.warning("bootstrap.sh file was not found")
            except OSError:
                LOG.warning("We could not connect to server")
            except paramiko.AuthenticationException:
                LOG.warning("Could not authenticate to server")
        elif skip_clean:
            LOG.warning("Cleanup skipped")
        else:
            LOG.warning("IP address was not found ")

    @utils.log_method_time()
    def step_001_erase_env_before_bv2(self):
        """Erase Management and bootstrap cluster using the seed node with bootstrap binaries"""
        stack_name = settings.ENV_NAME
        if settings.KAAS_SKIP_BOOTSTRAP_CLEANUP:
            LOG.warning('Skipping bootstrap cleanup stage step_001_erase_env_before_bv2')
            return
        LOG.info(f"Cleanup KaaS cluster from env {stack_name}")

        seed_ip = self.get_seed_ip()
        run_envs = {
            "CLUSTER_NAME": settings.CLUSTER_NAME,
            "KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL
        }
        envs_string = utils.make_export_env_strting(run_envs)

        if not seed_ip:
            raise Exception("Seed node IP address was not found")

        LOG.info(f"IP address {seed_ip}")
        remote = self.remote_seed()
        bootstrap_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR
        if remote.isfile(f"./{bootstrap_dir}/bootstrap.sh"):
            LOG.info("Run 'bootstrap.sh cleanup'")
            ret = remote.execute(f"{envs_string}; bash -x ./{bootstrap_dir}/bootstrap.sh cleanup",
                                 verbose=True)

            if ret.exit_code == 0:
                LOG.info("Cleanup has been finished successfully")
            else:
                LOG.warning("Cleanup is not successful, but trying to remove bootstrap cluster anyway")
        else:
            LOG.warning("bootstrap.sh file was not found")

        self.delete_bootstrapv2_cluster()

    @utils.log_method_time()
    def step_001_cleanup_seed_workspace(self, rm_unused_docker_data: bool = True):
        remote = self.remote_seed()
        LOG.info("Cleanup docker resources")
        clean_docker_cmd = "for i in $(docker ps -aq); do docker rm -f $i; done"
        if rm_unused_docker_data:
            clean_docker_cmd += " && docker system prune -a -f"
        remote.check_call(clean_docker_cmd,
                          verbose=True)
        LOG.info("Cleanup possible bootstrap metadata leftovers")
        # TODO vnaumov refactor cleanup after PRODX-9803
        leftovers = (".ssh/openstack_tmp ssh_key kaas_releases kubeconfig"
                     " logs.tar.gz mirantis.lic output.txt"
                     " passwords.yaml .bootstrap_resource_id .kube {}"
                     .format(settings.KAAS_BOOTSTRAP_TARGET_DIR))
        remote.check_call("rm -rf {}".format(leftovers), verbose=True)

    @utils.log_method_time()
    def step_003_cleanup_bootstrapv2_seed_workspace(self) -> None:
        """Removes everything related to a seed node
        except management cluster's kubeconfig and ssh-key due
        the requirement of existance of the these files for
        futher progress (i.e. regional deployment and logs collecting).

        This is mandatory for separated setup of deployments
        within the same seed node.
        """
        remote = self.remote_seed()

        LOG.info("Preserve selective bootstrap related metadata")

        PRESERVE_DIR_NAME = os.path.join(".", "preserve", "")
        remote.check_call(f"mkdir -p {PRESERVE_DIR_NAME}", verbose=True)

        preserved_meta = ["kubeconfig", os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR, "ssh_key"), "ssh_key"]
        for f in preserved_meta:
            directory, _, name = f.rpartition(os.path.sep)
            if directory:
                remote.check_call(f"mkdir -p {os.path.join(PRESERVE_DIR_NAME, directory)}", verbose=True)
            preserved_name = os.path.join(PRESERVE_DIR_NAME, directory, name)
            remote.check_call(f"[ -f {f} ] && cp {f} {preserved_name} || true", verbose=True)
        # now it's safe to remove everything
        self.step_001_cleanup_seed_workspace(rm_unused_docker_data=False)

    @utils.log_method_time()
    def step_004_restore_preserved_bootstrapv2_metadata(self) -> None:
        remote = self.remote_seed()

        LOG.info("Restoring preserved bootstrap metadata")

        PRESERVE_DIR_NAME = os.path.join(".", "preserve", "")
        remote.check_call(f"[ -d {PRESERVE_DIR_NAME} ] && "
                          f"rsync -a {PRESERVE_DIR_NAME} . && "
                          f"rm -fr {PRESERVE_DIR_NAME}",
                          verbose=True)

    def store_jenkins_url(self):
        """
        Store JOB_NAME/BUILD_ID on the seed node
        Returns: None

        """
        remote = self.remote_seed()
        jenkins_url_file = "JENKINS_URL"
        result_str = f"{settings.JENKINS_BUILD_URL}"
        LOG.info(f"Writing Jenkins URL to '{jenkins_url_file}' file")
        remote.check_call(f"echo '{result_str}' > ./{jenkins_url_file}", verbose=False)

    def save_data_to_si_config(self, to_key, data=None) -> None:
        if not data:
            LOG.warning('save_data_to_si_config: no data proposed!')
            return
        LOG.info(f'Update SI_CONFIG:{to_key} ')
        with templates.YamlEditor(settings.SI_CONFIG_PATH) as editor:
            current_content = editor.content
            current_content[to_key] = data
            editor.content = current_content

    def store_stack_info(self, data=None) -> None:
        """
        Store stack data at si_config
        Returns: None

        """
        self.save_data_to_si_config(data=data, to_key='si_heat_stack_info')

    def store_ansible_state_env_config(self, data=None) -> None:
        """
        Store data at si_config
        Returns: None

        """
        self.save_data_to_si_config(data=data, to_key='ansible_state_env_config')

    @utils.log_method_time()
    def step_003_prepare_seed_node_base(self):
        remote = self.remote_seed()
        # store jenkins url to file
        self.store_jenkins_url()
        _sfile = '/tmp/si_step_003_prepare_seed_node_base_done'
        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage step_003_prepare_seed_node_base,'
                        f'since {_sfile} exists ')
            return
        apt_for_cloud_str = 'mirror.mirantis.com'
        # Guess nearest mirror for OS cases
        if settings.USE_MIRA_APT_MIRRORS:
            if settings.CDN_REGION == 'internal-eu':
                apt_for_cloud_str = 'mirror-eu.mcp.mirantis.net'
            if settings.CDN_REGION == 'internal-ci':
                apt_for_cloud_str = 'mirror-us.mcp.mirantis.net'

        if remote.isfile('/etc/redhat-release'):
            LOG.info('Preparing RHEL seed node')
            self.setup_seed_node_rhel(remote)
        else:
            LOG.info('Preparing Ubuntu seed node')
            self.setup_seed_node_ubuntu(remote, apt_for_cloud_str)

        if settings.SEED_NODE_EXTRA_PKG_REBOOT_AFTER:
            self.reboot_seed_node()
            if remote.isfile('/etc/redhat-release'):
                remote.sudo_mode = True  # return sudo context after reboot
        remote.reconnect()
        utils.internal_check_docker_call(remote)
        remote.check_call(f'touch {_sfile}')

    def prepare_seed_node_region_templates(self, remote_templates_dir):
        """Prepare templates on seed node."""
        render_options = {}
        remote = self.remote_seed()
        if not settings.BOOTSTRAP_WITH_DEFAULTS:
            bootstrap_templates_path = self.get_release_templates_path()
        else:
            bootstrap_templates_path = self.get_release_templates_path(remote)

        bootstrap_templates = templates.TemplatesDir(
            bootstrap_templates_path)

        for template in bootstrap_templates:
            remote.check_call("mkdir -p {0}".format(
                os.path.join(remote_templates_dir,
                             os.path.dirname(template.path))))

        if settings.DISTRIBUTION_RELEVANCE != 'default':
            d_rel = settings.DISTRIBUTION_RELEVANCE
            distr_id = self.get_allowed_distributions()[d_rel]['id']
            render_options['DISTRIBUTION'] = distr_id
            LOG.info(f"DISTRIBUTION_RELEVANCE is set to {d_rel}. Will be used {distr_id} for machines")
        for template in bootstrap_templates:
            template_yaml = template.render(options=render_options)
            template_filename = os.path.join(remote_templates_dir,
                                             template.path)
            LOG.info("Writing to the seed node the template {0}"
                     .format(template_filename))
            with remote.open(template_filename, mode="w") as r_f:
                r_f.write(template_yaml)

    def prepare_bm_seed_node_templates(self, env_config_name):
        """Prepare templates on BM seed node."""
        render_options = dict()
        remote = self.remote_seed()
        remote_templates_dir = f'{settings.KAAS_BOOTSTRAP_TARGET_DIR}/templates'
        LOG.info(f"env_config_name: {env_config_name}\nremote templates dir: {remote_templates_dir}")

        if not settings.BOOTSTRAP_WITH_DEFAULTS:
            bootstrap_templates_path = self.get_release_templates_path()
        else:
            bootstrap_templates_path = self.get_release_templates_path(remote)
        if settings.DISTRIBUTION_RELEVANCE != 'default':
            d_rel = settings.DISTRIBUTION_RELEVANCE
            distr_id = self.get_allowed_distributions()[d_rel]['id']
            render_options['DISTRIBUTION'] = distr_id
            LOG.info(f"DISTRIBUTION_RELEVANCE is set to {d_rel}. Will be used {distr_id} for machines")

        env_config_file = self.__bm_ansible_env_config_file
        if remote.isfile(env_config_file):
            si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
            with remote.open(env_config_file) as f:
                env_config_str = f.read().decode()
                try:
                    env_config = yaml.load(env_config_str, Loader=yaml.BaseLoader)
                    LOG.info("env_config loaded:\n{}".format(yaml.dump(env_config)))
                except Exception:
                    raise Exception("Unable to load env_config:\n{}".format(env_config_str))
                # variable MUST be passed only from ansible inventory.
                # don't mix up it into settings.
                if utils.get_string_as_bool(env_config.get('SI_BM_FLAT_VIRTUAL_ENV', False), False):
                    LOG.info('ansible:inventory:SI_BM_FLAT_VIRTUAL_ENV detected')
                    env_config_extra = templates_adaptor.convert_heat_stack_data_to_os_env(
                        si_config.get_heat_stack_info())
                    env_config = {**env_config, **env_config_extra}
                    LOG.info("env_config transformed:\n{}".format(yaml.dump(env_config)))

                current_env_keys = set(os.environ.keys())
                env_config_keys = set(env_config.keys())
                # self-check, for unexpected behavior
                keys_in_common = current_env_keys & env_config_keys

                if keys_in_common:
                    raise Exception("env-config.yaml contains key(s) already"
                                    "defined in os.environ:\n{0}"
                                    .format(keys_in_common))
                LOG.info(f'Injecting data from {env_config_file} to current os.env')
                os.environ.update(env_config)
                # save state config, to be able to render child_data later during child deploy
                # required only for KAAS-BM (half-)virtual labs + child.
                self.store_ansible_state_env_config(data=env_config)
        else:
            LOG.info(f"Remote file '{env_config_file}' does not exist")

        bootstrap_templates = templates.TemplatesDir(bootstrap_templates_path, env_name=env_config_name)

        for template in bootstrap_templates:
            remote.check_call(f"mkdir -p {os.path.join(remote_templates_dir, os.path.dirname(template.path))}")

        for template in bootstrap_templates:
            template_yaml = template.render(options=render_options)
            template_filename = os.path.join(remote_templates_dir,
                                             template.path)
            LOG.info("Writing to the seed node the template {0}"
                     .format(template_filename))
            with remote.open(template_filename, mode="w") as r_f:
                r_f.write(template_yaml)

    def step_003_prepare_seed_node_templates(self):
        remote_templates_dir = f'{settings.KAAS_BOOTSTRAP_TARGET_DIR}/' \
                               f'templates'
        self.prepare_seed_node_region_templates(remote_templates_dir)

    def step_003_patch_bootstrapv2_release_refs(self, template_path_to_apply='templates') -> None:
        remote = self.remote_seed()
        cc_bin = os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR, "container-cloud")
        templates_path = os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR, template_path_to_apply)
        kind_kubeconfig_path = self.get_remote_kubeconfig_path(settings.KUBECONFIG_KIND_PATH)
        remote.check_call(f"{cc_bin} bootstrap cluster-template-v2 "
                          f"--templates-dir {templates_path} "
                          f"--bootstrap-cluster-kubeconfig {kind_kubeconfig_path}")

    @utils.log_method_time()
    def step_003_prepare_seed_node_get_bootstrap(self, endpoints=None):
        LOG.info("Wait for connection check to seed ip")
        self.wait_for_seed_node_ssh()
        remote = self.remote_seed()
        if settings.KAAS_RELEASE_CHARTS_VERSIONS_OVERRIDES:
            LOG.info(f"KAAS_RELEASE_CHARTS_VERSIONS_OVERRIDES is passed to job with next data: "
                     f"{settings.KAAS_RELEASE_CHARTS_VERSIONS_OVERRIDES}. Charts versions will be overwrittne")
            self.step_003_override_release_charts()
        self.rsync_run(from_dir=settings.KAAS_RELEASES_FOLDER,
                       todir=settings.KAAS_RELEASES_REMOTE_FOLDER)
        run_envs = {"KAAS_CDN_REGION": settings.CDN_REGION}
        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
            if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
                run_envs['PROXY_CA_CERTIFICATE_PATH'] = \
                    settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE
            run_envs['HTTP_PROXY'] = \
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['HTTPS_PROXY'] = \
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            LOG.info("HTTP_PROXY and HTTPS_PROXY is set to {0}".format(
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            ))
            # PRODX-13381: add EU/US fip range to no_proxy
            os_ranges = settings.OS_FIP_RANGES
            if endpoints:
                os_ranges = f"{os_ranges},{endpoints}"
            if settings.KAAS_PROXYOBJECT_NO_PROXY:
                run_envs['NO_PROXY'] = \
                    f"{settings.KAAS_PROXYOBJECT_NO_PROXY},{os_ranges}"
            else:
                run_envs['NO_PROXY'] = os_ranges
            run_envs['no_proxy'] = run_envs['NO_PROXY']
            LOG.info("NO_PROXY is set to {0}".format(
                run_envs['NO_PROXY']))

        if settings.MCC_AIRGAP:
            run_envs['MCC_CDN_DOCKER'] = \
                settings.MCC_CDN_DOCKER
            run_envs['MCC_CDN_BINARY'] = \
                settings.MCC_CDN_BINARY
            run_envs['MCC_CDN_DEBIAN'] = \
                settings.MCC_CDN_DEBIAN
            run_envs['MCC_CDN_MCR_REPO'] = \
                settings.MCC_CDN_MCR_REPO
            run_envs['MCC_CDN_CA_BUNDLE_PATH'] = settings.MCC_CDN_CA_BUNDLE_PATH

        if not settings.BOOTSTRAP_WITH_DEFAULTS:
            kaas_release_filename = self.get_kaas_release_file_name()
            run_envs['KAAS_RELEASE_YAML'] = \
                '{}/kaas/{}'.format(settings.KAAS_RELEASES_REMOTE_FOLDER,
                                    kaas_release_filename)
            run_envs['CLUSTER_RELEASES_DIR'] = \
                '{}/cluster'.format(settings.KAAS_RELEASES_REMOTE_FOLDER)
            run_envs['TARGET_DIR'] = settings.KAAS_BOOTSTRAP_TARGET_DIR
            envs_string = utils.make_export_env_strting(run_envs)
            remote.check_call(
                "{envs_string}; bash -x {kaas}/{script}".format(
                    envs_string=envs_string,
                    kaas=settings.KAAS_RELEASES_REMOTE_FOLDER,
                    script=settings.KAAS_GET_SCRIPT_NAME),
                verbose=True)
        else:

            run_envs["KAAS_RELEASES_BASE_URL"] = \
                "{}releases".format(settings.KAAS_RELEASES_BASE_URL)
            envs_string = utils.make_export_env_strting(run_envs)
            remote.check_call(
                "{envs_string}; bash -x {kaas}/{script}".format(
                    envs_string=envs_string,
                    kaas=settings.KAAS_RELEASES_REMOTE_FOLDER,
                    script=settings.KAAS_GET_SCRIPT_NAME),
                verbose=True)

        if settings.KAAS_RELEASE_REPO_URL:
            run_envs['KAAS_RELEASE_REPO_URL'] = settings.KAAS_RELEASE_REPO_URL
            run_envs["KAAS_CDN_BASE_URL"] = settings.KAAS_RELEASE_REPO_URL

        remote.check_call(
            "cd {}; "
            "mv templates templates.orig && mkdir templates".format(
                settings.KAAS_BOOTSTRAP_TARGET_DIR),
            verbose=True)
        if not settings.BOOTSTRAP_WITH_DEFAULTS:
            bootstrap_version = self.get_bootstrap_version()
            LOG.info("Save artifact with KaaS bootstrap version")
            with open("{0}/bootstrap_version".format(settings.ARTIFACTS_DIR),
                      "w") as f:
                f.write(bootstrap_version)

            kaas_version = self.get_kaas_version()
            LOG.info("Save artifact with KaaS bootstrap version")
            with open("{0}/management_version".format(settings.ARTIFACTS_DIR),
                      "w") as f:
                f.write(kaas_version)
        else:
            release_content = self.get_remote_default_kaas_data(remote)
            bootstrap_version = release_content['spec']['bootstrap']['version']
            LOG.info("Bootstrap version fetched from boot.env - {}".format(
                bootstrap_version))
            LOG.info("Save artifact with KaaS bootstrap version")
            with open("{0}/bootstrap_version".format(settings.ARTIFACTS_DIR),
                      "w") as f:
                f.write(bootstrap_version)
            kaas_version = release_content['spec']['version']
            LOG.info(f"KaaS version fetched from boot.env - {kaas_version}")

            LOG.info("Save artifact with KaaS bootstrap version")
            with open("{0}/management_version".format(settings.ARTIFACTS_DIR),
                      "w") as f:
                f.write(kaas_version)
        return run_envs

    @utils.log_method_time()
    def step_003_prepare_seed_node(self, endpoints: str = None, render_templates: bool = True) -> Dict:
        self.wait_for_seed_node_ssh()
        remote = self.remote_seed()

        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR and settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
            remote.upload(
                os.path.abspath(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE),
                settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE,
            )
            LOG.info("Ssl certificate has been uploaded to seed node")
            sudo_mode = True
            if settings.SEED_SSH_LOGIN in settings.SYSTEM_ADMINS:
                sudo_mode = False
            with remote.sudo(enforce=sudo_mode):
                remote.execute(f"cp {settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE} {settings.SEED_CERTS_DIR}")
                remote.execute("update-ca-certificates")

        LOG.info("Verify that the seed node isn't prepared")
        if remote.isdir(settings.KAAS_BOOTSTRAP_TARGET_DIR) and \
                remote.isfile('{}/clouds.yaml'.format(
                    settings.KAAS_BOOTSTRAP_TARGET_DIR)):
            LOG.error('Seed node already prepared')
            return {}

        self.step_003_prepare_seed_node_base()
        run_envs = self.step_003_prepare_seed_node_get_bootstrap(endpoints=endpoints)
        if render_templates:
            self.step_003_prepare_seed_node_templates()

            LOG.info("Render clouds.yaml and kaas templates to seed node")
            clouds_yaml = templates.render_template(settings.CLOUDS_YAML_PATH)
            with remote.open("{}/clouds.yaml".format(
                    settings.KAAS_BOOTSTRAP_TARGET_DIR), mode='w') as r_f:
                r_f.write(clouds_yaml)
        remote.upload(
            os.path.abspath(settings.KAAS_BOOTSTRAP_LICENSE_FILE),
            settings.KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE)

        LOG.info("License is uploaded")
        return run_envs

    # works only with SEED_STANDALONE_EXTERNAL_IP
    @utils.log_method_time()
    def step_003_prepare_standalone_seed_node(self, render_templates: bool = True) -> Dict:
        seed_ip = self.get_seed_ip()
        assert seed_ip
        LOG.info("Seed node ip address is {0}".format(seed_ip))
        with open('{0}/seed_node_ip'.format(settings.ARTIFACTS_DIR),
                  mode='w') as f:
            # TODO: use yaml editor to store the seed info
            f.write(str(seed_ip))

        remote = self.remote_seed()
        run_envs = self.step_003_prepare_seed_node_get_bootstrap()
        if render_templates:
            self.step_003_prepare_seed_node_templates()

        bootstrap_version = self.get_bootstrap_version()
        with open("{0}/bootstrap_version".format(settings.ARTIFACTS_DIR),
                  "w") as f:
            f.write(bootstrap_version)

        remote.upload(
            settings.KAAS_BOOTSTRAP_LICENSE_FILE,
            settings.KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE)
        LOG.info("License is uploaded")
        return run_envs

    def step_003_override_release_charts(self):
        overrides = yaml.safe_load(settings.KAAS_RELEASE_CHARTS_VERSIONS_OVERRIDES)
        release_path = self.get_kaas_release_yaml_path()
        with open(release_path, 'r') as f:
            data = yaml.safe_load(f)
            bootstrap_charts = data['spec']['bootstrap']['helmReleases']
            mgmt_charts = data['spec']['management']['helmReleases']

        def override_versions(release_charts, charts_overrides):

            for k, v in charts_overrides.items():
                chart_version_pattern = r'(\d+(\.\d+)*(-\w+)?)'
                for chart in release_charts:
                    if chart['name'] == k:
                        LOG.info(f"Overriding chart {k} to version {v}")
                        chart_url = chart['chartURL']
                        new_chart_url = re.sub(chart_version_pattern, v, chart_url)
                        chart['chartURL'] = new_chart_url
                        chart['version'] = v
                        if chart['values'].get('image', {}).get('tag', ''):
                            chart['values']['image']['tag'] = v
        LOG.info("Overriding bootstrap charts")
        override_versions(bootstrap_charts, overrides)
        LOG.info("Overriding mgmt charts")
        override_versions(mgmt_charts, overrides)
        with open(release_path, 'w') as new_f:
            new_f.write(yaml.dump(data))

    @utils.log_method_time()
    def step_003_prepare_bm_seed_node(self, endpoints=None):
        # TODO(alexz) looks like should be removed
        self.wait_for_seed_node_ssh()
        remote = self.remote_seed()
        # store jenkins url to file
        self.store_jenkins_url()
        _sfile = '/tmp/step_003_prepare_bm_seed_node'
        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage step_003_prepare_bm_seed_node,'
                        f'since {_sfile} exists ')
            return
        kaas_rel_dir = settings.KAAS_RELEASES_FOLDER

        # Save seed node IP
        seed_ip = self.get_seed_ip()
        assert seed_ip
        LOG.info("Seed node ip address is {0}".format(seed_ip))
        with open('{0}/seed_node_ip'.format(settings.ARTIFACTS_DIR),
                  mode='w') as f:
            # TODO: use yaml editor to store the seed info
            f.write(str(seed_ip))

        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR and settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
            remote.upload(
                os.path.abspath(settings.KAAS_SSL_PROXY_CERTIFICATE_FILE),
                settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE,
            )
            LOG.info("Ssl certificate has been uploaded to seed node")
            sudo_mode = True
            if settings.SEED_SSH_LOGIN in settings.SYSTEM_ADMINS:
                sudo_mode = False
            with remote.sudo(enforce=sudo_mode):
                remote.execute(f"cp {settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE} {settings.SEED_CERTS_DIR}")
                remote.execute("update-ca-certificates")

        self.step_003_prepare_seed_node_base()
        self.ansible_init()

        self.step_003_prepare_bm_env_override()

        if os.path.isdir(kaas_rel_dir):
            LOG.info("Uploading kaas-releases dir to 'node0'")
            self.rsync_run(from_dir=kaas_rel_dir,
                           todir=f'~/{settings.KAAS_RELEASES_REMOTE_FOLDER}')
        else:
            LOG.warning('kaas-releases dir has not been found.')

        # Upload license file
        remote.upload(
            os.path.abspath(settings.KAAS_BOOTSTRAP_LICENSE_FILE),
            settings.KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE)
        LOG.info("License is uploaded")

        kaas_version = self.get_kaas_version()
        LOG.info("Save artifact with KaaS bootstrap version")
        with open("{0}/management_version".format(settings.ARTIFACTS_DIR),
                  "w") as f:
            f.write(kaas_version)

        self.step_003_prepare_seed_node_get_bootstrap(endpoints=endpoints)
        remote.check_call(f'touch {_sfile}')

    @utils.log_method_time()
    def step_003_prepare_bm_env_override(self):
        """
        Prepare and upload ansible ENV file

        :param openstack_client: Openstack client
        :return: None
        """
        env_dict = dict(CLOUD_NAME=settings.CLOUD_NAME,
                        KAAS_CDN_REGION=settings.CDN_REGION,
                        CDN_BOOTSTRAP_REGION=settings.CDN_REGION,)
        kaas_version = self.get_kaas_version()
        if settings.DNS_NAMESERVER:
            env_dict['DNS_NAMESERVER'] = settings.DNS_NAMESERVER
        if settings.KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE:
            env_dict[
                'KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE'] = settings.KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE  # noqa
        # Pass global variables to ansible.
        if settings.KAAS_BM_PUBLIC_API_CHART_URL:
            env_dict[
                'KAAS_BM_PUBLIC_API_CHART_URL'] = settings.KAAS_BM_PUBLIC_API_CHART_URL  # noqa
        if settings.KAAS_BM_OPERATOR_CHART_URL:
            env_dict[
                'KAAS_BM_OPERATOR_CHART_URL'] = settings.KAAS_BM_OPERATOR_CHART_URL  # noqa
        if settings.KAAS_BM_OPERATOR_CHART_VERSION:
            env_dict['KAAS_BM_CHARTS_REPO'] = settings.KAAS_BM_CHARTS_REPO
        if settings.KAAS_BM_OPERATOR_DOCKER_IMAGE:
            env_dict[
                'KAAS_BM_OPERATOR_DOCKER_IMAGE'] = settings.KAAS_BM_OPERATOR_DOCKER_IMAGE  # noqa
        if settings.KAAS_BM_IPAM_CHART_URL:
            env_dict[
                'KAAS_BM_IPAM_CHART_URL'] = settings.KAAS_BM_IPAM_CHART_URL
        if settings.KAAS_BM_IPAM_DOCKER_IMAGE:
            env_dict[
                'KAAS_BM_IPAM_DOCKER_IMAGE'] = settings.KAAS_BM_IPAM_DOCKER_IMAGE  # noqa
        if settings.KAAS_BM_KAAS_IPAM_DOCKER_IMAGE:
            env_dict[
                'KAAS_BM_KAAS_IPAM_DOCKER_IMAGE'] = settings.KAAS_BM_KAAS_IPAM_DOCKER_IMAGE  # noqa
        if settings.KAAS_BM_IRONIC_OPERATOR_DOCKER_IMAGE:
            env_dict[
                'KAAS_BM_IRONIC_OPERATOR_DOCKER_IMAGE'] = settings.KAAS_BM_IRONIC_OPERATOR_DOCKER_IMAGE  # noqa
        if settings.CDN_REGION:
            env_dict['KAAS_CDN_REGION'] = settings.CDN_REGION
        if settings.KAAS_BM_KAAS_ANSIBLE_TAR_GZ:
            env_dict[
                'KAAS_BM_KAAS_ANSIBLE_TAR_GZ'] = settings.KAAS_BM_KAAS_ANSIBLE_TAR_GZ  # noqa
        if settings.KAAS_BM_KAAS_ANSIBLE_TAR_GZ_MD5:
            env_dict[
                'KAAS_BM_KAAS_ANSIBLE_TAR_GZ_MD5'] = settings.KAAS_BM_KAAS_ANSIBLE_TAR_GZ_MD5  # noqa
        if settings.KAAS_BM_DEPLOY_KAAS_BASED_ON_SOURCE:
            kaas_release_filename = self.get_kaas_release_file_name()
            kaas_release_yml_remote_path = os.path.join(
                settings.KAAS_RELEASES_REMOTE_FOLDER,
                'kaas',
                kaas_release_filename)
            # Processed to get_container_cloud.sh
            env_dict['KAAS_RELEASE_YAML'] = kaas_release_yml_remote_path
            env_dict['CLUSTER_RELEASES_DIR'] = '{}/cluster'.format(
                settings.KAAS_RELEASES_REMOTE_FOLDER)
        if settings.KAAS_PLATFORM_RELEASE_REF:
            env_dict[
                'KAAS_PLATFORM_RELEASE_REF'] = settings.KAAS_PLATFORM_RELEASE_REF  # noqa
        if settings.KAAS_RELEASE_REPO_URL:
            env_dict['KAAS_RELEASE_REPO_URL'] = settings.KAAS_RELEASE_REPO_URL
        if settings.KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE:
            env_dict['KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE'] = \
                settings.KAAS_BOOTSTRAP_LICENSE_REMOTE_FILE
        if settings.KEYCLOAK_LDAP_ENABLED:
            env_dict['KEYCLOAK_LDAP_ENABLED'] = settings.KEYCLOAK_LDAP_ENABLED
            env_dict['KEYCLOAK_LDAP_BIND_USERNAME'] = \
                settings.KEYCLOAK_LDAP_BIND_USERNAME
            env_dict['KEYCLOAK_LDAP_BIND_PASSWORD'] = \
                settings.KEYCLOAK_LDAP_BIND_PASSWORD
        if settings.USE_MIRA_APT_MIRRORS:
            env_dict['USE_MIRA_APT_MIRRORS'] = \
                settings.USE_MIRA_APT_MIRRORS

        seed_floating_ip = self.get_seed_ip()
        if seed_floating_ip:
            env_dict['BM_SEED_FLOATING_IP'] = seed_floating_ip

        if settings.KAAS_RELEASES_BASE_URL:
            env_dict['KAAS_RELEASES_BASE_URL'] = settings.KAAS_RELEASES_BASE_URL   # noqa
        if settings.KAAS_BM_CI_SCALEDOWNRESOURCES:
            env_dict['KAAS_BM_CI_SCALEDOWNRESOURCES'] = settings.KAAS_BM_CI_SCALEDOWNRESOURCES  # noqa
        env_dict['KAAS_BM_BOOT_UEFI'] = settings.KAAS_BM_BOOT_UEFI

        # As for release 2.3+ we need to pass correct region, even for
        # mgmt cluster deployment, especially for custom bmhp for mgmt case
        region = '' if utils.version_greater_than_2_26_0(kaas_version) else settings.BM_REGION
        env_dict['BM_REGION'] = region

        if settings.BM_DEPLOY_CHILD:
            env_dict['BM_DEPLOY_CHILD'] = settings.BM_DEPLOY_CHILD

        if settings.BM_HALF_VIRTUAL_ENV:
            env_dict['BM_HALF_VIRTUAL_ENV'] = settings.BM_HALF_VIRTUAL_ENV
            env_dict['BM_HALF_VIRTUAL_NODE'] = settings.BM_HALF_VIRTUAL_NODE or ''

        if settings.MCC_AIRGAP:
            LOG.info('Custom cdn enabled')
            env_dict['MCC_CDN_DOCKER'] = settings.MCC_CDN_DOCKER
            env_dict['MCC_CDN_DEBIAN'] = settings.MCC_CDN_DEBIAN
            env_dict['MCC_CDN_BINARY'] = settings.MCC_CDN_BINARY
            env_dict['MCC_CDN_MCR_REPO'] = settings.MCC_CDN_MCR_REPO
            env_dict['MCC_CDN_CA_BUNDLE_PATH'] = settings.MCC_CDN_CA_BUNDLE_PATH
        elif settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
            env_dict['HTTP_PROXY'] = \
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            env_dict['HTTPS_PROXY'] = \
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            env_dict['NO_PROXY'] = \
                settings.KAAS_NO_PROXY_ACCESS_STR
            if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
                env_dict['PROXY_CA_CERTIFICATE_PATH'] = f'/home/{settings.SEED_SSH_LOGIN}/'\
                                                        f'{settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE}'
            else:
                env_dict['PROXY_CA_CERTIFICATE_PATH'] = ''

        if settings.KAAS_OFFLINE_DEPLOYMENT:
            # Use the seed node as the default GW for cluster nodes
            # those pass ignored for virtual case, see setup-kaas-bm-pbook.yml
            # FIXME(alexz): those event for virt case must be refactored
            #  For virt case, manually set SET_PXE_NW_GW=10.0.0.15
            env_dict['SET_PXE_NW_GW'] = settings.PHYS_GATEWAY

        self.put_remote_file_yaml(env_dict, settings.ANSIBLE_ENV_OVERRIDE_FILE)

    def check_templates(self):
        """Match local templates to default templates from KaaS

        Match the local templates rendered with default variables
        from the file {KAAS_BOOTSTRAP_TEMPLATES_DIR}/.defaults
        to the default templates from the KaaS bootstrap archive.

        This allows to catch any changes made to default templates
        and fix it in the local templates in si-tests.

        Local templates by default should be placed in the directories
        according to the KaaS bootstrap archive version:
        templates/bootstrap/templates_{KAAS_VERSION}/

        Default KaaS templates are expected on the seed node in the
        directory '~/templates.orig/'.

        Scenario:

            1. Compare list of the local and default template files
            2. Render a local template to a string object using Jinja2
            3. Read the appropriate default template from the seed node
            4. Convert both templates to python objects using yaml.load
               to check the yaml syntax.
            5. Convert both templates back to string using yaml.dump
               to get the same result without comments, with the same
               indents and the same keys order
            6. Compare the resulting strings using assertEqual, to get
               a nice diff in the log.
        """
        if not settings.BOOTSTRAP_WITH_DEFAULTS:
            bootstrap_templates_path = self.get_release_templates_path()
        else:
            remote = self.remote_seed()
            bootstrap_templates_path = self.get_release_templates_path(remote)

        remote = self.remote_seed()
        tmpl_orig = f'{settings.KAAS_BOOTSTRAP_TARGET_DIR}/templates.orig'

        self._check_templates(tmpl_orig, bootstrap_templates_path, remote)

    def _check_templates(self, tmpl_orig, bootstrap_templates_path, remote):

        # Read .defaults file from the templates dir
        with open('{0}/.defaults'.format(bootstrap_templates_path)) as f:
            default_variables = yaml.load(f.read(), Loader=yaml.SafeLoader)

        def os_env(var_name, default=None, env_type=None):
            # Define variables that will produce the default template
            var = default_variables.get(var_name, default)

            if var is None:
                raise Exception("Environment variable '{0}' is undefined!"
                                .format(var_name))

            if env_type == "bool":
                var = settings.get_var_as_bool(var_name, default)

            return var

        def os_env_bool(var_name, default=None):
            return os_env(var_name, default, env_type='bool')

        res = remote.execute(f"cd {tmpl_orig}; find . -type f")
        orig_templates = [x.decode("utf-8").strip().replace('./', '')
                          for x in res.stdout]

        bootstrap_templates = templates.TemplatesDir(
            bootstrap_templates_path, test_mode=True)
        LOG.info(f'Compare files in\n'
                 f'remote:{tmpl_orig} VS {bootstrap_templates_path}')
        self.assertEqual(set(orig_templates),
                         set(bootstrap_templates.filenames))

        self.maxDiff = None
        render_options = {'os_env': os_env, 'os_env_bool': os_env_bool,
                          'TEST': True}

        for template in bootstrap_templates:
            # render with overriden os_env
            template_yaml = yaml.load_all(
                template.render(options=render_options),
                Loader=yaml.SafeLoader)

            template_name_file = os.path.join(tmpl_orig, template.path)

            with remote.open(template_name_file, mode="r") as r_f:
                orig_yaml = yaml.load_all(r_f.read(), Loader=yaml.SafeLoader)
                # NOTE(vsaienko): remove anchors from yamls.
                self.assertEqual(yaml.dump_all(template_yaml, Dumper=templates.NoAliasDumper),
                                 yaml.dump_all(orig_yaml, Dumper=templates.NoAliasDumper))

    # used to find suitable facility right before mgmt/regional bootstrap
    def override_equinix_facility(self, templates_folder='templates'):
        LOG.info('Override facility for MCC bootstrap to avoid quota limit')
        remote = self.remote_seed()

        # backup original
        template_file = "{0}/{1}/cluster.yaml.template".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            templates_folder)
        utils.backup_file(filepath=template_file, remote=remote)

        # find suitable facility for mgmt/regional
        facility = utils.get_available_equinix_metro(
            settings.KAAS_EQUINIX_USER_API_TOKEN,
            settings.KAAS_EQUINIX_MACHINE_TYPE, 3)

        with remote.open(template_file, 'r') as r_f:
            cluster_template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
            cluster_template['spec']['providerSpec'][
                'value']['facility'] = facility
        with remote.open(template_file, 'w') as r_f:
            r_f.write(yaml.dump(cluster_template))

    def equinixmetalv2_configure_networking(self, only_region=False):
        LOG.info('Configure network setup for equinixmetalv2 MCC bootstrap')
        # getting infra network config
        with open(settings.KAAS_EQUINIX_PRIVATE_NETWORK_CONFIG, 'r') as base:
            base_network_config = json.load(base)

        # equinixmetalv2_configure_networking starts only once before mgmt bootstrap,
        # so find the seed node which IP address match the SEED_STANDALONE_EXTERNAL_IP
        # as mgmt base, all others will be considered as regionals
        seed_ips = list(base_network_config["seed_nodes"]["value"].keys())
        assert settings.SEED_STANDALONE_EXTERNAL_IP in seed_ips, (
            f"No suitable seed node found to bootstrap a MCC management cluster. "
            f"List of seed nodes provided from KAAS_EQUINIX_PRIVATE_NETWORK_CONFIG: '{seed_ips}' ,"
            f"while the provided seed IP address SEED_STANDALONE_EXTERNAL_IP='{settings.SEED_STANDALONE_EXTERNAL_IP}'")
        mgmt_seed_ip = settings.SEED_STANDALONE_EXTERNAL_IP
        mgmt_seed_vlan_id = str(base_network_config["seed_nodes"]["value"][mgmt_seed_ip]["vlan_id"])
        mgmt_seed_metro = str(base_network_config["seed_nodes"]["value"][mgmt_seed_ip]["metro"])
        LOG.info(f"MCC mgmt equinixmetalv2 bootstrap will use seed node {mgmt_seed_ip} attached "
                 f"to VLAN {mgmt_seed_vlan_id} in metro {mgmt_seed_metro}")
        management_network_config = {}
        region_network_configs = []
        child_network_configs = []

        for metro, config in base_network_config["vlans"]["value"].items():
            for v in config:
                # generate MCC network specification
                network = ipaddress.IPv4Network(
                    u"{}/{}".format(v['subnet'], v['mask']))
                lb_host = str(network[10])
                # TODO(vnaumov) ipam/dhcp/metallb ranges will be reafactored in Backend soon
                dhcp_range = f"{network[11]}-{network[50]}"
                ipam_ranges = f"{network[51]}-{network[99]}"
                metallb_ranges = f"{network[100]}-{network[200]}"
                cluster_network_config = {
                    "provider": "equinixmetalv2",  # just to label the config
                    "mcc_regional": v['mcc_regional'],
                    "metro": metro,
                    "networkSpec": {
                        "private": True,
                        "vlanId": str(v['vlan_id']),
                        "loadBalancerHost": lb_host,
                        "metallbRanges": [metallb_ranges],
                        "cidr": str(network),
                        "gateway": v['router_addr'],
                        "includeRanges": [ipam_ranges],
                        "dhcpRanges": [dhcp_range],
                        "nameservers": [],
                    },
                }
                # specific bootstrap metadata for mgmt/regional
                if v['mcc_regional']:
                    pxe_ip = str(network[5])
                    pxe_mask = str(network.prefixlen)
                    cluster_network_config['bootstrap'] = {
                        "bm_pxe_ip": pxe_ip,
                        "bm_pxe_mask": pxe_mask,
                        "bm_dhcp_range": dhcp_range,
                    }

                # Determine which cluster config is there
                # 1. If still no management config - check that the current config has a vlan with a seed node inside
                if (not management_network_config
                        and v['mcc_regional']
                        and str(v['vlan_id']) == mgmt_seed_vlan_id
                        and v['metro'] == mgmt_seed_metro
                        and not only_region):
                    management_cluster_key = f"{settings.CLUSTER_NAMESPACE}/{settings.CLUSTER_NAME}"
                    management_network_config = {
                        management_cluster_key: cluster_network_config
                    }
                    LOG.info(f"""MCC Equinixmetalv2 mgmt ========:
                                 {management_network_config}\n""")
                # 2. If the network config don't match for management cluster - use it for region
                elif v['mcc_regional']:
                    region_network_configs.append(cluster_network_config)
                # 3. If mcc_regional is False, then it is a config for a child cluster
                else:
                    child_network_configs.append(cluster_network_config)

        # return clusters_net_config
        LOG.info(f"""MCC Equinixmetalv2 mgmt private network config:
                     {management_network_config}\n
                     Regional(s) private network config(s):
                     {region_network_configs}\n
                     Children private network config(s):
                     {child_network_configs}""")
        return management_network_config, region_network_configs, child_network_configs

    def equinixmetalv2_patch_templates(self,
                                       cluster_namespace,
                                       cluster_name,
                                       network_config,
                                       machines_amount,
                                       templates_folder='templates',
                                       proxy='',
                                       custom_region=False):
        """
        network_config: dict { "<cluster_namespace>/<cluster_name>": <network_parameters>, ...}
                        For a region cluster, can be taken from SI_CONFIG key ["network_config"]["equinixmetalv2"]
        """
        assert settings.SEED_STANDALONE_EXTERNAL_IP, 'works with SEED_STANDALONE_EXTERNAL_IP only'
        # getting MCC network configuration
        cluster_networking = utils.equinixmetalv2_get_network_config(cluster_namespace,
                                                                     cluster_name,
                                                                     network_config,
                                                                     machines_amount)

        remote = self.remote_seed()
        # backup original
        cluster_template_file = "{0}/{1}/cluster.yaml.template".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            templates_folder)
        utils.backup_file(filepath=cluster_template_file, remote=remote)

        use_metallbconfig = False
        metallbconfig_template_file = "{0}/{1}/metallbconfig.yaml.template".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            templates_folder)

        if remote.isfile(metallbconfig_template_file):
            LOG.info("MetallbConfig template file found. Configuring metallb ranges in it.")
            use_metallbconfig = True
            utils.backup_file(filepath=metallbconfig_template_file, remote=remote)
        else:
            LOG.info("MetallbConfig template file not found. Configuring metallb ranges "
                     "in cluster template.")

        with remote.open(cluster_template_file, 'r') as r_f:
            cluster_template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
            cluster_template['spec']['providerSpec']['value']['facility'] = \
                cluster_networking['metro']
            cluster_template['spec']['providerSpec']['value']['network'] = \
                copy.deepcopy(cluster_networking['network_config'][
                    'networkSpec'])
            if use_metallbconfig:
                cluster_template['spec']['providerSpec']['value']['network']['metallbRanges'] = []
            regional_spec = cluster_template['spec']['providerSpec']['value'][
                'kaas']['regional']
            ntp_server = cluster_networking['network_config']['networkSpec'][
                'gateway']
            ntp_values = {
                'config': {
                    'lcm': {
                        'ntp': {
                            'servers': [ntp_server]
                        }
                    }
                }
            }
            provider_spec = next(spec for spec in regional_spec if
                                 spec['provider'] == 'equinixmetalv2')
            releases = provider_spec.setdefault('helmReleases', [])
            provider_release = next((release for release in releases if
                                     release['name'] == 'equinix-provider'), {})
            if provider_release == {}:
                releases.append({
                    "name": "equinix-provider",
                    "values": ntp_values
                })
            else:
                provider_release.setdefault('values', {}).setdefault(
                    'config', {}).setdefault('lcm', {}).setdefault(
                    'ntp', {})['servers'] = [ntp_server]
            provider_spec['helmReleases'] = releases

            if proxy:
                cluster_template['spec']['providerSpec']['value']['proxy'] = proxy

            if custom_region:
                cluster_template['spec']['providerSpec']['value']['network']['nameservers'] = ['8.8.8.8']

            if settings.KAAS_EQUINIX_PROJECT_SSH_KEYS:
                cluster_template['spec']['providerSpec'][
                    'value']['projectSSHKeys'] = settings.KAAS_EQUINIX_PROJECT_SSH_KEYS.split(",")

        cluster_yaml = yaml.dump(cluster_template)
        LOG.debug(f"Patched cluster.yaml for equinixmetalv2:\n{cluster_yaml}")
        with remote.open(cluster_template_file, 'w') as r_f:
            r_f.write(cluster_yaml)
        if use_metallbconfig:
            with remote.open(metallbconfig_template_file, 'r') as r_f:
                metallbconfig_template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
                address_pool = {
                    'name': 'default',
                    'spec': {
                        'addresses': cluster_networking['network_config'][
                            'networkSpec']['metallbRanges'],
                        'autoAssign': True,
                        'avoidBuggyIPs': False
                    },
                }
                metallbconfig_template['spec']['ipAddressPools'] = \
                    [address_pool]

            metallbconfig_yaml = yaml.dump(metallbconfig_template)
            LOG.debug(f"Patched metallbconfig.yaml for equinixmetalv2:"
                      f"\n{metallbconfig_yaml}")
            with remote.open(metallbconfig_template_file, 'w') as r_f:
                r_f.write(metallbconfig_yaml)

    # disable stacklight inside custom templates.
    # should not be used is usual scenarios
    # assuming that lma enabled by default
    def disable_lma_logging(self, templates_folder='templates'):
        LOG.info("Disabling logging part of LMA on cluster")
        remote = self.remote_seed()
        bootstrap_templates_path = "{0}/{1}".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            templates_folder)

        res = remote.execute(
            "cd {}; find . -type f -name cluster.yaml.template".format(
                bootstrap_templates_path))

        bootstrap_templates = [
            os.path.join(bootstrap_templates_path,
                         x.decode("utf-8").strip().replace('./', ''))
            for x in res.stdout]

        LOG.info("Gathered templates: {}".format(bootstrap_templates))

        for cluster_template_path in bootstrap_templates:
            LOG.info("Processing template to disable LMA {}".format(
                cluster_template_path))
            utils.backup_file(filepath=cluster_template_path, remote=remote)

            with remote.open(cluster_template_path, 'r') as r_f:
                template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
                for hr in template['spec']['providerSpec'][
                        'value'].get('helmReleases'):
                    if hr['name'] == 'stacklight':
                        if not hr['values'].get('logging'):
                            LOG.warning("No logging control key found "
                                        "in templates. Will add it.")
                            hr['values']['logging'] = dict(enabled=False)
                        hr['values']['logging']['enabled'] = False
            with remote.open(cluster_template_path, 'w') as r_f:
                r_f.write(yaml.dump(template))

    # enable keycloak+ldap inside custom templates
    # assuming iam release does not customized (as in core templates)
    def enable_keycloak_ldap(self, templates_folder='templates'):
        iam_release = {
            "name": "iam",
            "values": {
                "keycloak": {
                    "userFederation": {
                        "providers": [
                            {
                                "displayName": "mirantis-ldap-services",
                                "providerName": "ldap",
                                "priority": 1,
                                "fullSyncPeriod": -1,
                                "changedSyncPeriod": -1,
                                "config": {
                                    "pagination": "true",
                                    "debug": "false",
                                    "searchScope": "1",
                                    "connectionPooling": "true",
                                    "usersDn": "ou=people,ou=services,dc=mirantis,dc=net",
                                    "userObjectClasses": "inetOrgPerson,organizationalPerson",
                                    "usernameLDAPAttribute": "uid",
                                    "rdnLDAPAttribute": "uid",
                                    "vendor": "ad",
                                    "editMode": "READ_ONLY",
                                    "uuidLDAPAttribute": "uid",
                                    "connectionUrl": "ldap://ldap-bud.bud.mirantis.net",
                                    "syncRegistrations": "false"
                                }
                            },
                            {
                                "displayName": "mirantis-ldap-users",
                                "providerName": "ldap",
                                "priority": 1,
                                "fullSyncPeriod": -1,
                                "changedSyncPeriod": -1,
                                "config": {
                                    "pagination": "true",
                                    "debug": "false",
                                    "searchScope": "1",
                                    "connectionPooling": "true",
                                    "usersDn": "ou=People,o=mirantis,dc=mirantis,dc=net",
                                    "userObjectClasses": "inetOrgPerson,organizationalPerson",
                                    "usernameLDAPAttribute": "uid",
                                    "rdnLDAPAttribute": "uid",
                                    "vendor": "ad",
                                    "editMode": "READ_ONLY",
                                    "uuidLDAPAttribute": "uid",
                                    "connectionUrl": "ldap://ldap-bud.bud.mirantis.net",
                                    "syncRegistrations": "false"
                                }
                            }
                        ],
                        "mappers": [
                            {
                                "name": "username",
                                "federationMapperType": "user-attribute-ldap-mapper",
                                "federationProviderDisplayName": "mirantis-ldap",
                                "config": {
                                    "ldap.attribute": "uid",
                                    "user.model.attribute": "username",
                                    "is.mandatory.in.ldap": "true",
                                    "read.only": "true",
                                    "always.read.value.from.ldap": "false"
                                }
                            },
                            {
                                "name": "full name",
                                "federationMapperType": "full-name-ldap-mapper",
                                "federationProviderDisplayName": "mirantis-ldap",
                                "config": {
                                    "ldap.full.name.attribute": "cn",
                                    "read.only": "true",
                                    "write.only": "false"
                                }
                            },
                            {
                                "name": "last name",
                                "federationMapperType": "user-attribute-ldap-mapper",
                                "federationProviderDisplayName": "mirantis-ldap",
                                "config": {
                                    "ldap.attribute": "sn",
                                    "user.model.attribute": "lastName",
                                    "is.mandatory.in.ldap": "true",
                                    "read.only": "true",
                                    "always.read.value.from.ldap": "true"
                                }
                            },
                            {
                                "name": "email",
                                "federationMapperType": "user-attribute-ldap-mapper",
                                "federationProviderDisplayName": "mirantis-ldap",
                                "config": {
                                    "ldap.attribute": "mail",
                                    "user.model.attribute": "email",
                                    "is.mandatory.in.ldap": "false",
                                    "read.only": "true",
                                    "always.read.value.from.ldap": "true"
                                }
                            }
                        ]
                    }
                }
            }
        }

        if settings.KEYCLOAK_LDAP_BIND_USERNAME and settings.KEYCLOAK_LDAP_BIND_PASSWORD:
            auth_block = {
                "authType": "simple",
                "bindDn": f"uid={settings.KEYCLOAK_LDAP_BIND_USERNAME},ou=people,ou=services,dc=mirantis,dc=net",
                "bindCredential": f"{settings.KEYCLOAK_LDAP_BIND_PASSWORD}",
            }
        else:
            auth_block = {
                "authType": "none"
            }

        for provider in iam_release['values']['keycloak']['userFederation']['providers']:
            utils.merge_dicts(provider['config'], auth_block)

        LOG.info("Adding LDAP integration to Keycloak via cluster templates")
        remote = self.remote_seed()
        bootstrap_templates_path = "{0}/{1}".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            templates_folder)

        res = remote.execute(
            "cd {}; find . -type f -name cluster.yaml.template".format(
                bootstrap_templates_path))

        bootstrap_templates = [
            os.path.join(bootstrap_templates_path,
                         x.decode("utf-8").strip().replace('./', ''))
            for x in res.stdout]

        LOG.info("Gathered templates: {}".format(bootstrap_templates))

        for cluster_template_path in bootstrap_templates:
            LOG.info("Processing template to enable LDAP {}".format(
                cluster_template_path))
            utils.backup_file(filepath=cluster_template_path, remote=remote)

            with remote.open(cluster_template_path, 'r') as r_f:
                template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
                mgm = template['spec']['providerSpec']['value']['kaas']['management']
                if not mgm.get('helmReleases'):
                    mgm['helmReleases'] = []
                mgm['helmReleases'].append(iam_release)
                LOG.info('Adding iam configuration to management helm releases')
                LOG.info(mgm)
            with remote.open(cluster_template_path, 'w') as r_f:
                r_f.write(yaml.dump(template))

    def set_ntp_servers(self, templates_folder='templates'):

        ntp_config = {
            "values": {
                "config": {
                    "lcm": {
                        "ntp": {
                            "servers": [
                            ]
                        }
                    }
                }
            }
        }

        ntp_servers = [i.strip() for i in settings.CORE_KAAS_NTP_LIST.split(',')]
        ntp_config['values']['config']['lcm']['ntp']['servers'] += ntp_servers

        vsphere_helmrelease = {
            "name": "vsphere-provider",
        }

        os_helmrelease = {
            "name": "openstack-provider",
        }

        utils.merge_dicts(vsphere_helmrelease, ntp_config)
        utils.merge_dicts(os_helmrelease, ntp_config)

        LOG.info("Adding NTP servers to cluster templates")
        remote = self.remote_seed()
        bootstrap_templates_path = "{0}/{1}".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            templates_folder)

        res = remote.execute(
            "cd {}; find . -type f -name cluster.yaml.template".format(
                bootstrap_templates_path))

        bootstrap_templates = [
            os.path.join(bootstrap_templates_path,
                         x.decode("utf-8").strip().replace('./', ''))
            for x in res.stdout]

        LOG.info("Gathered templates: {}".format(bootstrap_templates))

        for cluster_template_path in bootstrap_templates:
            LOG.info("Processing template  {}".format(
                cluster_template_path))
            utils.backup_file(filepath=cluster_template_path, remote=remote)

            with remote.open(cluster_template_path, 'r') as r_f:
                template = yaml.load(r_f.read(), Loader=yaml.SafeLoader)
                kaas = template['spec']['providerSpec']['value']['kaas']
                kaas['ntpEnabled'] = True
                for p in kaas['regional']:
                    # Filter only vSphere and OpenStack providers
                    if p['provider'] in ('openstack', 'vsphere'):
                        # if not helmreleases found - just append
                        if not p.get('helmReleases'):
                            p['helmReleases'] = []
                            if p['provider'] == 'openstack':
                                p['helmReleases'].append(os_helmrelease)
                            if p['provider'] == 'vsphere':
                                p['helmReleases'].append(vsphere_helmrelease)
                        else:
                            # Next code will not make changes if no target helmreleases found
                            for h_release in p.get('helmReleases'):
                                if h_release['name'] == 'openstack-provider' and p['provider'] == 'openstack':
                                    utils.merge_dicts(h_release, os_helmrelease)
                                if h_release['name'] == 'vsphere-provider' and p['provider'] == 'vsphere':
                                    utils.merge_dicts(h_release, vsphere_helmrelease)
                            # Get HR names
                            h_releases_names = [hr['name'] for hr in p.get('helmReleases')]
                            # Append helm releases only in case if no config for them found
                            if p['provider'] == 'openstack' and 'openstack-provider' not in h_releases_names:
                                p['helmReleases'].append(os_helmrelease)
                            if p['provider'] == 'vsphere' and 'vsphere-provider' not in h_releases_names:
                                p['helmReleases'].append(vsphere_helmrelease)
                LOG.info('Result:')
                LOG.info(kaas)
            with remote.open(cluster_template_path, 'w') as r_f:
                r_f.write(yaml.dump(template))

    def bm_determine_env_config_name(self) -> str:
        if settings.ENV_CONFIG_NAME:
            env_config_name = settings.ENV_CONFIG_NAME
        elif settings.BM_HALF_VIRTUAL_ENV:
            env_config_name = 'half-virtual'
        elif settings.KAAS_BM_CI_ON_EQUINIX:
            env_config_name = 'half-virtual-equinix'
        else:
            env_config_name = 'virtual'

        return env_config_name

    def bm_determine_ansible_inventory(self) -> list:
        env_config_name = self.bm_determine_env_config_name()
        env_config_dir = f"labs/{env_config_name}"
        ansible_inventory = [f"{env_config_dir}/inventory.yaml"]

        if settings.BM_DEPLOY_CHILD:
            # Deploy 6 more VMs for child cluster
            ansible_inventory.append(f"{env_config_dir}/child/inventory.yaml")

        return ansible_inventory

    @utils.log_method_time()
    def step_004_prepare_for_kaas_bm_cluster(self):
        remote = self.remote_seed()
        _sfile = '/tmp/step_004_prepare_for_kaas_bm_cluster'

        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage step_004_prepare_for_kaas_bm_cluster,'
                        f'since {_sfile} exists ')
            return

        ansible_inventory = self.bm_determine_ansible_inventory()
        kaas_version = self.get_kaas_version()
        ansible_vars = {
            'kaas_version': kaas_version
        }

        LOG.info('Base setup, render and create VMs using virsh')
        self.ansible_run(remote, "./setup-playbook.yml",
                         inventory=ansible_inventory,
                         extra_vars=ansible_vars,
                         tags=['prepare'])
        remote.reconnect()
        remote.check_call(f'touch {_sfile}')

    def add_credentials_data_to_si_config(self) -> None:
        """Add Keycloak users/passwords into si-config.yaml"""

        if settings.KEYCLOAK_USERS_GENERATE_PASSWORD:
            writer_password = utils.gen_random_password(30)
            reader_password = utils.gen_random_password(30)
            operator_password = utils.gen_random_password(30)
            stacklight_password = utils.gen_random_password(30)
        else:
            writer_password = 'password'
            reader_password = 'password'
            operator_password = 'password'
            stacklight_password = 'password'
        # Always generate serviceuser password
        serviceuser_password = utils.gen_random_password(30)

        keycloak_users = {
            'writer': writer_password,
            'reader': reader_password,
            'operator': operator_password,
            'stacklight': stacklight_password,
            'serviceuser': serviceuser_password,
        }

        assert settings.SI_CONFIG_PATH and os.path.isfile(settings.SI_CONFIG_PATH), (
            f"Environment variable 'SI_CONFIG_PATH' contains wrong path to YAML file: {settings.SI_CONFIG_PATH}")
        with templates.YamlEditor(settings.SI_CONFIG_PATH) as editor:
            current_content = editor.content
            current_content['keycloak_users'] = keycloak_users
            editor.content = current_content

        with open(f'{settings.ARTIFACTS_DIR}/keycloak_users', mode='w') as f:
            f.write(yaml.dump(keycloak_users))

        # Set environment variables to use them in Jinja2 templates generation later
        os.environ['KAAS_MANAGEMENT_SVC_USERNAME'] = 'serviceuser'
        os.environ['KAAS_MANAGEMENT_SVC_PASSWORD'] = serviceuser_password

    @utils.log_method_time()
    def step_004_prepare_bm_on_seed(self):
        """
        For virt envs only
        Run final ansible preparation on seed node,
        just before run bootstrap.sh
        """

        remote = self.remote_seed()
        _sfile = '/tmp/step_004_prepare_bm_on_seed'
        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage step_004_prepare_bm_on_seed,'
                        f'since {_sfile} exists ')
            return
        kaas_version = self.get_kaas_version()
        kaas_binary_version = self.get_kaas_binary_version()
        ansible_vars = {
            'kaas_version': kaas_version,
            'kaas_binary_version': kaas_binary_version,
            'mcc_bootstrap_version': settings.MCC_BOOTSTRAP_VERSION
        }

        ansible_inventory = self.bm_determine_ansible_inventory()
        playbook = ""
        try:
            playbook = "./setup-playbook.yml"
            self.ansible_run(
                remote, playbook,
                inventory=ansible_inventory,
                extra_vars=ansible_vars,
                tags=['configure'],
                timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
            remote.reconnect()

            self.prepare_bm_seed_node_templates(self.bm_determine_env_config_name())

            playbook = "./setup-kaas-bm-pbook.yml"
            self.ansible_run(
                remote, playbook,
                inventory=ansible_inventory,
                extra_vars=ansible_vars,
                tags=['deploy'],
                timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
            remote.reconnect()
        except Exception as e:
            LOG.error(f"Ansible playbook '{playbook}' "
                      f"(inventory: {ansible_inventory}) failed!\n{e}")
            raise e
        remote.check_call(f'touch {_sfile}')

    # It also can deploy regional cluster
    @utils.log_method_time()
    def step_004_deploy_kaas_bm_mgmt_cluster(self):
        """
        Exactly run bootstrap.sh for -bm case|s
        FIXME(alexz) First implementation, contain duplicates with
        bootstrap.env - will be fixed during bootstrapv2 impl.
        """
        remote = self.remote_seed()
        _sfile = '/tmp/step_004_deploy_kaas_bm_mgmt_cluster'
        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage step_004_deploy_kaas_bm_mgmt_cluster,'
                        f'since {_sfile} exists ')
            return
        bs_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR
        prefix = 'management'

        LOG.info(f"Bootstrap KaaS-BM {prefix} cluster")
        run_envs = {
            "KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL
        }
        if settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT:
            run_envs["KAAS_BOOTSTRAP_INFINITE_TIMEOUT"] = True
        if settings.CUSTOM_HOSTNAMES and settings.MCC_BOOTSTRAP_VERSION != "v2":
            run_envs["CUSTOM_HOSTNAMES"] = True

        envs_string = utils.make_export_env_strting(run_envs)
        deploy_result = False
        try:
            if settings.MCC_BOOTSTRAP_VERSION != "v2":
                remote.check_call(
                    f"cd {bs_dir}; "
                    f"bash -o pipefail -xc '{envs_string}; ./bootstrap.sh all |"
                    f"tee -a ./logs/fastlog.txt'",
                    verbose=True,
                    timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
            else:
                region = self.get_region_from_bootstrap_region(settings.BAREMETAL_PROVIDER_NAME)
                deploy_result = self.run_bootstrap_v2(settings.BAREMETAL_PROVIDER_NAME, region)
        except Exception as e:
            LOG.error("Bootstrap failed: {}".format(e))
            raise e
        finally:
            try:
                self.step_bs_collect_logs(remote)
                self.check_multiple_kaasreleases(remote)
                # Download management_kubeconfig artifact
                kubeconfig_path = f"./{bs_dir}/kubeconfig"
                if remote.isfile(kubeconfig_path):
                    remote.download(kubeconfig_path,
                                    "{0}/{1}_kubeconfig_orig"
                                    .format(settings.ARTIFACTS_DIR, prefix))
                    LOG.info(f"{prefix} kubeconfig has been downloaded")

                    with open("{0}/{1}_kubeconfig_orig".format(
                            settings.ARTIFACTS_DIR, prefix), 'r') as f:
                        kubeconfig = yaml.load(f.read(),
                                               Loader=yaml.SafeLoader)
                    with open("{0}/{1}_kubeconfig".format(
                            settings.ARTIFACTS_DIR, prefix), "w") as f:
                        f.write(yaml.dump(kubeconfig))
                # Download test-0-bmh.yaml generated just for test
                # in case of DEPLOY_CHILD=false
                if remote.isfile('bootstrap/test/test-0-bmh.yaml'):
                    remote.download("bootstrap/test/test-0-bmh.yaml",
                                    "{0}/test-0-bmh.yaml"
                                    .format(settings.ARTIFACTS_DIR))
                else:
                    LOG.warning("Test bmh file was not found")

                if settings.MCC_BOOTSTRAP_VERSION == "v2" and (deploy_result or not settings.KEEP_ENV_AFTER):
                    self.delete_bootstrapv2_cluster()
            except Exception as e:
                LOG.error(
                    "Unable to collect bootstrap logs: {}".format(e))
                raise e

        LOG.info(f"{prefix} cluster has been deployed successfully")
        remote.check_call(f'touch {_sfile}')

        if settings.MCC_BOOTSTRAP_VERSION == "v2" and deploy_result:
            LOG.info("Cleanup bootstrap cluster")
            self.delete_bootstrapv2_cluster()

    # Prepare ansible for physical BM deploy
    def step_004_prepare_bm_physical_on_seed(self, lab_id):
        remote = self.remote_seed()
        env_config_dir = f"labs/{lab_id}"
        kaas_version = self.get_kaas_version()
        ansible_vars = {
            'kaas_binary_version': kaas_version,
            'mcc_bootstrap_version': settings.MCC_BOOTSTRAP_VERSION
        }

        LOG.info('Ansible base')
        # We need to get ipmitools and networking here
        playbook = f"setup-kaas-bm-{lab_id}.yml"
        ansible_inventory = ["inventory.yaml",
                             f"{env_config_dir}/inventory.yaml"]
        try:
            self.ansible_run(
                remote, playbook,
                inventory=ansible_inventory,
                extra_vars=ansible_vars,
                tags=['prepare', 'configure'],
                timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
            remote.reconnect()

            self.prepare_bm_seed_node_templates(lab_id)

            self.ansible_run(
                remote, playbook,
                inventory=ansible_inventory,
                extra_vars=ansible_vars,
                tags=['deploy'],
                timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
            remote.reconnect()
        except Exception as e:
            LOG.error(f"Ansible playbook '{playbook}' "
                      f"(inventory: {ansible_inventory}) failed!\n{e}")
            raise e
        remote.reconnect()

    # It also can deploy regional cluster
    @utils.log_method_time()
    def step_004_deploy_kaas_bm_cluster_physical(self, lab_id):
        if settings.MCC_BOOTSTRAP_VERSION != "v2":
            # Prepare ansible. For BV2 it is already done during seed node prepare
            self.step_004_prepare_bm_physical_on_seed(lab_id)

        bs_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR
        remote = self.remote_seed()
        prefix = 'management'

        if remote.isfile(f'{bs_dir}/container-cloud') and settings.KEEP_ENV_BEFORE:
            LOG.warning('Power off HW skipped')
        else:
            if lab_id in settings.KAAS_BM_EXTRA_BMC_ENVS:
                # cold reset take from 3-10 min to restore, but we dont care
                # since next interaction with bmc will be on bootstrap.sh time -
                # far away from current phase.
                acts = ['chassis power off', 'bmc reset cold']
                offsleep = 10
                for act in acts:
                    ret = remote.check_call(
                        f" source kaas-bm-env/labs/{lab_id}/patch.sh ; "
                        f" set_power '{act}' all",
                        verbose=True, raise_on_err=False)
                    if ret.exit_code != 0:
                        LOG.warning(f'Action:"{act}" HW failed!')
                    if "off" in act:
                        LOG.warning(f"Sleep  for {offsleep} secs.")
                        sleep(offsleep)
            else:
                ret = remote.check_call(
                    f" source kaas-bm-env/labs/{lab_id}/patch.sh ; "
                    f" set_power off all",
                    verbose=True, raise_on_err=False)
                if ret.exit_code != 0:
                    LOG.warning('Power off HW failed!')

        LOG.info(f"Bootstrap KaaS-BM {prefix} cluster")
        run_envs = {
            "KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL
        }
        if settings.MCC_BOOTSTRAP_VERSION != "v2":
            if settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT:
                run_envs["KAAS_BOOTSTRAP_INFINITE_TIMEOUT"] = True
            if settings.CUSTOM_HOSTNAMES:
                run_envs["CUSTOM_HOSTNAMES"] = True

        envs_string = utils.make_export_env_strting(run_envs)
        deploy_result = False
        try:
            if settings.MCC_BOOTSTRAP_VERSION != "v2":
                remote.check_call(
                    f"cd {bs_dir}; "
                    f"bash -o pipefail -xc '{envs_string}; ./bootstrap.sh all |"
                    f"tee -a ./logs/fastlog.txt'",
                    verbose=True,
                    timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
            else:
                region = self.get_region_from_bootstrap_region(settings.BAREMETAL_PROVIDER_NAME)
                deploy_result = self.run_bootstrap_v2(settings.BAREMETAL_PROVIDER_NAME, region)
        except Exception as e:
            LOG.error("Bootstrap failed: {}".format(e))
            raise e
        finally:
            try:
                self.step_bs_collect_logs(remote)
                self.check_multiple_kaasreleases(remote)
                kubeconfig_path = f'{bs_dir}/kubeconfig'
                if remote.isfile(kubeconfig_path):
                    remote.download(kubeconfig_path,
                                    f"{settings.ARTIFACTS_DIR}"
                                    f"/{prefix}_kubeconfig")
                    LOG.info(f"{prefix} kubeconfig has been downloaded")
            except Exception as e:
                LOG.error(
                    "Unable to collect bootstrap logs: {}".format(e))
                raise e

        LOG.info(f"{prefix} cluster has been deployed successfully")

        if settings.MCC_BOOTSTRAP_VERSION == "v2" and deploy_result:
            LOG.info("Cleanup bootstrap cluster")
            self.delete_bootstrapv2_cluster()

    @utils.log_method_time()
    def step_004_deploy_kaas_cluster(self, extra_config=None, endpoints=None, seed_instance=None):
        """Run bootstrap cluster, download logs"""
        extra_config = extra_config or {}

        LOG.info("Verify that seed node have been prepared")
        remote = self.remote_seed()

        LOG.info("Deploy KaaS cluster")
        run_envs = {
            "CLUSTER_NAME": settings.CLUSTER_NAME,
            "KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL
        }
        kaas_release_filename = self.get_kaas_release_file_name()

        if settings.CUSTOM_HOSTNAMES and settings.MCC_BOOTSTRAP_VERSION != "v2":
            run_envs["CUSTOM_HOSTNAMES"] = settings.CUSTOM_HOSTNAMES
        if settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT:
            run_envs["KAAS_BOOTSTRAP_INFINITE_TIMEOUT"] = True

        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
            if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
                run_envs['PROXY_CA_CERTIFICATE_PATH'] = settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE
            run_envs['HTTP_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['HTTPS_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            LOG.info("HTTP_PROXY and HTTPS_PROXY is set to {0}".format(
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            ))
            # PRODX-13381: add EU/US fip range to no_proxy
            os_ranges = self.get_noproxy_fips(remote)
            if endpoints:
                os_ranges = f"{os_ranges},{endpoints}".strip(',')
            if settings.KAAS_PROXYOBJECT_NO_PROXY:
                run_envs['NO_PROXY'] = \
                    f"{settings.KAAS_PROXYOBJECT_NO_PROXY},{os_ranges}".strip(',')
            else:
                run_envs['NO_PROXY'] = os_ranges
            if settings.KAAS_AWS_ENABLED:
                run_envs['NO_PROXY'] += f",{settings.KAAS_PROXYOBJECT_NO_PROXY_AWS},"\
                                        f"ec2.{settings.AWS_DEFAULT_REGION}.amazonaws.com"\
                                        f",elb.{settings.AWS_DEFAULT_REGION}.amazonaws.com"\
                                        f",elasticloadbalancing.{settings.AWS_DEFAULT_REGION}.amazonaws.com"
                run_envs['NO_PROXY'] = run_envs['NO_PROXY'].strip(',')
            run_envs['no_proxy'] = run_envs['NO_PROXY']
            LOG.info("NO_PROXY is set to {0}".format(
                run_envs['NO_PROXY']))

        if settings.KAAS_RELEASES_FOLDER and \
                os.path.isdir(settings.KAAS_RELEASES_FOLDER) and \
                kaas_release_filename and not \
                settings.BOOTSTRAP_WITH_DEFAULTS:
            LOG.info('Deploy with custom releases')
            run_envs['KAAS_RELEASE_YAML'] = '{}/kaas/{}'.format(
                settings.KAAS_RELEASES_REMOTE_FOLDER,
                kaas_release_filename.strip())
            run_envs['CLUSTER_RELEASES_DIR'] = '{}/cluster'.format(
                settings.KAAS_RELEASES_REMOTE_FOLDER)

        provider = settings.OPENSTACK_PROVIDER_NAME
        if settings.KAAS_AWS_ENABLED:
            # For AWS provider
            LOG.info('Enable AWS provider')
            provider = settings.AWS_PROVIDER_NAME
            # NOTE: these variables are required for CC binary to generate CF
            run_envs['AWS_SECRET_ACCESS_KEY'] = settings.AWS_SECRET_ACCESS_KEY
            run_envs['AWS_ACCESS_KEY_ID'] = settings.AWS_ACCESS_KEY_ID
            run_envs['AWS_DEFAULT_REGION'] = settings.AWS_DEFAULT_REGION
            if settings.MCC_BOOTSTRAP_VERSION != "v2":
                run_envs['KAAS_AWS_ENABLED'] = True
        elif settings.KAAS_VSPHERE_ENABLED:
            # For Vsphere provider
            LOG.info('Enable Vsphere provider')
            provider = settings.VSPHERE_PROVIDER_NAME
            if settings.MCC_BOOTSTRAP_VERSION != "v2":
                run_envs['KAAS_VSPHERE_ENABLED'] = True
        elif settings.KAAS_EQUINIX_ENABLED:
            if settings.MCC_BOOTSTRAP_VERSION == "v2":
                raise UnknownProviderException(
                    f"Provider {Provider.equinixmetal.provider_name} not supported in bootstrap v2")
            # For Equinix provider
            LOG.info('Enable Equinix provider')
            run_envs['KAAS_EQUINIX_ENABLED'] = True
        elif settings.KAAS_EQUINIXMETALV2_ENABLED:
            LOG.info('Enable EquinixmetalV2 provider')
            provider = settings.EQUINIXMETALV2_PROVIDER_NAME
            if settings.MCC_BOOTSTRAP_VERSION == "v1":
                run_envs['KAAS_EQUINIXMETALV2_ENABLED'] = True
                run_envs['KAAS_BM_PXE_BRIDGE'] = \
                    settings.KAAS_EQUINIX_PRIVATE_PXE_BRIDGE

                # getting mgmt network config for bootstrap
                mgmt_key = "{}/{}".format(settings.CLUSTER_NAMESPACE,
                                          settings.CLUSTER_NAME)
                mgmt_config = extra_config.get("network_config", {}).get(mgmt_key, {})
                assert extra_config, (
                    "'extra_config' don't have network config for equinixmetalv2 management cluster deploy")
                run_envs['KAAS_BM_PXE_IP'] = mgmt_config['bootstrap']['bm_pxe_ip']
                run_envs['KAAS_BM_PXE_MASK'] = mgmt_config['bootstrap']['bm_pxe_mask']
                # EQUINIXMETALV2 needs proxy for bootstrap
                proxy_url = "http://{}:{}".format(mgmt_config['networkSpec']['gateway'],
                                                  settings.KAAS_EQUINIX_PRIVATE_PROXY_PORT)
                run_envs['HTTP_PROXY'] = proxy_url
                run_envs['HTTPS_PROXY'] = proxy_url
                run_envs['http_proxy'] = proxy_url
                run_envs['https_proxy'] = proxy_url
                LOG.info(f"HTTP_PROXY and HTTPS_PROXY is set to {proxy_url}")

        elif settings.KAAS_AZURE_ENABLED:
            # For Azure provider
            LOG.info('Enable Azure provider')
            provider = settings.AZURE_PROVIDER_NAME
            if settings.MCC_BOOTSTRAP_VERSION != "v2":
                run_envs['KAAS_AZURE_ENABLED'] = True
        else:
            if settings.MCC_BOOTSTRAP_VERSION != "v2":
                assert remote.isfile(f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/clouds.yaml"), (
                    f"OpenStack cloud config {settings.KAAS_BOOTSTRAP_TARGET_DIR}/clouds.yaml not found on seed node")
        assert remote.isfile(f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/container-cloud"), (
            f"MCC binary {settings.KAAS_BOOTSTRAP_TARGET_DIR}/container-cloud not found on seed node")
        if not settings.BOOTSTRAP_WITH_DEFAULTS:
            bootstrap_version = self.get_bootstrap_version()
        else:
            bootstrap_version = self.get_remote_default_kaas_data(
                remote)['spec']['bootstrap']['version']
        remote.check_call('{0}/container-cloud version 2>&1 | grep {1}'.format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR,
            bootstrap_version), verbose=True)
        LOG.info('Deploy with vanilla releases')
        envs_string = utils.make_export_env_strting(run_envs)
        LOG.info('Current envs_string {}'.format(envs_string))
        if settings.KAAS_AWS_ENABLED:
            # Check cloudformation created or not existed
            # to avoid conflict when stack is in CREATE_IN_PROGRESS state
            cloudformation_stack_name = settings.AWS_CLOUDFORMATION_STACK_NAME
            aws_client = AwsManager(
                aws_access_key_id=settings.AWS_ACCESS_KEY_ID,
                aws_secret_access_key=settings.AWS_SECRET_ACCESS_KEY,
                region_name=settings.AWS_DEFAULT_REGION)

            stack = aws_client.get_cloudformation_stack(
                cloudformation_stack_name)
            if not stack:
                LOG.info("Cloudformation stack is not existed. "
                         "Will proceed with creation")
            else:
                stack_status = stack.get('StackStatus')
                if stack_status == 'CREATE_COMPLETE':
                    LOG.info("Stack is created. Will proceed with updating")
                elif stack_status == 'CREATE_IN_PROGRESS':
                    LOG.info("Cloudformation stack is CREATE_IN_PROGRESS "
                             "status. Waiting for CREATE_COMPLETE")
                    waiters.wait(lambda: aws_client.get_cloudformation_stack(
                        cloudformation_stack_name).get(
                            'StackStatus') == 'CREATE_COMPLETE', interval=30, timeout=300)
                elif stack_status == 'DELETE_IN_PROGRESS':
                    LOG.info("Cloudformation stack is DELETE_IN_PROGRESS "
                             "status. Waiting for deletion")
                    waiters.wait(
                        lambda: not aws_client.get_cloudformation_stack(
                            cloudformation_stack_name), interval=30, timeout=300)
                    LOG.info("Stack is deleted")
                else:
                    LOG.warning("Stack {} is existed and has incorrect "
                                "status. Current status: {}. Will try "
                                "to delete".format(cloudformation_stack_name,
                                                   stack_status))
                    aws_client.aws_cloudformation_client.delete_stack(
                        StackName=cloudformation_stack_name)
                    waiters.wait(
                        lambda: not aws_client.get_cloudformation_stack(
                            cloudformation_stack_name), interval=30, timeout=300)
                    LOG.info("Stack is deleted")

            ret = remote.execute(
                "{envs_string}; "
                "./{bootstrap}/container-cloud bootstrap aws policy".format(
                    envs_string=envs_string,
                    bootstrap=settings.KAAS_BOOTSTRAP_TARGET_DIR),
                verbose=True)
            bootstrapper_users = [user.get('UserName', '') for user
                                  in aws_client.get_users()
                                  if 'bootstrapper'
                                  in user.get('UserName', '')]
            assert len(bootstrapper_users) > 0, ("Bootstrapper user was "
                                                 "not created")
            try:
                LOG.info("Trying to create new credentials for "
                         "user {}".format(bootstrapper_users[0]))
                response = aws_client.aws_iam_client.create_access_key(
                    UserName=bootstrapper_users[0]
                )
                access_key_id = response['AccessKey'].get('AccessKeyId')
                secret_access_key = response['AccessKey'].get('SecretAccessKey')
                run_envs['AWS_SECRET_ACCESS_KEY'] = secret_access_key
                run_envs['AWS_ACCESS_KEY_ID'] = access_key_id
                LOG.info("Credentials created successfuly: {}".format(
                    access_key_id))
                envs_string = utils.make_export_env_strting(run_envs)
                LOG.info('Updated envs_string with new AWS creds: {}'
                         .format(envs_string))
            except aws_client.aws_iam_client.exceptions.LimitExceededException\
                    as e:
                LOG.warning("Limit Exceeded: {}".format(e))
                LOG.info(
                    "Admin credentials will be used")

        deploy_result = False
        try:
            if settings.MCC_BOOTSTRAP_VERSION == "v2":
                deploy_result = self.run_bootstrap_v2(
                    provider=provider,
                    region=self.get_region_from_bootstrap_region(provider=provider)
                )
            else:
                ret = remote.execute(
                    "{envs_string}; bash -x ./{bootstrap}/bootstrap.sh all".format(
                        envs_string=envs_string,
                        bootstrap=settings.KAAS_BOOTSTRAP_TARGET_DIR),
                    verbose=True,
                    timeout=settings.KAAS_BOOTSTRAP_TIMEOUT)
                deploy_result = ret.exit_code == 0
        except ExecHelperTimeoutError as e:
            LOG.error("Timeout for bootstrap exceeded: {}".format(e))
            raise e
        finally:
            try:
                # Download management_kubeconfig artifact
                local_kubeconfig_path = f"{settings.ARTIFACTS_DIR}/management_kubeconfig"
                kubeconfig_path = "kubeconfig"
                if settings.KAAS_BM_ENABLED:
                    kubeconfig_path = f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/kubeconfig"

                if remote.isfile(kubeconfig_path):
                    remote.download(kubeconfig_path, local_kubeconfig_path)
                    LOG.info("Management kubeconfig has been downloaded")

                # Collect BV2 logs only
                self.step_bs_collect_bv2_logs(remote)
                if settings.MCC_BOOTSTRAP_VERSION == "v2" and (deploy_result or not settings.KEEP_ENV_AFTER):
                    self.delete_bootstrapv2_cluster()
                # Collect Management cluster logs only, after removing BV2 cluster
                self.step_bs_collect_logs(remote)

                self.check_multiple_kaasreleases(remote)

                mgmt_cluster_uid_path = ".bootstrap_cluster_uid"
                cluster_uid_path = os.path.join(settings.ARTIFACTS_DIR, 'management_cluster_uid')
                if os.path.isfile(local_kubeconfig_path):
                    cluster_name_path = "{0}/management_cluster_fullname".format(
                        settings.ARTIFACTS_DIR)

                    LOG.info("Try to fetch mgmt cluster UID from kubeconfig")
                    kaas_manager = Manager(kubeconfig=local_kubeconfig_path)
                    ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
                    cluster = ns.get_cluster(settings.CLUSTER_NAME)

                    kaas_uid_meta = cluster.get_cluster_annotation(
                        "kaas.mirantis.com/uid")
                    cluster_uid = kaas_uid_meta["kaas.mirantis.com/uid"]
                    LOG.info(f"Cluster deployed with UID - {cluster_uid}")

                    with open(cluster_name_path, 'w') as fp:
                        fp.write(f"{settings.CLUSTER_NAMESPACE}/{settings.CLUSTER_NAME}")
                    with open(cluster_uid_path, 'w') as fp:
                        fp.write(str(cluster_uid))

                    if seed_instance and not settings.SEED_STANDALONE_EXTERNAL_IP and settings.CHAIN_SEED_WITH_MCC:
                        LOG.info('Set KaaS UID to seed instance object')
                        seed_instance.manager.set_meta_item(
                            seed_instance, 'KaaS', cluster_uid)
                        seed_instance.manager.set_meta_item(
                            seed_instance, 'ENV_NAME', settings.ENV_NAME)
                elif remote.isfile(mgmt_cluster_uid_path):
                    remote.download(mgmt_cluster_uid_path, cluster_uid_path)
                    LOG.info(f"Seed '{mgmt_cluster_uid_path}' file has been downloaded into '{cluster_uid_path}'")
            except Exception as e:
                LOG.warning(
                    "Unable to collect logs and/or cluster meta: {}".format(e))

        if deploy_result:
            LOG.info("Mgmt cluster has been deployed successfully")
        else:
            LOG.error("Mgmt cluster has not been deployed successfully")
            # In case of bootstrap fail before pivoting phase, KinD cluster may contain useful logs
            try:
                self.step_collect_logs(remote,
                                       cluster_name=settings.CLUSTER_NAME,
                                       cluster_namespace=settings.CLUSTER_NAMESPACE,
                                       kubeconfig_path=settings.KUBECONFIG_KIND_PATH,
                                       cluster_type="bootstrap")
            except Exception as e:
                LOG.warning("Unable to collect KinD logs: {}".format(e))
            if settings.MCC_BOOTSTRAP_VERSION == "v2":
                raise AssertionError("BootstrapV2 Management cluster failed to deploy")
            raise CalledProcessError(result=ret, expected=(0,))

    def create_keycloak_user_cli(
            self, username, password, role, management_kubeconfig_path, retry=False):
        remote = self.remote_seed()

        class ExecResult(object):
            def __init__(self, exec_result=None):
                self.exec_result = exec_result

        def create_user(ret):
            ret.exec_result = remote.execute(
                f"./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/container-cloud "
                f"bootstrap user add --username {username} "
                f"--roles {role} "
                f"--kubeconfig {management_kubeconfig_path} "
                f"--password-stdin "
                f"-v 4",
                stdin=f"{password}\n",
                verbose=True,
                timeout=30)
            if ret.exec_result.exit_code == 0:
                LOG.info(f"User '{username}' created successfully")
                return ret
            else:
                LOG.error(f"Failed to create the user '{username}': "
                          f"{ret.exec_result}")

        ret = ExecResult()
        if retry:
            waiters.wait(
                lambda: create_user(ret), timeout=150, interval=30,
                timeout_msg=f"Timeout waiting for user creation: {username}")
        else:
            create_user(ret)
        if ret.exec_result.exit_code != 0:
            raise CalledProcessError(result=ret.exec_result, expected=(0,))

    def create_keycloak_user(self, kaas_manager, username, roles, email=None, extra_role=None):
        keycloak_admin_client = self.get_keycloak_admin_client(kaas_manager)
        password = kaas_manager.si_config.get_keycloak_user_password(username)
        user_id = keycloak_admin_client.user_create(username=username, password=password)
        user_name = username + "-" + user_id.split("-")[0]

        def _user_created(user_name):
            if kaas_manager.get_iamusers(name_prefix=user_name, request_timeout=30):
                LOG.info(f"User {user_name} created successfully")
                return True
            else:
                LOG.info(f"User {user_name} has not been created yet. Waiting...")
                return False

        waiters.wait(
            lambda: _user_created(user_name=user_name), timeout=600, interval=20,
            timeout_msg=f"Timeout waiting for user: {user_name}")

        LOG.info(f"Create iamglobalrolebinding for user '{user_name}'")
        for role in roles:
            rnd_string = utils.gen_random_string(6)
            kaas_manager.create_iamglobalrolebinding(f'{user_name}-{rnd_string}', role, user_name)
        if email:
            LOG.info('Set email for user')
            keycloak_admin_client.user_update(username=username,
                                              body={'email': f'{username}@kaas.local'})
        if extra_role:
            LOG.info(f'Add role {extra_role} to user {username}')
            keycloak_admin_client.assign_roles(username=username, roles=extra_role)

    @utils.log_method_time()
    def step_004_create_default_mcc_users(self, custom_bootstrap_path=None):
        kubeconfig_path = "{0}/management_kubeconfig" \
            .format(settings.ARTIFACTS_DIR)
        if not os.path.isfile(kubeconfig_path):
            raise Exception(
                f"{kubeconfig_path} not found in artifacts!"
            )
        kaas_manager = Manager(kubeconfig=kubeconfig_path)

        management_kubeconfig_path = os.path.join(
            custom_bootstrap_path or ".", "kubeconfig")

        # Get passwords from the management_kubeconfig extra field
        reader_password = kaas_manager.si_config.get_keycloak_user_password('reader')

        # Create Keycloak users by cli
        self.create_keycloak_user_cli('reader', reader_password, 'reader', management_kubeconfig_path, retry=True)
        # Create Keycloak users
        self.create_keycloak_user(kaas_manager, "writer", ["operator", "bm-pool-operator", "global-admin"],
                                  email="writer@kaas.local", extra_role="m:os@admin")
        self.create_keycloak_user(kaas_manager, "operator", ["operator", "bm-pool-operator", "global-admin"],
                                  email="operator@kaas.local")
        self.create_keycloak_user(kaas_manager, "stacklight", ["stacklight-admin"],
                                  email="stacklight@kaas.local")

    def step_005_check_versions(self):
        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        kaasreleases = kaas_manager.get_kaasrelease_names()

        # define actual version of kaasrelease for mgmt cluster
        ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
        cluster = ns.get_cluster(settings.CLUSTER_NAME)
        actual_kaasrelease = cluster.get_kaasrelease_version()
        LOG.info("Actual (deployed) kaasrelease version is {}".format(
            actual_kaasrelease))

        # fetch actual kaasrelease crd
        kaas_release_crd = kaas_manager.get_kaasrelease(
            actual_kaasrelease).data['spec']
        # fetch actual helmbundle crd
        helm_bundle_crd = kaas_manager.get_helmbundle(
            settings.CLUSTER_NAME).data['spec']

        # "kaas-0-3-0" > "kaas-0-3-0-rc" > "kaas-0-2-0" > "kaas-0-1-0-rc"
        latest_kaasrelease = str(max([version.parse(x) for x in kaasreleases]))

        # just a warning if we use not the latest version
        if latest_kaasrelease != actual_kaasrelease:
            LOG.warning("Actual kaasrelease ({0}) is not the latest one ({1})"
                        .format(actual_kaasrelease, latest_kaasrelease))

        # get kaasrelease file
        kaas_release_filename = self.get_kaas_release_file_name()
        if settings.KAAS_RELEASES_FOLDER and \
                os.path.isdir(settings.KAAS_RELEASES_FOLDER) and \
                kaas_release_filename and not \
                settings.BOOTSTRAP_WITH_DEFAULTS:
            kaas_release_path = "{}/kaas/{}".format(
                settings.KAAS_RELEASES_REMOTE_FOLDER,
                kaas_release_filename)
        else:
            actualrelease = actual_kaasrelease.replace(
                "kaas-", "").replace("-", ".")
            if 'rc' in actual_kaasrelease:
                release_parts = actualrelease.split('.')
                kaas_release_path = "releases/" \
                                    "kaas/{0}.{1}.{2}-{3}.yaml" \
                    .format(release_parts[0], release_parts[1],
                            release_parts[2], release_parts[3])
            else:
                kaas_release_path = (
                    "{0}/releases/kaas/{1}.yaml").format(
                    settings.KAAS_BOOTSTRAP_TARGET_DIR,
                    actualrelease)

        LOG.info("Accessing kaasrelease file in '{}'"
                 .format(kaas_release_path))
        kaas_release_file = self.get_remote_file_yaml(kaas_release_path)['spec']

        LOG.debug("Provided kaas_release_file {0}:\n{1}".format(
            kaas_release_path, yaml.dump(kaas_release_file)))
        LOG.debug("Active kaas_release_crd from the management cluster:\n{0}"
                  .format(yaml.dump(kaas_release_crd)))

        # compare helmRepositories
        cdn_region_definition = settings.CDN_REGION
        assert cdn_region_definition in settings.CDN_REGION_DICT.keys(), \
            ("Failed to find suitable CDN URL according to {} definition"
             .format(cdn_region_definition))

        cdn_url = settings.CDN_REGION_DICT[cdn_region_definition]
        crd_helm_repos = kaas_release_crd.get('helmRepositories', [])
        file_helm_repos = kaas_release_file.get('helmRepositories', [])
        self.assertEqual(crd_helm_repos,
                         [{'name': x['name'],
                           'url': cdn_url + '/' + x['url']}
                          for x in file_helm_repos],
                         msg="helmRepositories in kaasrelease crd "
                             "and file are not the same")

        LOG.info("Prepare objects for kaas releases comparison")
        obj_types = {
            'kaas_release_k8s_object': kaas_release_crd,
            'kaas_release_file_template': kaas_release_file,
        }
        for entity, release in obj_types.items():
            LOG.debug(f"Before: {release}")

            LOG.info(f"Removing redundant helmRepositories in {entity}")
            release.pop('helmRepositories', None)

            LOG.info(f"Removing empty provider sections in {entity}")
            for r in release['supportedClusterReleases']:
                if 'providers' in r and not r['providers']:
                    LOG.info(f"Removing 'providers' from {r}")
                    del r['providers']
            LOG.debug(f"After: {release}")

        # compare kaasrelease
        # solution for yaml aliases - this dumps full yaml w/o aliases
        yaml.Dumper.ignore_aliases = lambda *args: True
        self.assertEqual(yaml.dump(kaas_release_crd),
                         yaml.dump(kaas_release_file),
                         msg="Actual kaasrelease crd {0} does not match the "
                             "expected kaasrelease from {1}".format(
                             actual_kaasrelease, kaas_release_filename))

        # compare helmbundles crd with helmbundles data in kaas_release_file
        helm_bundle_file = kaas_release_file['management']
        for chart_in_crd in helm_bundle_crd['releases']:
            for chart_in_file in helm_bundle_file['helmReleases']:
                if chart_in_file['name'] == chart_in_crd['name']:
                    ver_in_file = helpers.get_chart_version(chart_in_file)
                    ver_in_crd = helpers.get_chart_version(chart_in_crd)
                    msg = "Actual version of the chart {0} is {1}, while" \
                          " expected is {2}".format(chart_in_crd['name'],
                                                    ver_in_crd,
                                                    ver_in_file)
                    res = ver_in_file == ver_in_crd
                    assert res, msg

        # Fetch data from cluster object
        cluster = ns.get_cluster(settings.CLUSTER_NAME)
        # cluster_release_version is deployed version of tne cluster
        cluster_release_version = \
            cluster.data['spec']['providerSpec']['value']['release']
        cluster_release_version_current = \
            cluster.data['status']['providerStatus'][
                'releaseRefs']['current']['name']
        LOG.info("Actual (deployed) clusterrelease version is {}".format(
            cluster_release_version))
        assert cluster_release_version == cluster_release_version_current

        # Make sure we have cluster_release_version
        # in the list of clusterreleases crd objects
        assert cluster_release_version in [
            x.name for x in kaas_manager.get_clusterreleases()], \
            "Deployed cluster release version is not in clusterreleases list"

        # Comparing vs kaas_release object
        assert cluster_release_version == kaas_release_crd['clusterRelease'], \
            "Deployed cluster release version doesn't match version " \
            "specified in kaasrelease"
        assert cluster_release_version in [
            x['name'] for x in kaas_release_crd['supportedClusterReleases']], \
            "Deployed cluster release version is not in kaasrelease" \
            "Supported Cluster Releases list"

        # Comparing clusterrelease crd vs clusterrelease file
        # first, fetch short 'version' value which is filename for
        # cluster yaml
        cluster_release_filename = \
            [x['version'] for x in kaas_release_crd['supportedClusterReleases']
             if x['name'] == cluster_release_version][0]

        # get cluster file
        if settings.KAAS_RELEASES_FOLDER and \
                os.path.isdir(settings.KAAS_RELEASES_FOLDER) and \
                kaas_release_filename and not \
                settings.BOOTSTRAP_WITH_DEFAULTS:
            cluster_release_path = "{}/cluster/{}.yaml".format(
                settings.KAAS_RELEASES_REMOTE_FOLDER,
                cluster_release_filename)
        else:
            cluster_release_path = (
                "{0}/releases/cluster/{1}.yaml").format(
                settings.KAAS_BOOTSTRAP_TARGET_DIR,
                cluster_release_filename)

        LOG.info("Accessing clusterrelease file in {}".format(
            cluster_release_path))
        cluster_release_file = self.get_remote_file_yaml(cluster_release_path)['spec']

        assert cluster_release_file == kaas_manager.get_clusterrelease(
            cluster_release_version).read().spec

    @utils.log_method_time()
    def step_006_wait_for_pods(self):

        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)

        cluster = kaas_manager.get_mgmt_cluster()
        cluster.check.check_cluster_readiness()
        cluster.check.check_k8s_pods()

    @utils.log_method_time()
    def step_006_postdeployment_checks(self, check_svc=False):

        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        cluster = kaas_manager.get_mgmt_cluster()
        cluster.check.check_helmbundles()
        cluster.check.check_persistent_volumes_mounts()
        if cluster.provider in (Provider.baremetal, Provider.equinixmetalv2):
            cluster.check.check_actual_expected_kernel_versions()
        if cluster.lcm_type_is_ucp:
            # Check/wait for correct docker service replicas in cluster
            cluster.check.check_actual_expected_docker_services()
        if settings.CUSTOM_CERTIFICATES:
            cluster.check.check_actual_expected_pods(timeout=900,
                                                     check_all_nss=True)
        else:
            cluster.check.check_actual_expected_pods(timeout=900,
                                                     check_all_nss=True)

        # TODO enable after verification on all providers
        # Related blocker PRODX-19064
        # cluster.check.check_actual_expected_rolebindings()
        # skip this check for 2.0.0 version
        kaas_version = cluster.get_kaasrelease_version()
        if kaas_version != '2.0.0':
            cluster.check.check_resource_discovery()
        if check_svc:
            cluster.check.check_svc_access(self.remote_seed())
        if settings.KAAS_OFFLINE_DEPLOYMENT:
            cluster.check.check_offline_isolation(
                settings.KAAS_EXTERNAL_PROXY_ACCESS_STR)
        if settings.CUSTOM_CERTIFICATES:
            services = {
                'ui': {
                    'namespace': 'kaas',
                    'name': 'kaas-kaas-ui'
                },
                'keycloak': {
                    'namespace': 'kaas',
                    'name': 'iam-keycloak-http'
                },
                'iam-proxy-alerta': {
                    'namespace': 'stacklight',
                    'name': 'iam-proxy-alerta'
                },
                'iam-proxy-alertmanager': {
                    'namespace': 'stacklight',
                    'name': 'iam-proxy-alertmanager'
                },
                'iam-proxy-grafana': {
                    'namespace': 'stacklight',
                    'name': 'iam-proxy-grafana'
                },
                'iam-proxy-prometheus': {
                    'namespace': 'stacklight',
                    'name': 'iam-proxy-prometheus'
                },

            }

            if cluster.logging_enabled():
                services['iam-proxy-kibana'] = {
                    'namespace': 'stacklight',
                    'name': 'iam-proxy-kibana',
                }

            for service in services:
                name = services[service]['name']
                namespace = services[service]['namespace']
                svc = cluster.k8sclient.services.get(
                    name=name,
                    namespace=namespace)
                ip = svc.get_external_addr()
                ips = [ip] if ip else None
                dns_name = svc.get_external_hostname()
                dns_names = [dns_name] if dns_name else None
                host = dns_name or ip

                LOG.info(f"Generate cert for {name}")
                cert_pem, key_pem, ca_pem = self.generate_cert(ips=ips, dns_names=dns_names)

                LOG.info(f"Apply cert for {name}")
                if service != 'keycloak':
                    ca_pem = None

                cert_manager = CertManager(kaas_manager)
                cert_manager.apply_cert_for_app(app=service, cluster=cluster, hostname=host,
                                                cert_pem=cert_pem, key_pem=key_pem, ca_pem=ca_pem)

                cluster.check.check_cert_conversation(host)

            cluster.check.check_actual_expected_pods(
                timeout=1800,
                interval=60,
                check_all_nss=True)
        cluster.check.check_overlay_encryption_functionality()
        if settings.KAAS_EQUINIXMETALV2_ENABLED:
            cluster.check.check_deploy_stage_success(skipped_stages_names='IAM objects created')
        elif settings.KAAS_VSPHERE_ENABLED and not settings.KAAS_VSPHERE_IPAM_ENABLED:
            cluster.check.check_deploy_stage_success(skipped_stages_names='Network prepared')
        else:
            cluster.check.check_deploy_stage_success()
        cluster.check.check_audit()

        # Check that Machine hostname is created with respect to Cluster flag 'customHostnamesEnabled'
        cluster.check.check_custom_hostnames_on_machines()

        # For BM environments, check that BareMetalHostInventory are present for each Machine (for MCC 2.29.0+)
        if cluster.provider == Provider.baremetal:
            cluster.check.check_bmh_inventory_presense()

    def step_006_enable_mcc_upgrade_auto_delay(self):
        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        body = {
            "spec": {
                "autoDelay": True,
            }
        }
        kaas_manager.api.kaas_mccupgrades.update(name='mcc-upgrade', body=body)
        LOG.info("Enabled auto-delay for mcc upgrade")
        mccupgrade = kaas_manager.api.kaas_mccupgrades.get(name='mcc-upgrade')
        LOG.info(f"Current mccugrade object:\n{yaml.dump(mccupgrade.data)}")

    @utils.log_method_time()
    def step_007_download_bootstrap_artifacts(self,
                                              nsname=settings.CLUSTER_NAMESPACE,  # noqa
                                              clname=settings.CLUSTER_NAME):
        """
         Also support bootstrap from region.
         For region:
            access to cluster - via mgmt kubeconf
            access to helmbundle - via region mgmt kubeconf
        """
        remote = self.remote_seed()
        # to collect image version from
        sys_namespaces = ['kaas', 'stacklight']

        # Here must be passed exactly management_kubeconfig, fetched on prev.
        # steps, to be able get access to all clusters
        kubeconfig_path_mgmt = get_kubeconfig_path()
        kaas_manager_mgmt = Manager(kubeconfig=kubeconfig_path_mgmt)
        # define actual version of kaasrelease, based on mgmt cluster
        ns = kaas_manager_mgmt.get_namespace(nsname)
        cluster = ns.get_cluster(clname)

        cluster_prefix = 'regional' if cluster.is_regional else 'management'
        kubectl_client = cluster.k8sclient
        LOG.info(f"Cluster: {nsname}/{clname} is {cluster_prefix}")

        kaas_release = cluster.get_kaasrelease_version()
        actual_kaasrelease = kaas_release.replace("kaas-",
                                                  "").replace("-", ".")
        # get kaasrelease file
        remote.download(
            f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}releases/kaas"
            f"/{actual_kaasrelease}.yaml",
            f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_kaasrelease.yaml")

        # copy key
        key_path = self.get_bootstrap_ssh_keyfile_path(remote)
        assert remote.download(
            key_path,
            f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_id_rsa"), \
            f"Failed to download {key_path} SSH key from remote machine"

        key = utils.load_keyfile(f"{settings.ARTIFACTS_DIR}/"
                                 f"{cluster_prefix}_id_rsa")
        pub_key = key["public"]
        with open(f'{settings.ARTIFACTS_DIR}/{cluster_prefix}_id_rsa_pub',
                  mode='w') as f:
            f.write("ssh-rsa " + str(pub_key))

        ver = kubectl_client.api_version.get_code()
        k8s_version = "{0}.{1}".format(ver.major, ver.minor).replace("+", ".x")
        with open(f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_k8s_version",
                  "w") as f:
            f.write(k8s_version)

        LOG.info(f"collect versions of images for all containers, "
                 f"in {sys_namespaces}")
        containers = []
        for ns in sys_namespaces:
            ns_containers = {}
            pods = \
                kubectl_client.pods.list_raw(namespace=ns).to_dict()['items']
            for pod in pods:
                for container in pod['spec']['containers']:
                    ns_containers[container['name']] = container['image']
            containers.append({ns: ns_containers})
        with open(f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_containers",
                  "w") as f:
            f.write(yaml.dump(containers))

        remote.download(
            f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/out/cluster.yaml",
            f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_cluster.yaml")
        remote.download(
            f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/out/machines.yaml",
            f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_machines.yaml")

        bootstrap_version = \
            kaas_manager_mgmt.get_kaasrelease(kaas_release).data['spec'][
                'bootstrap']['version']

        with open(f"{settings.ARTIFACTS_DIR}/bootstrap_version",
                  "w") as f:
            f.write(bootstrap_version)

        helm_bundle_crd = cluster.get_helmbundle(clname).data['spec']
        helm_charts_versions = [{"chart": x['name'],
                                 "version": helpers.get_chart_version(x)}
                                for x in helm_bundle_crd['releases']]
        with open(f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_helmbundles",
                  "w") as f:
            f.write(yaml.dump(helm_charts_versions))

        public_services = cluster.describe_public_services()
        services_txt = '\n'.join([
            "{0}: {1}".format(s, ' '.join([
                "<a href={0}>{0}</a>".format(h) for h in public_services[s]]))
            for s in public_services.keys()])
        passwords_file = "passwords.yaml" if remote.isfile("passwords.yaml") \
            else f"{settings.KAAS_BOOTSTRAP_TARGET_DIR}/passwords.yaml"
        if remote.isfile(passwords_file):
            res = remote.execute('cat {0}'.format(passwords_file))
            services_txt += '\n' + res.stdout_str
        for user in 'writer', 'reader', 'operator', 'stacklight':
            password = kaas_manager_mgmt.si_config.get_keycloak_user_password(user)
            services_txt += f"\nPassword for user {user}: {password}"
        with open(f"{settings.ARTIFACTS_DIR}/{cluster_prefix}_public_urls", "w") as f:
            f.write(services_txt)
        cluster.provider_resources.save_artifact()
        # For EM2/BM cases, external IP address for dhcp-relay service
        cluster._save_service_external_ip(service_name='dhcp-lb',
                                          service_namespace='kaas',
                                          filename="service_dhcp-lb_external_ip.txt")

    @retry(CalledProcessError, delay=5, tries=5, jitter=1, logger=LOG)
    def update_bootstrap_version(self, mgmt_k8s_ip):
        run_envs = {"KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL}
        if settings.KAAS_EXTERNAL_PROXY_ACCESS_STR:
            run_envs['HTTP_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['HTTPS_PROXY'] = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            run_envs['NO_PROXY'] = f"{settings.OS_FIP_RANGES},{mgmt_k8s_ip}"
            LOG.info(f"HTTP_PROXY and HTTPS_PROXY are set to {settings.KAAS_EXTERNAL_PROXY_ACCESS_STR}")
        envs_string = utils.make_export_env_strting(run_envs)
        remote = self.remote_seed()
        bootstrap_dir = settings.KAAS_BOOTSTRAP_TARGET_DIR
        mgmt_kubeconf = "kubeconfig" if remote.isfile("kubeconfig") else \
            "{0}/kubeconfig".format(bootstrap_dir)

        res = remote.check_call(
            "{envs_string}; "
            "./{bootstrap_dir}/container-cloud bootstrap download "
            "--target_dir ./{bootstrap_dir} "
            "--management-kubeconfig {mgmt_kubeconf}".format(
                envs_string=envs_string,
                bootstrap_dir=bootstrap_dir,
                mgmt_kubeconf=mgmt_kubeconf),
            verbose=True)
        return res

    def step_007_erase_env_after(self):
        remote = self.remote_seed()
        run_envs = {
            "CLUSTER_NAME": settings.CLUSTER_NAME,
            "REGIONAL_CLUSTER_NAME": settings.REGIONAL_CLUSTER_NAME,
            "KAAS_BOOTSTRAP_LOG_LVL": settings.KAAS_BOOTSTRAP_LOG_LVL
        }
        # handle clean seed regional specific scenario
        regional_artifacts = self.get_regional_artifacts(settings.REGIONAL_CLUSTER_NAME)
        run_envs["REGIONAL_KUBECONFIG"] = regional_artifacts["kubeconfig_path"]
        envs_string = utils.make_export_env_strting(run_envs)

        LOG.info("Cleanup KaaS cluster")
        remote.check_call(
            "{envs_string}; bash -x ./{bootstrap}/bootstrap.sh "
            "cleanup".format(
                envs_string=envs_string,
                bootstrap=settings.KAAS_BOOTSTRAP_TARGET_DIR),
            verbose=True)

    def step_bs_collect_bv2_logs(self, remote):
        LOG.info("Try to collect bootstrap logs")
        run_envs = {
            "CLUSTER_NAME": settings.CLUSTER_NAME,
            "KAAS_BOOTSTRAP_LOG_LVL": "0",  # set default verbosity level
            "COLLECT_EXTENDED_LOGS": "true",
            'KUBECONFIG': '~/does-not-exist'  # Set KUBECONFIG path to non-existing file
                                              # to force bootstrap.sh collecting logs from CLUSTER_NAME
                                              # instead of REGIONAL_CLUSTER_NAME
        }
        self._step_bs_collect_logs(remote, run_envs)

    def step_bs_collect_logs(self, remote):
        LOG.info("Try collect cluster logs")
        run_envs = {
            "CLUSTER_NAME": settings.CLUSTER_NAME,
            "KAAS_BOOTSTRAP_LOG_LVL": "0",  # set default verbosity level
            "COLLECT_EXTENDED_LOGS": "true"
        }
        if remote.isfile('./{}/kubeconfig'.format(
                settings.KAAS_BOOTSTRAP_TARGET_DIR)):
            run_envs['KUBECONFIG'] = './{}/kubeconfig'.format(
                settings.KAAS_BOOTSTRAP_TARGET_DIR)
        self._step_bs_collect_logs(remote, run_envs)

    @staticmethod
    def _step_bs_collect_logs(remote, run_envs):
        envs_string = utils.make_export_env_strting(run_envs)
        if settings.BM_COLLECT_LOGS:
            remote.execute(
                "{envs_string}; bash -x ./{bootstrap}/bootstrap.sh "
                "collect_logs".format(
                    envs_string=envs_string,
                    bootstrap=settings.KAAS_BOOTSTRAP_TARGET_DIR),
                verbose=True)
        else:
            LOG.warning("bootstrap.sh collect_logs skipped!")
        LOG.info('Try fetch logs')
        # We must avoid using sudo on macos user administrator and on linux user root because it brings password prompt
        sudo_mode = True
        if settings.SEED_SSH_LOGIN in settings.SYSTEM_ADMINS:
            sudo_mode = False
        if remote.isdir(f"./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/logs"):
            with remote.sudo(enforce=sudo_mode):
                remote.execute(
                    f"chown -R "
                    f"{settings.SEED_SSH_LOGIN}:{settings.SEED_SSH_LOGIN} "
                    f"./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/logs")
                remote.execute(
                    f"chmod -R 755 "
                    f"./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/logs")
                remote.execute(
                    f"find ./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/logs "
                    f"-type f -print0 | xargs -0 chmod 644")
            remote.check_call(f"tar zcfv logs.tar.gz "
                              f"./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/logs")
            remote.download("logs.tar.gz",
                            f"{settings.ARTIFACTS_DIR}/management_logs.tar.gz")
            LOG.info(f"Remote logs placed to {settings.ARTIFACTS_DIR}"
                     f"/management_logs.tar.gz")
        # fastlog supposed to be created by ansible
        flog = "./{}/logs/fastlog.txt".format(
            settings.KAAS_BOOTSTRAP_TARGET_DIR)
        if remote.isfile(flog):
            remote.download(flog,
                            "{0}/bootstrap.log".format(settings.ARTIFACTS_DIR))

    def step_collect_logs(self,
                          remote,
                          cluster_name=settings.CLUSTER_NAME,
                          cluster_namespace=settings.CLUSTER_NAMESPACE,
                          kubeconfig_path="kubeconfig",
                          keyfile_path="",
                          bootstrap_dir=settings.KAAS_BOOTSTRAP_TARGET_DIR,
                          log_level=settings.KAAS_BOOTSTRAP_LOG_LVL,
                          cluster_type="",
                          management_kubeconfig_path="",
                          delete_old_logs=settings.KAAS_DELETE_COLLECTED_LOGS):
        """
        This step call exactly in-house
        `container-cloud collect logs` collector, and pack results to artifacts.
        Please don't intend any guess magic here.
        :param remote:
        :param cluster_name:
        :param cluster_namespace:
        :param kubeconfig_path:
        :param keyfile_path:
        :param bootstrap_dir:
        :param log_level:
        :param cluster_type:
        :param management_kubeconfig_path:
        :param delete_old_logs:
        :return:
        """

        if not settings.BM_COLLECT_LOGS:
            LOG.warning("Step step_collect_logs skipped!")
            # simulate exit_code == 0
            res = remote.execute("exit 0")
            return res
        if not keyfile_path:
            keyfile_path = self.get_bootstrap_ssh_keyfile_path(remote)

        logs_dir_base = f"./{bootstrap_dir}/logs"
        if cluster_type != "":
            logs_dir_base = f"./{bootstrap_dir}/logs/{cluster_type}"

        logs_dir_cluster = f"{logs_dir_base}/{cluster_name}"
        if remote.isdir(logs_dir_cluster):
            if delete_old_logs:
                LOG.info(f"Remove folder '{logs_dir_cluster}' before collect logs")
                remote.execute(f"rm -rf {logs_dir_cluster}", verbose=True)
            else:
                # archive, then delete folder, save tar
                logs_dir_cluster_backup = logs_dir_cluster + '_bak_' + datetime.now().strftime("%Y_%m_%d_%H_%M_%S")
                LOG.info(f"Move folder '{logs_dir_cluster}' before collect logs")
                remote.execute(f"mv  {logs_dir_cluster} {logs_dir_cluster_backup}", verbose=True)
                remote.execute(
                    f"tar zcf {logs_dir_cluster_backup}.tgz {logs_dir_cluster_backup}",
                    verbose=True)
                LOG.info(f"Remove folder '{logs_dir_cluster_backup}' before collect logs")
                remote.execute(f"rm -rf {logs_dir_cluster_backup}", verbose=True)

        mgmt_kubeconf = management_kubeconfig_path or (
            "kubeconfig" if remote.isfile("kubeconfig") else f"{bootstrap_dir}/kubeconfig")

        # the v2 check is to preserve the behavior of v1
        if settings.MCC_BOOTSTRAP_VERSION == "v2":
            # NOTE(alexz:) This not work with -bm clusters.
            # check presence of both regional and mgmt kubeconfigs
            if not remote.isfile(mgmt_kubeconf):
                LOG.warning(f'File: {mgmt_kubeconf} not exist.'
                            f'Looks like mgmt cluster has not been deployed')
                mgmt_kubeconf = settings.KUBECONFIG_KIND_PATH
            if cluster_type == "regional" and not remote.isfile(kubeconfig_path):
                kubeconfig_path = settings.KUBECONFIG_KIND_PATH

        if kubeconfig_path:
            kubeconfig = f"--kubeconfig {kubeconfig_path}"
        else:
            kubeconfig = ""

        LOG.info(f"Try to collect logs for {cluster_name}/{cluster_namespace} =>{logs_dir_base}")
        res = remote.execute(
            f"./{bootstrap_dir}/container-cloud collect logs "
            f"{kubeconfig} "
            f"--v {log_level} "
            f"--cluster-name {cluster_name} "
            f"--cluster-namespace {cluster_namespace} "
            f"--key-file {keyfile_path} "
            f"--management-kubeconfig {mgmt_kubeconf} "
            f"--output-dir {logs_dir_base} "
            f"--extended",
            verbose=True)

        LOG.info('Try to fetch logs')
        cluster_id = cluster_name
        if cluster_type != "":
            cluster_id = f"{cluster_type}_{cluster_id}"
        tar_name = f"{cluster_id}_logs" + datetime.now().strftime("%Y_%m_%d_%H_%M_%S") + ".tar.gz"

        # We must avoid using sudo on macOS user administrator and on linux user root because it brings password prompt
        sudo_mode = True
        if settings.SEED_SSH_LOGIN in settings.SYSTEM_ADMINS:
            sudo_mode = False
        if remote.isdir(logs_dir_cluster):
            with remote.sudo(enforce=sudo_mode):
                remote.execute(
                    f"chown -R "
                    f"{settings.SEED_SSH_LOGIN}:{settings.SEED_SSH_LOGIN} "
                    f"{logs_dir_cluster}")
                remote.execute(
                    f"chmod -R 755 "
                    f"{logs_dir_cluster}")
                remote.execute(
                    f"find {logs_dir_cluster} "
                    f"-type f -print0 | xargs -0 chmod 644")
            remote.check_call(f"tar zcf logs.tar.gz {logs_dir_cluster}")
            remote.download("logs.tar.gz", f"{settings.ARTIFACTS_DIR}/{tar_name}")
            LOG.info(f"Remote logs placed to {settings.ARTIFACTS_DIR}/{tar_name}")
        return res

    # Always run only after self.step_bs_collect_logs()
    @staticmethod
    def check_multiple_kaasreleases(remote):
        """Check for multiple kaasreleases after bootstrap

        Raises Exception if more than one KaaSRelease appeared
        in the kind cluster (there should not be KaaS updates)

        Logs error if more than one KaaSRelease appeared in the
        management cluster (can be normal when bug PRODX-4819
        is fixed, if the update is started right after successful
        deploy and logs are collected at this moment)
        """
        ls_res = remote.execute(
            "ls -1 {0}/logs/kaas-bootstrap-cluster/"
            "objects/cluster/kaas.mirantis.com/kaasreleases/"
            .format(settings.KAAS_BOOTSTRAP_TARGET_DIR))
        if ls_res.exit_code == 0:
            kaasreleases = ls_res.stdout
            if len(kaasreleases) > 1:
                raise Exception("KaaSReleases number more than one: {0}"
                                .format(','.join(kaasreleases)))

        ls_res = remote.execute(
            "ls -1 {0}/logs/kaas-management-cluster/"
            "objects/cluster/kaas.mirantis.com/kaasreleases/")
        if ls_res.exit_code == 0:
            kaasreleases = ls_res.stdout
            if len(kaasreleases) > 1:
                LOG.error("KaaSReleases number more than one: {0}"
                          .format(','.join(kaasreleases)))

    def ansible_init(self):
        """
        Install venv with ansible.
        TODO: parametrize?
        :return:
        """
        remote = self.remote_seed()
        seed_key = settings.SEED_SSH_PRIV_KEY_FILE
        # refresh dir and key, w\o looking to already created env
        kaas_bm_env_dir = os.path.join(settings.SI_TESTS_REPO_ROOT,
                                       'si_tests/kaas-bm-env')
        self.rsync_run(from_dir=kaas_bm_env_dir, todir='~/kaas-bm-env/')
        # key required to get acesss to KVM nodes accross env
        LOG.info('Upload kaas-bm-env-key to seed_node')
        self.rsync_run(from_file=seed_key,
                       tofile='~/kaas-bm-env-key')
        if remote.check_call('[ -d ~/ansible-venv ]',
                             verbose=True,
                             raise_on_err=False).exit_code == 0:
            LOG.info('Ansible already init')
            return

        with remote.sudo(enforce=True):
            # yes, we still need python3-venv even if we have 3.8-venv
            pkgs = 'curl python3-venv python3-dev ' \
                   'python3-venv python3-apt ' \
                   'python3-pip libvirt-dev gcc'
            LOG.info(f'Install all necessary packages for ansible: {pkgs}')
            remote.check_call(f'apt -y install {pkgs}')

        LOG.info('Init ansible venv')
        cert_cmd = ""
        if settings.KAAS_SSL_PROXY_CERTIFICATE_FILE:
            cert_cmd = f"--cert {settings.KAAS_SSL_PROXY_CERTIFICATE_REMOTE_FILE}"
        remote.check_call('python3 -m venv  --system-site-packages'
                          ' --copies  ~/ansible-venv')
        remote.check_call('source ~/ansible-venv/bin/activate;'
                          ' pip3 install --upgrade pip')
        remote.check_call('source ~/ansible-venv/bin/activate;'
                          ' pip3 install -U wheel jmespath'
                          f' netaddr ansible Jinja2 lxml {cert_cmd}', verbose=True)
        remote.check_call('mkdir -vp ~/bootstrap ~/.ansible_state',
                          verbose=True)

    def ansible_run(self, remote, playbook, inventory=None, extra_vars=None,
                    tags=None, skip_tags=None, **kwargs):
        """
        Run pre-setup ansible-playbook.
        :param remote:
        :param playbook:
        :param inventory:
        :param extra_vars:
        :param skip_tags:
        :param tags:
        :return:
        """

        if not extra_vars:
            extra_vars = {}
        elif not isinstance(extra_vars, dict):
            raise Exception("'extra_vars' could be only 'dict', "
                            "got {}".format(extra_vars))

        def resolve_inventory_file(filename):
            result = remote.execute(f"readlink -e 'kaas-bm-env/{filename}'",
                                    verbose=True)
            if result.exit_code == 0:
                return result.stdout[0].rstrip().decode("utf-8")
            else:
                raise Exception(f"Inventory file '{filename}' "
                                f"does not exist on remote.")

        cmd = []
        if inventory:
            if isinstance(inventory, list):
                for item in inventory:
                    cmd.append("-i '{}'".format(resolve_inventory_file(item)))
            elif isinstance(inventory, str):
                cmd.append("-i '{}'".format(resolve_inventory_file(inventory)))
            else:
                raise Exception("Inventory could be either string or list, "
                                "got '{}'".format(inventory))

        if tags:
            cmd.append("--tags '{}'".format(','.join(tags)))

        if skip_tags:
            cmd.append("--skip-tags '{}'".format(','.join(skip_tags)))

        if settings.FEATURE_FLAGS.len():
            extra_vars['feature_flags'] = settings.FEATURE_FLAGS.list()

        if extra_vars:
            cmd.append("-e '{}'".format(json.dumps(extra_vars)))

        cmd.append(playbook)

        return remote.check_call(
            "source ~/ansible-venv/bin/activate; "
            "export ANSIBLE_CONFIG=~/kaas-bm-env/ansible.cfg; "
            "cd ~/kaas-bm-env; "
            "$(which ansible-playbook) {}".format(' '.join(cmd)),
            verbose=True, **kwargs)

    @retry(Exception, delay=5, tries=3, logger=LOG)
    def rsync_run(self, username=settings.SEED_SSH_LOGIN,
                  sshkeyfile=settings.SEED_SSH_PRIV_KEY_FILE, reverse=False,
                  hostname=None, from_dir=None, todir=None, from_file=None,
                  tofile=None):
        """
        Run local rsync call
        :param username:
        :param hostname:
        :param sshkeyfile:
        :param from_dir:
        :param todir:
        :param from_file:
        :param tofile:
        :param reverse: False -> sync from local to host
                        True -> sync from host to local
        :return:
        """
        hostname = hostname or self.get_seed_ip()
        # we need to full path to key,in case need to run from pushd
        sshopts = f'"ssh -i' \
                  f' {os.path.abspath(sshkeyfile)}' \
                  ' -o StrictHostKeyChecking=no' \
                  ' -o ConnectionAttempts=10 ' \
                  ' -o UserKnownHostsFile=/dev/null "'
        if not (todir or tofile):
            raise Exception("Target not set")
        r = Subprocess()
        r.logger.addHandler(logger.console)
        if from_file:
            _to = tofile or todir
            if reverse:
                LOG.info(f"Downloading '{hostname}:{from_file}' to "
                         f"'{_to}'")
                cmd = f'rsync -e {sshopts} -azqP ' \
                      f'{username}@{hostname}:{from_file} {_to}'
                r.check_call(cmd, raise_on_err=True, verbose=True)
                LOG.info(f"Downloaded '{hostname}:{from_file}' to "
                         f"'{_to}'")
            else:
                cmd = f'rsync -e {sshopts} -azqP ' \
                      f'{from_file} {username}@{hostname}:{_to}'
                LOG.info(f"Uploading '{from_file}' to "
                         f"'{hostname}:{_to}'")
                r.check_call(cmd, raise_on_err=True, verbose=True)
                LOG.info(f"Uploaded '{from_file}' to "
                         f"'{hostname}:{tofile or todir}'")
        elif from_dir:
            if reverse:
                from_dir = from_dir + '/'
                cmd = f'rsync -e {sshopts}  -azqP ' \
                      f'{username}@{hostname}:{from_dir} .'
                with utils.pushd(todir):
                    LOG.info(f"Downloading '{hostname}:{from_dir}' dir to"
                             f" '{todir}'")
                    r.check_call(cmd, raise_on_err=True, verbose=True)
                    LOG.info(f"Downloaded '{from_dir}' dir to"
                             f" '{todir}'")
            else:
                cmd = f'rsync -e {sshopts}  -azqP . ' \
                      f'{username}@{hostname}:{todir}'
                with utils.pushd(from_dir):
                    LOG.info(f"Uploading '{from_dir}' dir to"
                             f" '{hostname}:{todir}'")
                    r.check_call(cmd, raise_on_err=True, verbose=True)
                    LOG.info(f"Uploaded '{from_dir}' dir to"
                             f" '{hostname}:{todir}'")
        else:
            raise Exception('Wrong invocation')

    def get_keycloak_admin_client(self, kaas_manager):
        keycloak_ip = kaas_manager.get_keycloak_ip()
        pwd = kaas_manager.get_secret_data('iam-api-secrets',
                                           'kaas', 'keycloak_password')
        return keycloak_client.KeycloakAdminClient(ip=keycloak_ip,
                                                   user=settings.KEYCLOAK_USER,
                                                   password=pwd)

    @utils.log_method_time()
    def check_boot_from_volume(self, openstack_client,
                               boot_volume_size=120):
        LOG.info("Check machines booted from volume correctly")
        # Get cluster
        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
        cluster = ns.get_cluster(settings.CLUSTER_NAME)

        failed_bfv_machines = {}
        all_machines = cluster.get_machines()
        for machine in all_machines:
            try:
                cluster.check.check_boot_from_volume_by_machine_name(machine.name,
                                                                     openstack_client,
                                                                     boot_volume_size)
            except (AssertionError, Exception) as error:
                failed_bfv_machines.update({machine.name: error})
        if failed_bfv_machines:
            raise Exception(
                f'bootFromVolume checked failed, list of failed machines with errors: '
                f'\n {yaml.dump(failed_bfv_machines)}')
        else:
            LOG.info("All machines from cluster deployed with option bootFromVolume successfully")

    @utils.log_method_time()
    def compare_cluster_runtime_with_desired(self):
        # Get cluster
        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
        cluster = ns.get_cluster(settings.CLUSTER_NAME)
        cluster.check.compare_cluster_runtime_with_desired()

    @utils.log_method_time()
    def check_keystone_ldap_integration(self):
        LOG.info("Check keystone LDAP integration")
        # Get cluster
        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
        cluster = ns.get_cluster(settings.CLUSTER_NAME)
        check_resource_path = "apis/kaas.mirantis.com/v1alpha1/clusterreleases"

        # Negative check
        # Also required to force sync for the LDAP user to Keycloak "Users"
        # so we can grant the role to that LDAP user later.
        LOG.info(f"Login to UI using {settings.MCC_LDAP_UI_USER}, and check that "
                 f"access to the {check_resource_path} is restricted")
        d_client = cluster.get_mcc_dashboardclient(user=settings.MCC_LDAP_UI_USER,
                                                   password=settings.MCC_LDAP_UI_PASSWORD)
        d_resources = d_client.get_resource(check_resource_path,
                                            raise_on_error=False)
        LOG.debug(f"Response from {check_resource_path} using a new LDAP user "
                  f"(must be restricted for the user with no roles):\n"
                  f"{d_resources}")
        assert 'code' in d_resources and d_resources['code'] == 403, (
            f"Access to {check_resource_path} is allowed "
            f"for the LDAP user with no roles, while it should be restricted")

        # Grant 'writer' role
        LOG.info(f"Grant bm-pool-operator and global-admin role to {settings.MCC_LDAP_UI_USER}")
        username = settings.MCC_LDAP_UI_USER
        keycloak_admin_client = self.get_keycloak_admin_client(kaas_manager)
        ldap_user = keycloak_admin_client.user_get(username=username)
        ldap_user_name = ldap_user['username'] + "-" + ldap_user['id'].split("-")[0]
        rnd_string = utils.gen_random_string(6)

        def check_user_created(user_name):
            user = None
            try:
                user = kaas_manager.get_iamuser(user_name).read().to_dict()
            except Exception as e:
                LOG.warning(e)
            return user
        waiters.wait(
            lambda: check_user_created(user_name=ldap_user_name), timeout=600, interval=15,
            timeout_msg=f"Timeout waiting for user: {ldap_user_name}")

        kaas_manager.create_iamglobalrolebinding(f'{username}-admin-{rnd_string}', 'global-admin', ldap_user_name)
        kaas_manager.create_iamglobalrolebinding(f'{username}-{rnd_string}', 'bm-pool-operator', ldap_user_name)
        LOG.info(f'Add role m:os@admin to user {ldap_user_name}')
        keycloak_admin_client.assign_roles(username=username, roles="m:os@admin")

        # Re-login with new grants
        LOG.info(f"Re-login to UI using {settings.MCC_LDAP_UI_USER}, and check"
                 f"that access to the {check_resource_path} is allowed now")

        def resource_path_accessible():
            d_resources = None
            d_client = cluster.get_mcc_dashboardclient(
                user=settings.MCC_LDAP_UI_USER,
                password=settings.MCC_LDAP_UI_PASSWORD)
            # Positive check
            try:
                d_resources = d_client.get_resource(check_resource_path)
            except Exception as e:
                LOG.warning(e)
            return d_resources

        d_resources = waiters.wait(
            lambda: resource_path_accessible(), timeout=300, interval=15,
            timeout_msg=f"Timeout waiting for successful request "
                        f"to {check_resource_path}")

        LOG.debug(f"{check_resource_path} read with LDAP credentials:\n"
                  f"{d_resources}")
        LOG.info("Keystone LDAP integration check is PASSED")

    @utils.log_method_time()
    def check_audit(self):
        LOG.info("Checking audit on management cluster ...")
        kubeconfig_path = get_kubeconfig_path()
        kaas_manager = Manager(kubeconfig=kubeconfig_path)
        ns = kaas_manager.get_namespace(settings.CLUSTER_NAMESPACE)
        cluster = ns.get_cluster(settings.CLUSTER_NAME)
        cluster.check.check_audit()

    @utils.log_method_time()
    def setup_seed_node_ubuntu(self, remote, apt_for_cloud_str):
        _sfile = '/tmp/setup_seed_node_ubuntu'
        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage setup_seed_node_ubuntu,'
                        f'since {_sfile} exists ')
            return
        with remote.sudo(enforce=True):
            LOG.info('Stop apt self-update services')
            remote.check_call(
                'systemctl stop apt-daily-upgrade.service '
                'apt-daily.service apt-daily-upgrade.timer '
                'apt-daily.timer unattended-upgrades.service')
            remote.check_call(
                'systemctl disable apt-daily-upgrade.service '
                'apt-daily.service apt-daily-upgrade.timer '
                'apt-daily.timer unattended-upgrades.service')

            LOG.info('Block apt to install suggest pkgs')
            remote.check_call(
                'echo "APT::Get::Install-Suggests "false";" '
                ' > /etc/apt/apt.conf.d/si-tests;'
                'echo "APT::Get::Install-Recommends "false";"'
                '>> /etc/apt/apt.conf.d/si-tests')

            LOG.info('Switch apt to mirantis local mirror')
            remote.check_call(
                ''.join(['export DISTR=$(lsb_release -sc | awk "{print $2}");',
                         'echo "deb [arch=amd64] http://',
                         apt_for_cloud_str,
                         '/nightly/ubuntu/ ${DISTR} main restricted universe"',
                         ' > /etc/apt/sources.list;',
                         'echo "deb [arch=amd64] http://',
                         apt_for_cloud_str,
                         '/nightly/ubuntu/ ${DISTR}-updates main restricted',
                         ' universe" >> /etc/apt/sources.list']))

            LOG.info("Get DISTRIB_RELEASE from /etc/lsb-release")
            lsb_release = remote.execute(
                ". /etc/lsb-release && echo ${DISTRIB_RELEASE}",
            ).stdout[0].strip().decode("utf-8")
            LOG.info(f"DISTRIB_RELEASE={lsb_release}")

            if lsb_release == '20.04':
                LOG.info("Remove 50command-not-found APT config")
                remote.check_call(
                    "rm /etc/apt/apt.conf.d/50command-not-found || true")

            LOG.info("Update APT cache")
            remote.check_call('apt update')

            remote.check_call('mkdir -p /etc/docker/')
            remote.check_call(
                "echo '{ \"bip\": \"192.168.91.1/24\", \"tls\" : false }' "
                " > /etc/docker/daemon.json ")
            LOG.info('Install docker.io to seed node')
            # upstream docker.io now complains about restart. so, need to cover
            # it with non-interactive.
            remote.check_call('export DEBIAN_FRONTEND=noninteractive ;'
                              'export DEBCONF_NONINTERACTIVE_SEEN=true ;'
                              'export LANG=C.UTF-8; apt install docker.io')
            remote.check_call(
                "usermod -aG docker {}".format(settings.SEED_SSH_LOGIN))

            LOG.info("Restart docker service")
            remote.check_call("systemctl restart docker")

            LOG.info('Install base packages to seed node')
            remote.check_call('apt install dh-autoreconf '
                              'python3-venv rng-tools')

            if settings.SEED_NODE_EXTRA_PKG:
                LOG.warning('Install {}'.format(settings.SEED_NODE_EXTRA_PKG))
                remote.check_call('DEBIAN_FRONTEND=noninteractive apt install '
                                  '{}'.format(settings.SEED_NODE_EXTRA_PKG))
            if settings.SEED_NODE_EXTRA_DO_UPGRADE:
                LOG.warning('Upgrading system')
                remote.check_call(
                    'DEBIAN_FRONTEND=noninteractive apt dist-upgrade')
        remote.check_call(f'touch {_sfile}')

    def setup_seed_node_rhel(self, remote):
        _sfile = '/tmp/setup_seed_node_rhel'
        if remote.isfile(_sfile):
            LOG.warning(f'Skipping stage setup_seed_node_rhel,'
                        f'since {_sfile} exists ')
            return
        with remote.sudo(enforce=True):
            LOG.info('Attach RHEL subscription')
            remote.check_call(
                "subscription-manager register --username {} "
                "--password {}".format(
                    settings.KAAS_CI_RHEL_LICENSES_USERNAME,
                    settings.KAAS_CI_RHEL_LICENSES_PASSWORD),
                log_mask_re=settings.KAAS_CI_RHEL_LICENSES_PASSWORD)
            remote.check_call("subscription-manager attach --auto")
            LOG.info('Install base packages to seed node')
            remote.check_call('yum -y install rng-tools yum-utils wget vim iptables bash-completion')
            LOG.info('Get RHEL version')
            rhel_version_out = remote.check_call('source /etc/os-release; echo "${VERSION_ID%%.*}"')
            rhel_version = rhel_version_out.stdout[0].rstrip().decode("utf-8")
            if rhel_version == '7':
                LOG.info('Enable required mirrors on seed node')
                remote.check_call('yum-config-manager --enable rhel-7-server-extras-rpms')
            proxy = '_none_'
            if settings.KAAS_OFFLINE_DEPLOYMENT:
                proxy = settings.KAAS_EXTERNAL_PROXY_ACCESS_STR
            LOG.info('Configure docker EE repo')
            remote.check_call(
                "cat <<EOF > /etc/yum.repos.d/docker-ee.repo\n"
                "[docker-ee]\nname=Docker EE\ngpgcheck=0\nenabled=1\npriority=1\n"
                "baseurl=https://repos.mirantis.com/rhel/{}/x86_64/stable-23.0/\n"
                "module_hotfixes=1\n"
                "proxy={}\nEOF".format(rhel_version, proxy))
            LOG.info('Install and configure docker on seed node')
            remote.check_call(
                'yum install docker-ee -y; '
                'systemctl start docker; '
                'chmod 666 /var/run/docker.sock;')

            if settings.SEED_NODE_EXTRA_PKG:
                LOG.warning('Install {}'.format(settings.SEED_NODE_EXTRA_PKG))
                remote.check_call('yum -y install {}'.format(
                    settings.SEED_NODE_EXTRA_PKG))
            if settings.SEED_NODE_EXTRA_DO_UPGRADE:
                LOG.warning('Upgrading system')
                remote.check_call('yum update -y')

            LOG.info('Cleanup RHEL subscription from seed node')
            remote.check_call('subscription-manager remove --all; '
                              'subscription-manager unregister; '
                              'subscription-manager clean')
        remote.check_call(f'touch {_sfile}')

    def sync_configs_to_seed_PRODX_15087(self):
        """
        Hack function, to pass kubeconfig/etc files from local
        si-tests env, to target seed node.
        WARNING: function totally hardcoded.
        """
        remote = self.remote_seed()
        over_files = {os.path.join(settings.ARTIFACTS_DIR,
                                   'management_kubeconfig'): '~/kubeconfig',
                      os.path.join(settings.ARTIFACTS_DIR,
                                   'management_id_rsa'): '~/ssh_key'}
        for _, (k, v) in enumerate(over_files.items()):
            if os.path.isfile(k):
                if remote.isfile(v):
                    LOG.warning(f"Remote file {v} already exist")
                else:
                    self.rsync_run(from_file=k, tofile=v)
            else:
                raise Exception(f"Expected file {k} not found in SI artifacts")

    def create_seed_bash_completion(self):
        content = """
            PATH=${PATH}:~/kaas-bootstrap/bin:/home/ubuntu/bootstrap/dev/bin
            if [ -f ~/kubeconfig ]; then export KUBECONFIG=~/kubeconfig ; fi
            if [ -f ~/bootstrap/dev/kubeconfig ]; then export KUBECONFIG=~/bootstrap/dev/kubeconfig; fi
            echo "KUBECONFIG=${KUBECONFIG}"
            echo "kubectl=$(which kubectl)"
            if [ -f ~/JENKINS_URL ]; then echo "JENKINS_URL=$(cat ~/JENKINS_URL 2>/dev/null)" ; fi
        """
        remote = self.remote_seed()
        remote.check_call("sudo mkdir -p /etc/bash_completion.d/ && sudo chmod 777 /etc/bash_completion.d/")
        with remote.open("/etc/bash_completion.d/kaas", mode="w") as r_f:
            r_f.write(content)

    def get_noproxy_fips(self, remote):
        if not settings.FEATURE_FLAGS.enabled("empty-noproxy-bootstrap"):
            return settings.OS_FIP_RANGES
        return ""

    def get_region_from_bootstrap_region(self, provider: str, path=None) -> str:
        if not path:
            bootstrap_region_template_path = os.path.join(
                self._get_templates_path(provider), "bootstrapregion.yaml.template")
        else:
            bootstrap_region_template_path = os.path.join(
                self._get_templates_path(provider, path), "bootstrapregion.yaml.template")

        btr_data = self.get_remote_file_yaml(bootstrap_region_template_path)
        LOG.info(f"Bootstrap region data {btr_data}")
        return btr_data['metadata']['name']

    def _get_templates_path(self, provider: str, path=None) -> str:
        """
        Function to determine path to templates depending on management
        or regional cluster deployment and running tests type (core or SI).
        """

        if provider == settings.BAREMETAL_PROVIDER_NAME:
            provider = "bm"  # baremetal provider's directory named bm

        # mgmt cluster `TEMPLATES_DIR` always `TARGET/templates/provider_name`
        # for core-ci and si-tests both
        if not path:
            templates_path = utils.get_provider_specific_path(
                os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR, "templates"), provider)
        else:
            templates_path = utils.get_provider_specific_path(
                os.path.join(settings.KAAS_BOOTSTRAP_TARGET_DIR, path), provider)
        LOG.info(f"Template path is {templates_path}")
        return templates_path

    def bootstrapv2_print_controller_logs(self,
                                          kubectl: kubectl_utils.RemoteKubectl,
                                          interval: int = 30,
                                          extra_resource_message: str = "Bootstrap region status:\n",
                                          extra_resource_type: str = "bootstrapregions",
                                          extra_command: str = "-o jsonpath='{.items[0].status}'",
                                          extra_namespace: str = "default") -> None:
        # print out bootstrapregion status
        resource_status = kubectl.get(
            resource_type=extra_resource_type,
            command=extra_command,  # only 1 object exists
            namespace=extra_namespace,
            string=True)
        LOG.info(f"{extra_resource_message}{resource_status}")

        # at first fetch all logs
        controller_pod_name = kubectl.get(
            resource_type="pods",
            command="-lapp.kubernetes.io/name=bootstrap-provider -o jsonpath='{.items[0].metadata.name}'",
            namespace="kaas",
            string=True)
        pod_logs = kubectl.logs(pod_name=controller_pod_name, namespace="kaas").split("\n")

        TEMP_FILE = "bootstrap_controller.log"

        remote = self.remote_seed()
        remote.execute(command=f"touch {TEMP_FILE}")
        if not remote.isfile(TEMP_FILE):
            LOG.info(f"No tmp file {TEMP_FILE} found, skip printing bootstrap-controller logs")
            return

        with remote.open(TEMP_FILE, 'r') as r_f:
            # get already fetched logs
            existing_logs = r_f.readlines()

        differs = False
        diff = ""
        if len(pod_logs) > len(existing_logs):
            differs = True
            diff = '    ' + '\n    '.join(pod_logs[len(existing_logs):])

        if not differs:
            # nothing to do
            LOG.info(f"No new logs for {interval} seconds")
            return

        # dump logs for the next compare
        with remote.open(TEMP_FILE, 'w') as w_f:
            w_f.write('\n'.join(pod_logs))

        # print new logs
        LOG.info(f"\nLogs from bootstrap controller since {interval} seconds:"
                 f"\n--------------------------------------------------------"
                 f"\n{diff}\n")

    def setup_bootstrapv2_cluster(self, envs_string: str) -> bool:
        remote = self.remote_seed()
        # All env vars should be already inside bootstrap.env
        result = remote.execute(
            f"{envs_string}; bash -x ./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/bootstrap.sh bootstrapv2",
            verbose=True,
            timeout=settings.MCC_BOOTSTRAPV2_INIT_TIMEOUT)

        if result and result.exit_code != 0:
            # In case of bootstrap fail before pivoting phase, KinD cluster may contain useful logs
            try:
                self.step_collect_logs(remote,
                                       cluster_name=settings.CLUSTER_NAME,
                                       cluster_namespace=settings.CLUSTER_NAMESPACE,
                                       kubeconfig_path=settings.KUBECONFIG_KIND_PATH,
                                       cluster_type="bootstrap")
            except Exception as e:
                LOG.warning("Unable to collect KinD logs: {}".format(e))
            return False
        return True

    def delete_bootstrapv2_cluster(self) -> bool:
        remote = self.remote_seed()
        # we don't expect to have multiple kind clusters so delete all
        kind_bin = f"./{settings.KAAS_BOOTSTRAP_TARGET_DIR}/bin/kind"
        # Destroy kind clusters
        result = remote.execute(
            command=f"for cluster in $({kind_bin} get clusters); do "
            f"{kind_bin} delete cluster --name=\"$cluster\"; done",
            verbose=True, timeout=120)
        if result and result.exit_code != 0:
            LOG.warn("failed to delete kind clusters")
            return False

        # Delete tmp file, otherwise it will contain mgmt logs
        remote.execute("rm -f bootstrap_controller.log")
        return True

    def apply_bootstrapv2_templates(self, kubectl: kubectl_utils.RemoteKubectl, provider: str,
                                    region: str, cluster_name: str, templates_dir: str,
                                    start_deployment: bool = True) -> None:

        templater = templates.Bootstrapv2Applier(kubectl, templates_dir, provider, region)

        if provider != settings.BAREMETAL_PROVIDER_NAME:
            templater.apply_bootstrap_region_and_wait()
        else:
            templater.apply_bootstrap_region()

        # Patch serviceusers.yaml.template with a generated password.
        # It will be used in validations afterward.
        si_config = si_config_manager.SIConfigManager(si_config_path=settings.SI_CONFIG_PATH)
        templater.patch_service_user_template(si_config.get_keycloak_user_password("serviceuser"))
        templater.apply_service_user()

        if provider != settings.BAREMETAL_PROVIDER_NAME:
            templater.apply_provider_credentials_and_wait()

        if provider == settings.VSPHERE_PROVIDER_NAME:
            templater.apply_metallb_config()
            templater.apply_rhel_license()
            if settings.VSPHERE_USE_VVMT_OBJECTS:
                templater.apply_vvmt_and_wait()

        if provider == settings.EQUINIXMETALV2_PROVIDER_NAME:
            templater.apply_bmhp()
            templater.apply_metallb_config()

        templater.apply_cluster()

        if provider == settings.BAREMETAL_PROVIDER_NAME:
            templater.apply_ipam()
            templater.apply_metallb_config()
            templater.apply_bmh()
            templater.apply_bmhp()
            templater.apply_vmbc()  # For Virt envs only. Wait is included

        templater.apply_machines()

        if provider == settings.BAREMETAL_PROVIDER_NAME:
            templater.wait_bootstrap_region()
            if settings.KAAS_BOOTSTRAP_INFINITE_TIMEOUT:
                templater.wait_all_bmhs(timeout=24 * 60 * 60)
            else:
                templater.wait_all_bmhs()

        if not start_deployment:
            return

        templater.start_cluster_deployment(
            cluster_name,
            provider,
            self.bootstrapv2_print_controller_logs)

    def run_bootstrap_v2(self, provider: str, region: str) -> bool:
        LOG.info(f"Started bootstrapV2 test for provider: {provider} with region: {region} ...")

        remote = self.remote_seed()

        kubeconfig_path = self.get_remote_kubeconfig_path(settings.KUBECONFIG_KIND_PATH)
        kubectl = kubectl_utils.RemoteKubectl(
            remote=remote,
            binary_path=os.path.join(".", settings.KAAS_BOOTSTRAP_TARGET_DIR, "bin", "kubectl"),
            kubeconfig=kubeconfig_path)

        # Apply templates and start deployment process
        self.apply_bootstrapv2_templates(
            kubectl=kubectl,
            provider=provider, region=region,
            cluster_name=settings.CLUSTER_NAME,
            templates_dir=self._get_templates_path(provider),
            start_deployment=True)

        return True

    def run_airgapped_command(
        self, action, mcc_version=None, cluster_args="", extra_args="", env_vars=None, timeout=3600
    ):
        """
        Helper function to run: airgapped.sh <action> --kaas-release <mcc_version> <cluster_args> ...
        to reduce duplication in the main test.

        :param action: str, e.g. "inspect", "demo --init", "sync", "push", "validate"
        :param mcc_version: str, MCC version from bootstrap
        :param cluster_args: str, e.g. "--cluster-release mke-16-3-0-3-7-12 --cluster-release mke-16-3-4-3-7-17"
        :param extra_args: str, any additional arguments
        :param env_vars: dict, environment variables if needed
        """
        air_gap_path = env_vars["AIR_GAP_PATH"]
        release_part = f"--kaas-release {mcc_version}" if mcc_version else ""
        cmd = f"{air_gap_path} {action} {release_part} {cluster_args} {extra_args}"
        env_vars = utils.make_export_env_strting(env_vars)
        remote = self.remote_seed()
        remote.check_call(f'{env_vars}; {cmd}', raise_on_err=True, verbose=True, timeout=timeout)

    def prepare_airgap_env(self, home_dir, docker_image):
        """
        Function prepares environment for airgap tests

        :param home_dir: str, home dir, e.g. /home/{settings.SEED_SSH_LOGIN}
        :param docker_image: str, airgapped image
        """
        env_vars = os.environ.copy()
        env_vars["AIRGAPPED_WORKSPACE"] = "/opt/airgapped"
        env_vars["AIRGAPPED_IMAGE"] = docker_image
        env_vars["RELEASE_ARTIFACTS_TAR"] = settings.KAAS_RELEASE_ARTIFACTS_TAR_URL
        env_vars["RELEASES_TAR"] = settings.KAAS_RELEASES_TAR
        # We'll store path to airgapped.sh in env_vars so run_airgapped_commdand can use it
        env_vars["AIR_GAP_PATH"] = f'{home_dir}/{settings.KAAS_BOOTSTRAP_TARGET_DIR}/airgapped.sh'

        settings.KAAS_RELEASES_FOLDER = f'{home_dir}/kaas_releases'

        return env_vars

    def build_cluster_release_args_airgap(self):
        """
        Builds the '--cluster-release ...' arguments for airgapped.sh
        """
        mcc_mgmt_cluster_release = self.get_clusterrelease_from_kaas()
        mcc_managed_cluster_release = settings.KAAS_CHILD_CLUSTER_RELEASE_NAME

        args = f"--cluster-release {mcc_mgmt_cluster_release}"
        if mcc_managed_cluster_release and mcc_mgmt_cluster_release != mcc_managed_cluster_release:
            args += f" --cluster-release {mcc_managed_cluster_release}"
        return args
